package tecgraf.openbus.core;

import java.security.interfaces.RSAPublicKey;
import java.util.Arrays;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.omg.CORBA.Any;
import org.omg.CORBA.BAD_PARAM;
import org.omg.CORBA.CompletionStatus;
import org.omg.CORBA.INTERNAL;
import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.ORB;
import org.omg.CORBA.TCKind;
import org.omg.IOP.Codec;
import org.omg.IOP.ServiceContext;
import org.omg.IOP.CodecPackage.FormatMismatch;
import org.omg.IOP.CodecPackage.InvalidTypeForEncoding;
import org.omg.IOP.CodecPackage.TypeMismatch;
import org.omg.PortableInterceptor.ForwardRequest;
import org.omg.PortableInterceptor.InvalidSlot;
import org.omg.PortableInterceptor.ServerRequestInfo;
import org.omg.PortableInterceptor.ServerRequestInterceptor;

import tecgraf.openbus.CallDispatchCallback;
import tecgraf.openbus.Connection;
import tecgraf.openbus.core.Session.ServerSideSession;
import tecgraf.openbus.core.v1_05.access_control_service.Credential;
import tecgraf.openbus.core.v1_05.access_control_service.CredentialHelper;
import tecgraf.openbus.core.v2_0.OctetSeqHolder;
import tecgraf.openbus.core.v2_0.credential.CredentialContextId;
import tecgraf.openbus.core.v2_0.credential.CredentialData;
import tecgraf.openbus.core.v2_0.credential.CredentialDataHelper;
import tecgraf.openbus.core.v2_0.credential.CredentialReset;
import tecgraf.openbus.core.v2_0.credential.CredentialResetHelper;
import tecgraf.openbus.core.v2_0.credential.SignedCallChain;
import tecgraf.openbus.core.v2_0.credential.SignedCallChainHelper;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.CallChain;
import tecgraf.openbus.core.v2_0.services.access_control.CallChainHelper;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidChainCode;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidCredentialCode;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidLoginCode;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidLogins;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidPublicKeyCode;
import tecgraf.openbus.core.v2_0.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_0.services.access_control.NoCredentialCode;
import tecgraf.openbus.core.v2_0.services.access_control.NoLoginCode;
import tecgraf.openbus.core.v2_0.services.access_control.UnknownBusCode;
import tecgraf.openbus.core.v2_0.services.access_control.UnverifiedLoginCode;
import tecgraf.openbus.exception.CryptographyException;
import tecgraf.openbus.security.Cryptography;

/**
 * Interceptador servidor.
 * 
 * @author Tecgraf
 */
final class ServerRequestInterceptorImpl extends InterceptorImpl implements
  ServerRequestInterceptor {

  /** Instncia de logging. */
  private static final Logger logger = Logger
    .getLogger(ServerRequestInterceptorImpl.class.getName());

  /** Valor de busid desconhecido. */
  private static final String UNKNOWN_BUS = "";

  /**
   * Construtor.
   * 
   * @param name nome do interceptador
   * @param mediator o mediador do ORB
   */
  ServerRequestInterceptorImpl(String name, ORBMediator mediator) {
    super(name, mediator);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void receive_request_service_contexts(ServerRequestInfo arg0)
    throws ForwardRequest {
    // do nothing
  }

  /**
   * Recupera a credencial do contexto.
   * 
   * @param ri informao do contexto
   * @return Wrapper para a credencial extrada.
   */
  private CredentialWrapper retrieveCredential(ServerRequestInfo ri) {
    ORB orb = this.getMediator().getORB();
    Codec codec = this.getMediator().getCodec();
    byte[] encodedCredential = null;
    try {
      ServiceContext requestServiceContext =
        ri.get_request_service_context(CredentialContextId.value);
      encodedCredential = requestServiceContext.context_data;
    }
    catch (BAD_PARAM e) {
      // BAD_PARAM caso no exista service context com o id especificado
      // CORBA ESPEC define com minor code 26.
      // JacORB define com minor code 23.
      switch (e.minor) {
        case 26:
        case 23:
          break;
        default:
          throw e;
      }
    }
    if (encodedCredential != null) {
      Any any;
      try {
        any =
          codec.decode_value(encodedCredential, CredentialDataHelper.type());
        CredentialData credential = CredentialDataHelper.extract(any);
        return new CredentialWrapper(false, credential, null);
      }
      catch (TypeMismatch e) {
        String message = "Falha inesperada ao decodificar a credencial";
        logger.log(Level.SEVERE, message, e);
        throw new INTERNAL(message);
      }
      catch (FormatMismatch e) {
        String message = "Falha inesperada ao decodificar a credencial";
        logger.log(Level.SEVERE, message, e);
        throw new INTERNAL(message);
      }
    }
    else {
      int legacyContextId = 1234;
      byte[] encodedLegacyCredential = null;
      try {
        ServiceContext serviceContext =
          ri.get_request_service_context(legacyContextId);
        encodedLegacyCredential = serviceContext.context_data;
      }
      catch (BAD_PARAM e) {
        // BAD_PARAM caso no exista service context com o id especificado
        // CORBA ESPEC define com minor code 26.
        // JacORB define com minor code 23.
        switch (e.minor) {
          case 26:
          case 23:
            break;
          default:
            throw e;
        }
      }
      if (encodedLegacyCredential != null) {
        CredentialWrapper wrapper = new CredentialWrapper();
        try {
          Any anyLegacy =
            codec
              .decode_value(encodedLegacyCredential, CredentialHelper.type());
          Credential legacyCredential = CredentialHelper.extract(anyLegacy);
          // extraindo informaes da credencial 1.5
          String loginId = legacyCredential.identifier;
          String entity = legacyCredential.owner;

          LoginInfo[] originators;
          if (!legacyCredential.delegate.equals("")) {
            originators = new LoginInfo[1];
            originators[0] =
              new LoginInfo("<unknown>", legacyCredential.delegate);
          }
          else {
            originators = new LoginInfo[0];
          }
          /*
           * campo target s pode ser preenchido aps definio da conexo que
           * tratar o request. Esta definio  feita aps a validao da
           * cadeia
           */
          CallChain callChain =
            new CallChain("", originators, new LoginInfo(loginId, entity));
          Any anyCallChain = orb.create_any();
          CallChainHelper.insert(anyCallChain, callChain);
          byte[] encodedCallChain = codec.encode_value(anyCallChain);

          // construindo uma credencial 2.0
          CredentialData credential = new CredentialData();
          credential.bus = UNKNOWN_BUS;
          credential.login = loginId;
          credential.session = -1;
          credential.ticket = -1;
          credential.hash = LEGACY_HASH;
          credential.chain =
            new SignedCallChain(LEGACY_ENCRYPTED_BLOCK, encodedCallChain);
          // salvando informaes no wrapper
          wrapper.isLegacy = true;
          wrapper.credential = credential;
          wrapper.legacyCredential = legacyCredential;
          return wrapper;
        }
        catch (TypeMismatch e) {
          String message =
            String.format("Falha ao decodificar a credencial 1.5: %s", e
              .getClass().getSimpleName());
          logger.log(Level.SEVERE, message, e);
          throw new INTERNAL(message, 0, CompletionStatus.COMPLETED_NO);
        }
        catch (FormatMismatch e) {
          String message =
            String.format("Falha ao decodificar a credencial 1.5: %s", e
              .getClass().getSimpleName());
          logger.log(Level.SEVERE, message, e);
          throw new INTERNAL(message, 0, CompletionStatus.COMPLETED_NO);
        }
        catch (InvalidTypeForEncoding e) {
          String message =
            String.format(
              "Falha ao construir credencial 2.0 a partir da 1.5: %s", e
                .getClass().getSimpleName());
          logger.log(Level.SEVERE, message, e);
          throw new INTERNAL(message, 0, CompletionStatus.COMPLETED_NO);
        }
      }
    }
    String message = "Nenhuma credencial suportada encontrada";
    logger.info(message);
    throw new NO_PERMISSION(message, NoCredentialCode.value,
      CompletionStatus.COMPLETED_NO);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void receive_request(ServerRequestInfo ri) {
    String operation = ri.operation();
    byte[] object_id = ri.object_id();
    ORB orb = this.getMediator().getORB();
    OpenBusContextImpl context = this.getMediator().getContext();
    CredentialWrapper wrapper = retrieveCredential(ri);
    try {
      CredentialData credential = wrapper.credential;
      if (credential != null) {
        String busId = credential.bus;
        String loginId = credential.login;
        ConnectionImpl conn =
          getConnForDispatch(context, busId, loginId, object_id, operation);
        context.setCurrentConnection(conn);
        boolean valid = false;
        if (!wrapper.isLegacy) {
          try {
            valid = conn.cache.logins.validateLogin(loginId);
          }
          catch (NO_PERMISSION e) {
            if (e.minor == NoLoginCode.value) {
              String message =
                "Erro ao validar o login. Conexo dispatcher est deslogada.";
              logger.log(Level.SEVERE, message, e);
              throw new NO_PERMISSION(UnknownBusCode.value,
                CompletionStatus.COMPLETED_NO);
            }
            else {
              String message = "Erro ao validar o login.";
              logger.log(Level.SEVERE, message, e);
              throw new NO_PERMISSION(UnverifiedLoginCode.value,
                CompletionStatus.COMPLETED_NO);
            }
          }
          catch (Exception e) {
            String message = "Erro ao validar o login.";
            logger.log(Level.SEVERE, message, e);
            throw new NO_PERMISSION(UnverifiedLoginCode.value,
              CompletionStatus.COMPLETED_NO);
          }
          if (!valid) {
            throw new NO_PERMISSION(InvalidLoginCode.value,
              CompletionStatus.COMPLETED_NO);
          }
        }
        else {
          // caso com credencial 1.5
          try {
            if (conn.cache.logins.validateLogin(loginId)
              && conn.cache.valids.isValid(wrapper.legacyCredential)) {
              valid = true;
            }
          }
          catch (Exception e) {
            String message = "Erro ao validar o login 1.5.";
            logger.log(Level.SEVERE, message, e);
            throw new NO_PERMISSION(0, CompletionStatus.COMPLETED_NO);
          }
          if (!valid) {
            String msg =
              "Login de credencial 1.5 no  vlido: login (%s) operao (%s)";
            logger.fine(String.format(msg, loginId, operation));
            throw new NO_PERMISSION(0, CompletionStatus.COMPLETED_NO);
          }
        }
        OctetSeqHolder pubkey = new OctetSeqHolder();
        String entity;
        try {
          entity = conn.cache.logins.getLoginEntity(loginId, pubkey);
        }
        catch (InvalidLogins e) {
          String message = "Erro ao verificar o login.";
          logger.log(Level.SEVERE, message, e);
          throw new NO_PERMISSION(InvalidLoginCode.value,
            CompletionStatus.COMPLETED_NO);
        }
        catch (ServiceFailure e) {
          String message = "Erro ao verificar o login.";
          logger.log(Level.SEVERE, message, e);
          throw new NO_PERMISSION(UnverifiedLoginCode.value,
            CompletionStatus.COMPLETED_NO);
        }
        catch (NO_PERMISSION e) {
          if (e.minor == NoLoginCode.value) {
            String message =
              "Erro ao verificar o login. Conexo dispatcher est deslogada.";
            logger.log(Level.SEVERE, message, e);
            throw new NO_PERMISSION(UnknownBusCode.value,
              CompletionStatus.COMPLETED_NO);
          }
          else {
            String message = "Erro ao verificar o login.";
            logger.log(Level.SEVERE, message, e);
            throw new NO_PERMISSION(UnverifiedLoginCode.value,
              CompletionStatus.COMPLETED_NO);
          }
        }
        catch (Exception e) {
          String message = "Erro ao verificar o login.";
          logger.log(Level.SEVERE, message, e);
          throw new NO_PERMISSION(UnverifiedLoginCode.value,
            CompletionStatus.COMPLETED_NO);
        }
        if (validateCredential(credential, ri, conn)) {
          if (validateChain(credential, pubkey, ri, conn)) {
            // salvando informao do barramento que atendeu a requisio
            Any any = orb.create_any();
            any.insert_string(conn.busid());
            ri.set_slot(this.getMediator().getBusSlotId(), any);
            String msg =
              "Recebendo chamada pelo barramento: login (%s) entidade (%s) operao (%s)";
            logger.fine(String.format(msg, loginId, entity, operation));
          }
          else {
            logger.finest(String.format(
              "Recebeu chamada com cadeia invlida: %s", operation));
            throw new NO_PERMISSION(InvalidChainCode.value,
              CompletionStatus.COMPLETED_NO);
          }
        }
        else {
          logger.finest(String.format(
            "Recebeu chamada sem sesso associda: %s", operation));
          // credencial no  vlida. Resetando a credencial da sesso.
          doResetCredential(ri, orb, conn, loginId, pubkey.value);
          throw new NO_PERMISSION(InvalidCredentialCode.value,
            CompletionStatus.COMPLETED_NO);
        }
        // salva a conexo utilizada no dispatch
        // CHECK se o bug relatado no finally for verdadeiro, isso pode ser desnecessario
        setCurrentConnection(ri, conn);
      }
      else {
        logger.fine(String.format("Recebeu chamada fora do barramento: %s",
          operation));
      }
    }
    catch (InvalidSlot e) {
      String message = "Falha inesperada ao acessar o slot da credencial";
      logger.log(Level.SEVERE, message, e);
      throw new INTERNAL(message);
    }
    catch (CryptographyException e) {
      String message = "Falha ao criptografar com chave pblica";
      logger.log(Level.SEVERE, message, e);
      throw new NO_PERMISSION(InvalidPublicKeyCode.value,
        CompletionStatus.COMPLETED_NO);
    }
    finally {
      // Talvez essa operao nao deveria ser necessaria, 
      // pois o PICurrent deveria acabar junto com a thread de interceptao.
      // CHECK possvel bug! Esta operao modifica o valor setado no ri e PICurrent
      //context.setCurrentConnection(null);
    }
  }

  /**
   * Recupera a conexo a ser utilizada no dispatch.
   * 
   * @param context Gerenciador de contexto do ORB que recebeu a chamada.
   * @param busId Identificao do barramento atravs do qual a chamada foi
   *        feita.
   * @param loginId Informaes do login que se tornou invlido.
   * @param object_id Idenficador opaco descrevendo o objeto sendo chamado.
   * @param operation Nome da operao sendo chamada.
   * 
   * @return Conexo a ser utilizada para receber a chamada.
   */
  private ConnectionImpl getConnForDispatch(OpenBusContextImpl context,
    String busId, String loginId, byte[] object_id, String operation) {
    ConnectionImpl conn = null;
    CallDispatchCallback onCallDispatch = context.onCallDispatch();
    if (onCallDispatch != null) {
      try {
        conn =
          (ConnectionImpl) onCallDispatch.dispatch(context, busId, loginId,
            object_id, operation);
      }
      catch (Exception e) {
        logger.log(Level.SEVERE,
          "Callback 'onCallDispatch' gerou um erro durante execuo.", e);
      }
      // CHECK caso callback gere um erro, busco a conexao default ou NO_PERMISSION?
    }
    if (conn == null) {
      conn = (ConnectionImpl) context.getDefaultConnection();
      if (conn == null) {
        throw new NO_PERMISSION(UnknownBusCode.value,
          CompletionStatus.COMPLETED_NO);
      }
    }
    if (conn.login() == null
      || (!conn.busid().equals(busId) && !UNKNOWN_BUS.equals(busId))) {
      throw new NO_PERMISSION(UnknownBusCode.value,
        CompletionStatus.COMPLETED_NO);
    }
    return conn;
  }

  /**
   * Realiza o protocolo de reiniciar a credencial da sesso.
   * 
   * @param ri informao do request.
   * @param orb o orb.
   * @param conn a conexo.
   * @param caller identificador do login originador da chamada.
   * @param publicKey a chave pblica do requisitante.
   * @throws CryptographyException
   */
  private void doResetCredential(ServerRequestInfo ri, ORB orb,
    ConnectionImpl conn, String caller, byte[] publicKey)
    throws CryptographyException {
    byte[] newSecret = newSecret();
    Cryptography crypto = Cryptography.getInstance();
    byte[] encriptedSecret =
      crypto.encrypt(newSecret, crypto
        .generateRSAPublicKeyFromX509EncodedKey(publicKey));
    int sessionId = conn.nextAvailableSessionId();
    ServerSideSession newSession =
      new ServerSideSession(sessionId, newSecret, caller);
    conn.cache.srvSessions.put(newSession.getSession(), newSession);
    CredentialReset reset =
      new CredentialReset(conn.login().id, sessionId, encriptedSecret);
    Any any = orb.create_any();
    CredentialResetHelper.insert(any, reset);
    byte[] encodedCredential;
    try {
      encodedCredential = this.getMediator().getCodec().encode_value(any);
    }
    catch (InvalidTypeForEncoding e) {
      String message = "Falha inesperada ao codificar a credencial";
      logger.log(Level.SEVERE, message, e);
      throw new INTERNAL(message);
    }
    logger.finest("Resetando a credencial: " + ri.operation());
    ServiceContext requestServiceContext =
      new ServiceContext(CredentialContextId.value, encodedCredential);
    ri.add_reply_service_context(requestServiceContext, false);
  }

  /**
   * Verifica se a credencial  vlida, recalculando o hash da mesma.
   * 
   * @param credential a credencial
   * @param ri informao do request.
   * @param conn a conexo em uso.
   * @return <code>true</code> caso a credencial seja vlida, ou
   *         <code>false</code> caso contrrio.
   */
  private boolean validateCredential(CredentialData credential,
    ServerRequestInfo ri, ConnectionImpl conn) {
    if (Arrays.equals(credential.hash, LEGACY_HASH)) {
      // credencial OpenBus 1.5
      logger.finest("Credencial OpenBus 1.5");
      return true;
    }
    ServerSideSession session = conn.cache.srvSessions.get(credential.session);
    if (session != null && session.getCaller().equals(credential.login)) {
      logger.finest(String.format("sesso utilizada: id = %d ticket = %d",
        session.getSession(), credential.ticket));
      byte[] hash =
        this.generateCredentialDataHash(ri, session.getSecret(),
          credential.ticket);
      if (Arrays.equals(hash, credential.hash)
        && session.checkTicket(credential.ticket)) {
        return true;
      }
      else {
        logger.finest("Falha na validao do hash da credencial");
      }
    }
    return false;
  }

  /**
   * Valida a cadeia da credencial.
   * 
   * @param credential a credencial
   * @param pubkey a chave pblica da entidade
   * @param ri informaes do request
   * @param conn a conexo em uso.
   * @return <code>true</code> caso a cadeia seja vlida, ou <code>false</code>
   *         caso contrrio.
   */
  private boolean validateChain(CredentialData credential,
    OctetSeqHolder pubkey, ServerRequestInfo ri, ConnectionImpl conn) {
    Cryptography crypto = Cryptography.getInstance();
    RSAPublicKey busPubKey = conn.getBusPublicKey();
    SignedCallChain chain = credential.chain;
    boolean isValid = false;

    if (chain != null) {
      CallChain callChain = unmarshallSignedChain(chain, logger);
      if (!Arrays.equals(chain.signature, LEGACY_ENCRYPTED_BLOCK)) {
        try {
          boolean verified =
            crypto.verifySignature(busPubKey, chain.encoded, chain.signature);
          if (verified) {
            LoginInfo loginInfo = conn.login();
            if (callChain.target.equals(loginInfo.entity)) {
              LoginInfo caller = callChain.caller;
              if (caller.id.equals(credential.login)) {
                isValid = true;
              }
            }
            else {
              ORB orb = this.getMediator().getORB();
              logger
                .finest(String
                  .format(
                    "O login no  o mesmo do alvo da cadeia.  necessrio refazer a sesso de credencial atravs de um reset. Operao: %s",
                    ri.operation()));
              // credencial no  vlida. Resetando a credencial da sesso.
              doResetCredential(ri, orb, conn, credential.login, pubkey.value);
              throw new NO_PERMISSION(InvalidCredentialCode.value,
                CompletionStatus.COMPLETED_NO);
            }
          }
        }
        catch (CryptographyException e) {
          String message =
            "Falha inesperada ao verificar assinatura da cadeia.";
          logger.log(Level.SEVERE, message, e);
          throw new INTERNAL(message);
        }
      }
      else {
        // cadeia 1.5  sempre vlida
        logger.finest("Cadeia OpenBus 1.5");
        /*
         * Atualizando a informao target da cadeia gerada a partir de
         * informaes 1.5, pois foi necessrio aguardar a definio da conexo
         * que trataria a requisio legada
         */
        callChain.target = conn.login().entity;
        isValid = true;
      }
      if (isValid) {
        try {
          ORB orb = this.getMediator().getORB();
          // salvando a cadeia
          Any singnedAny = orb.create_any();
          SignedCallChainHelper.insert(singnedAny, chain);
          ri.set_slot(this.getMediator().getSignedChainSlotId(), singnedAny);
        }
        catch (InvalidSlot e) {
          String message =
            "Falha inesperada ao armazenar o dados no slot de contexto";
          logger.log(Level.SEVERE, message, e);
          throw new INTERNAL(message);
        }
      }
    }
    return isValid;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void send_reply(ServerRequestInfo ri) {
    // CHECK deveria setar o currentConnection antigo?
    removeCurrentConnection(ri);

    // CHECK verificar se preciso limpar mais algum slot
    Any any = this.getMediator().getORB().create_any();
    try {
      ri.set_slot(this.getMediator().getSignedChainSlotId(), any);
      ri.set_slot(this.getMediator().getBusSlotId(), any);
    }
    catch (InvalidSlot e) {
      String message = "Falha inesperada ao limpar informaes nos slots";
      logger.log(Level.SEVERE, message, e);
      throw new INTERNAL(message);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void send_exception(ServerRequestInfo ri) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void send_other(ServerRequestInfo ri) {
  }

  /**
   * Gera um novo segredo.
   * 
   * @return o segredo.
   */
  private byte[] newSecret() {
    int size = 16;
    byte[] secret = new byte[size];
    Random random = new Random();
    random.nextBytes(secret);
    return secret;
  }

  /**
   * Configura a conexo corrente desta requisio.
   * 
   * @param ri informao da requisio
   * @param conn a conexo corrente.
   */
  private void setCurrentConnection(ServerRequestInfo ri, Connection conn) {
    try {
      long id = Thread.currentThread().getId();
      OpenBusContextImpl context = this.getMediator().getContext();
      Any any = this.getMediator().getORB().create_any();
      any.insert_longlong(id);
      ri.set_slot(context.getCurrentConnectionSlotId(), any);
      context.setConnectionByThreadId(id, conn);
    }
    catch (InvalidSlot e) {
      String message = "Falha inesperada ao acessar o slot da thread corrente";
      logger.log(Level.SEVERE, message, e);
      throw new INTERNAL(message);
    }
  }

  /**
   * Remove a conexo corrente do slot de ThreadId
   * 
   * @param ri a informao do request.
   */
  private void removeCurrentConnection(ServerRequestInfo ri) {
    try {
      OpenBusContextImpl context = this.getMediator().getContext();
      Any slot = ri.get_slot(context.getCurrentConnectionSlotId());
      if (slot.type().kind().value() != TCKind._tk_null) {
        long id = slot.extract_longlong();
        context.setConnectionByThreadId(id, null);
      }
      else {
        // nunca deveria acontecer.
        String message =
          "BUG: Falha inesperada ao acessar o slot da conexo corrente";
        logger.log(Level.SEVERE, message);
        throw new INTERNAL(message);
      }

      // limpando informao da conexo
      // TODO Dvida:
      /*
       * o orb garante que no modificou a informao no PICurrent da thread que
       * originalmente fez um setCurrentConnection?
       */
      Any any = this.getMediator().getORB().create_any();
      ri.set_slot(context.getCurrentConnectionSlotId(), any);
    }
    catch (InvalidSlot e) {
      String message = "Falha inesperada ao acessar o slot da thread corrente";
      logger.log(Level.SEVERE, message, e);
      throw new INTERNAL(message);
    }

  }
}
