package org.tecgraf.tdk.cache;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.geotools.data.DataUtilities;
import org.geotools.data.DefaultQuery;
import org.geotools.data.Query;
import org.geotools.data.store.EmptyFeatureCollection;
import org.geotools.factory.CommonFactoryFinder;
import org.geotools.factory.GeoTools;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureCollections;
import org.geotools.feature.FeatureIterator;
import org.geotools.feature.SchemaException;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.feature.Feature;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.filter.Filter;
import org.opengis.filter.FilterFactory2;
import org.opengis.filter.identity.Identifier;
import org.tecgraf.tdk.cache.query.QueryAnalyzer;

import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.index.SpatialIndex;
import com.vividsolutions.jts.index.quadtree.Quadtree;

public class BasicFeatureCacher implements FeatureCacher<SimpleFeatureType,SimpleFeature> {

	
	private static Logger _logger = Logger.getLogger(BasicFeatureCacher.class);
	
	SpatialIndex _index;
	private Query _cachedQuery;
	private FilterFactory2 _filterFactory = CommonFactoryFinder.getFilterFactory2(GeoTools.getDefaultHints());
	private CachingFeatureAccessFactory<SimpleFeatureType, SimpleFeature> _cachingFeatureAccessFactory;
	private ReferencedEnvelope _cachedBounds;

	private SimpleFeatureType _completeSchema;
	
	public BasicFeatureCacher(SimpleFeatureType completeSchema, CachingFeatureAccessFactory<SimpleFeatureType, SimpleFeature> cachingFeatureAccessFactory) {
		if(completeSchema == null) throw new IllegalArgumentException("completeSchema can't be null");
		if(cachingFeatureAccessFactory == null) throw new IllegalArgumentException("cachingFeatureAccessFactory can't be null");
		
		_cachingFeatureAccessFactory = cachingFeatureAccessFactory;
		_completeSchema = completeSchema;
		init();
	}


	@Override
	public void clear() {
		_logger.debug("Clearing cache for type '"+_cachedQuery.getTypeName()+"'");
		init();
	}


	private void init() {
		_cachedQuery = new DefaultQuery(_completeSchema.getTypeName(),Filter.EXCLUDE);
		_cachedBounds = new ReferencedEnvelope(_completeSchema.getCoordinateReferenceSystem());
		_index = new  Quadtree();
	}


	@Override
	public FeatureCollection<SimpleFeatureType, SimpleFeature> get(Query query)
	{
		
		
		if(query == null) throw new IllegalArgumentException("query can't be null");
		_logger.trace("Retrieving features for type '"+_cachedQuery.getTypeName()+"' and query "+query);
		
		SimpleFeatureType alternate = _completeSchema;
		try
		{
		    if (query.getPropertyNames() != Query.ALL_NAMES) 
		    {
			      alternate = DataUtilities.createSubType(_completeSchema, query.getPropertyNames());
			      if (alternate.equals(_completeSchema))
			          alternate = _completeSchema;
		    }
		}
		catch(SchemaException e)
		{
			_logger.error("Could not retype features to this new schema, returning features in full schema",e);
		}
		
		FeatureCollection<SimpleFeatureType,SimpleFeature> collection = FeatureCollections.newCollection();
		
		//Check type name
		if(!query.getTypeName().equalsIgnoreCase(_cachedQuery.getTypeName()))
		{
			_logger.warn("The cacher expects a different type name. '"+query.getTypeName()+"' is not '"+_cachedQuery.getTypeName()+"'.");
			return new EmptyFeatureCollection( _completeSchema );
		}
		
		//Check basic filters
		if(query.getFilter().equals(Filter.EXCLUDE))
		{
			_logger.warn("Query is using a 'exclude all' filter. Returning empty collection.");
			return new EmptyFeatureCollection( _completeSchema );
		}
		
		if(query.getFilter().equals(Filter.INCLUDE) || query.equals(_cachedQuery))
		{
			List<SimpleFeature> features = _index.query(_cachedBounds);
			for(SimpleFeature feature : features)
			{
				collection.add(new CachingSimpleFeature(DataUtilities.reType(alternate, feature)));
			}
			_logger.trace("Returning "+collection.size()+" features.");
			if(collection.size() == 0) collection = new EmptyFeatureCollection( _completeSchema );
			return collection;
		}
		
		//Check queries relation.
		//QueryAnalyzer queryAnalyzer = _cachingFeatureAccessFactory.createQueryAnalyzer(_cachedQuery);
		
		List<SimpleFeature> features;
		QueryAnalyzer subQueryAnalizer = _cachingFeatureAccessFactory.createQueryAnalyzer(query);
		Envelope envelope = subQueryAnalizer.getEnvelope();
		if(envelope.isNull()) 
		{
			features = _index.query(_cachedBounds);
		}
		else
		{
			features = _index.query(envelope);
		}
		
		for(SimpleFeature feature : features)
		{
			if(query.getFilter().evaluate(feature))
			{
				collection.add(new CachingSimpleFeature(DataUtilities.reType(alternate, feature)));
			}
		}
		
		_logger.trace("Returning "+collection.size()+" features.");
		if(collection.size() == 0) collection = new EmptyFeatureCollection( _completeSchema );
		return collection;
	}


	@Override
	public Query getCachedQuery() {
		return _cachedQuery;
	}


	@Override
	public void add(FeatureCollection<SimpleFeatureType, SimpleFeature> featureCollection, Query query) 
	{
		if(featureCollection == null) throw new IllegalArgumentException("featureCollection can't be null");
		if(query == null) throw new IllegalArgumentException("query can't be null");
		
		if(!_cachedQuery.getTypeName().equalsIgnoreCase(query.getTypeName()))
		{
			_logger.warn("Not adding features . Query type name differs from this cache type. Expected '"+_cachedQuery.getTypeName()+"' but was '"+query.getTypeName()+"'.");
			return;
		}
		
		if(!_cachedQuery.getTypeName().equalsIgnoreCase(featureCollection.getSchema().getTypeName()))
		{
			_logger.warn("Not adding features . Collection type name differs from this cache type. Expected '"+_cachedQuery.getTypeName()+"' but was '"+featureCollection.getSchema().getTypeName()+"'.");
			return;
		}
		
		_logger.debug("Refilling cache from " + query);
		
		if(featureCollection.size() == 0)
		{
			_logger.debug("Empty collection, nothing added to the cache.");
			return;
		}
		
		_logger.debug("Adding " + featureCollection.size()+" features.");
		
		
		
		if(_cachedQuery.getFilter().equals(Filter.EXCLUDE))
		{
			FeatureIterator<SimpleFeature> iterator = featureCollection.features();
			try
			{
				while(iterator.hasNext())
				{
					SimpleFeature feature = iterator.next();
					ReferencedEnvelope featureBounds = new ReferencedEnvelope(feature.getBounds()); 
					_cachedBounds.expandToInclude(featureBounds);
					_index.insert(featureBounds, feature);
				}
				_cachedQuery = query;
			}
			finally
			{
				iterator.close();
			}
			
		}
		else
		{
			QueryAnalyzer queryAnalyzer = _cachingFeatureAccessFactory.createQueryAnalyzer(_cachedQuery);
			FeatureIterator<SimpleFeature> iterator = featureCollection.features();
			//Filter filter = _cachedQuery.getFilter(); 
			try
			{
				while(iterator.hasNext())
				{
					SimpleFeature feature = iterator.next();
					//if(!filter.evaluate(feature))
					//{
						ReferencedEnvelope featureBounds = new ReferencedEnvelope(feature.getBounds()); 
						_cachedBounds.expandToInclude(featureBounds);
						_index.insert(new ReferencedEnvelope(feature.getBounds()), feature);
					//}
				}
			}
			finally
			{
				iterator.close();
			}
			
			if(!queryAnalyzer.isSubQuery(query))
			{
				_cachedQuery = new DefaultQuery(_cachedQuery.getTypeName(),_filterFactory.or(_cachedQuery.getFilter(),query.getFilter()));
			}
		}

	}

	@Override
	public void add(SimpleFeature feature)
	{
		if(feature == null) throw new IllegalArgumentException("feature can't be null");
		_logger.debug("Adding '"+feature.getType().getTypeName()+"->"+feature.getID()+"' to cache.");
		
		FeatureCollection<SimpleFeatureType,SimpleFeature> collection = FeatureCollections.newCollection();
		collection.add(feature);
		Set<Identifier> idSet = new HashSet<Identifier>();
		idSet.add(feature.getIdentifier());
		Query query = new DefaultQuery(_cachedQuery.getTypeName(),_filterFactory.id(idSet));

		add(collection,query);
		
//		_index.insert(new ReferencedEnvelope(feature.getBounds()), feature);
//		
//		if(_cachedQuery.getFilter().equals(Filter.EXCLUDE))
//		{
//			Set<Identifier> idSet = new HashSet<Identifier>();
//			idSet.add(feature.getIdentifier());
//			_cachedQuery = new DefaultQuery(_cachedQuery.getTypeName(),_filterFactory.id(idSet));
//			_cachedBounds = new ReferencedEnvelope(feature.getBounds());
//		}
//		else
//		{
//			if(!_cachedQuery.getFilter().evaluate(feature))
//			{
//				Set<Identifier> idSet = new HashSet<Identifier>();
//				idSet.add(feature.getIdentifier());
//				_cachedQuery = new DefaultQuery(_cachedQuery.getTypeName(),_filterFactory.or(_cachedQuery.getFilter(),_filterFactory.id(idSet)));
//				_cachedBounds.expandToInclude(new ReferencedEnvelope(feature.getBounds()));
//			}
//		}
		
	}

	@Override
	public void remove(SimpleFeature feature) {
		
		ReferencedEnvelope boundingBox = new ReferencedEnvelope(feature.getBounds());
		List<Feature> features = _index.query(boundingBox);
		
		for(Feature indexedFeature : features)
		{
			if(indexedFeature.getIdentifier().equals(feature.getIdentifier()))
			{
				_index.remove(boundingBox, indexedFeature);
				break;
			}
		}
	}


	@Override
	public void update(SimpleFeature feature) {
		if(feature instanceof CachingSimpleFeature)
		{
			CachingSimpleFeature cachingSimpleFeature = (CachingSimpleFeature) feature;
			SimpleFeature originalFeature = cachingSimpleFeature.getDelegate();
			
			ReferencedEnvelope oldBoundingBox = new ReferencedEnvelope(originalFeature.getBounds());
			List<Feature> features = _index.query(oldBoundingBox);
			
			//update the original feature values;
			originalFeature.setAttributes(cachingSimpleFeature.getAttributes());
			
			for(Feature indexedFeature : features)
			{
				if(indexedFeature.getIdentifier().equals(feature.getIdentifier()))
				{
					_index.remove(oldBoundingBox, indexedFeature);
					_index.insert(new ReferencedEnvelope(originalFeature.getBounds()), originalFeature);
					break;
				}
			}
		}
		else //Try to update the feature, probably it came from a non cached feature source, and feature and the indexed feature 
			// instances are different
		{
			List<Feature> features = _index.query(_cachedBounds);
	    	for(Feature indexedFeature : features)
	    	{
	    		if(indexedFeature.getIdentifier().equals(feature.getIdentifier()))
	    		{
	    			_index.remove(new ReferencedEnvelope(indexedFeature.getBounds()), indexedFeature);
	    			_index.insert(new ReferencedEnvelope(feature.getBounds()), feature);
	    			break;
	    		}
	    	}
		}
		
	}


	@Override
	public String getTypeName() {
		return _cachedQuery.getTypeName();
	}


    /* (non-Javadoc)
     * @see org.tecgraf.tdk.cache.FeatureCacher#getFeatureType()
     */
    @Override
    public SimpleFeatureType getFeatureType()
    {
        return _completeSchema;
    }

}
