package org.tecgraf.tdk.cache;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.geotools.feature.DecoratingFeature;
import org.geotools.feature.GeometryAttributeImpl;
import org.geotools.feature.type.Types;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.feature.GeometryAttribute;
import org.opengis.feature.IllegalAttributeException;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.AttributeType;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.feature.type.Name;
import org.opengis.filter.identity.Identifier;
import org.opengis.geometry.BoundingBox;

import com.vividsolutions.jts.geom.Geometry;

public class CachingSimpleFeature extends DecoratingFeature {

	Map<Integer,Object> _updatedValues = new HashMap<Integer, Object>();
	List<Object> _cachedValuesList;
	private Map<Object,Object> _cachedUserData;
	private Map<Object,Object>[] attributeUserData;
	private Map<String,Integer> _index;
	
	public CachingSimpleFeature(SimpleFeature delegate)
	{
		super(delegate);
		buildIndex(delegate.getType());
	}

	private void buildIndex(SimpleFeatureType type) {
		_index = new HashMap<String, Integer>();
		List<AttributeDescriptor> descriptors = type.getAttributeDescriptors();
		int index = 0;
		for(AttributeDescriptor descriptor : descriptors)
		{
			_index.put(descriptor.getLocalName(), index);
			index++;
		}
	}


	@Override
	public Object getAttribute(int index) {
		if(_updatedValues.containsKey(index))
		{
			return _updatedValues.get(index);
		}
		return super.getAttribute(index);
	}

	@Override
	public Object getAttribute(Name name) {
		return getAttribute(_index.get(name.getLocalPart()));
	}

	@Override
	public Object getAttribute(String path) {
		return getAttribute(_index.get(path));
	}

	@Override
	public List<Object> getAttributes() {
		if(_cachedValuesList != null) return _cachedValuesList;
		_cachedValuesList = new ArrayList<Object>();
		List<Object> originalValues = super.getAttributes();
		int index = 0 ;
		for(Object originalValue : originalValues)
		{
			if(_updatedValues.containsKey(index))
			{
				_cachedValuesList.add(_updatedValues.get(index));
			}
			else
			{
				_cachedValuesList.add(originalValue);
			}
			index++;
		}
		return _cachedValuesList;
	}

	@Override
	public BoundingBox getBounds() {
		ReferencedEnvelope bounds = new ReferencedEnvelope( super.getFeatureType().getCoordinateReferenceSystem() );
		List<Object> values = getAttributes();
		
		//Copied from SimpleFeatureImpl
        for ( Object o : values ) {
            if ( o instanceof Geometry ) {
                Geometry g = (Geometry) o;
                //TODO: check userData for crs... and ensure its of the same 
                // crs as the feature type
                if ( bounds.isNull() ) {
                    bounds.init(g.getEnvelopeInternal());
                }
                else {
                    bounds.expandToInclude(g.getEnvelopeInternal());
                }
            }
        }
        
        return bounds;
	}

	@Override
	public Object getDefaultGeometry() {
		return getAttribute(getFeatureType().getGeometryDescriptor().getName());
	}
	
	public GeometryAttribute getDefaultGeometryProperty() {
        GeometryDescriptor geometryDescriptor = getFeatureType().getGeometryDescriptor();
        GeometryAttribute geometryAttribute = null;
        if(geometryDescriptor != null){
            Object defaultGeometry = getDefaultGeometry();
            geometryAttribute = new GeometryAttributeImpl(defaultGeometry, geometryDescriptor, null);            
        }
        return geometryAttribute;
    }

    public void setDefaultGeometryProperty(GeometryAttribute geometryAttribute) {
        if(geometryAttribute != null)
            setDefaultGeometry(geometryAttribute.getValue());
        else
            setDefaultGeometry(null);
    }
	
    public Collection<Property> getProperties(Name name) {
        return getProperties( name.getLocalPart() );
    }

    public Collection<Property> getProperties(String name) {
        final Integer idx = _index.get(name);
        if(idx != null) {
            // cast temporarily to a plain collection to avoid type problems with generics
            Collection c = Collections.singleton( new Attribute( idx ) );
            return c;
        } else {
            return Collections.emptyList();
        }
    }

    public Property getProperty(Name name) {
        return getProperty( name.getLocalPart() );
    }

    public Property getProperty(String name) {
        final Integer idx = _index.get(name);
        if(idx == null){
            return null;
        }else{
            int index = idx.intValue();
            AttributeDescriptor descriptor = getFeatureType().getDescriptor(index);
            if(descriptor instanceof GeometryDescriptor){
                return new GeometryAttributeImpl(getAttribute(idx), (GeometryDescriptor) descriptor, null); 
            }else{
                return new Attribute( index );
            }
        }
    }
    
	@Override
	public Collection<Property> getProperties() {
		return new AttributeList();
	}

	@Override
	public Map<Object, Object> getUserData() {
		if(_cachedUserData == null)
		{
			_cachedUserData = new HashMap<Object, Object>(super.getUserData());
		}
		return _cachedUserData;
	}

	@Override
	public Collection<? extends Property> getValue() {
		return new AttributeList();
	}

	@Override
	public void setAttribute(int index, Object value) {
		_updatedValues.put(index, value);
		_cachedValuesList = null;
	}
	
	@Override
	public void setAttribute(Name name, Object value) {
		setAttribute(_index.get(name.getLocalPart()), value);
	}

	@Override
	public void setAttribute(String path, Object value) {
		setAttribute(_index.get(path), value);
	}

	@Override
	public void setAttributes(List<Object> values) {
		int index = 0;
		for(Object value : values)
		{
			setAttribute(index, value);
			index++;
		}
	}

	@Override
	public void setAttributes(Object[] values) {
		setAttributes(Arrays.asList(values));
	}

	@Override
	public void setDefaultGeometry(Geometry geometry)
			throws IllegalAttributeException {
		setAttribute(getDefaultGeometryProperty().getName(),geometry);
	}

	@Override
	public void setDefaultGeometry(Object value) {
		setAttribute(getDefaultGeometryProperty().getName(),value);
	}

	@Override
	public void setValue(Collection<Property> properties) {
		int index = 0;
		for(Property property : properties)
		{
			setAttribute(index, property.getValue());
			index++;
		}
	}
	
	@Override
	public void setValue(Object value) {
		setValue( (Collection<Property>) value);
	}

	protected SimpleFeature getDelegate()
	{
		return delegate;
	}
	

	/**
     * Live collection backed directly on the value array
     */
    class AttributeList extends AbstractList<Property> {

        public Attribute get(int index) {
            return new Attribute( index );
        }
        
        public Attribute set(int index, Property element) {
            setAttribute(index, element.getValue());
            return null;
        }
        
        public int size() {
            return getAttributeCount();
        }
    }

    /**
     * Attribute that delegates directly to the value array
     */
    class Attribute implements org.opengis.feature.Attribute {

        int index;
		
        
        Attribute( int index ) {
            this.index = index;
        }
        
        public Identifier getIdentifier() {
            return null;
        }

        public AttributeDescriptor getDescriptor() {
            return getFeatureType().getDescriptor(index);
        }

        public AttributeType getType() {
            return getFeatureType().getType(index);
        }

        public Name getName() {
            return getDescriptor().getName();
        }

        public Map<Object, Object> getUserData() {
            // lazily create the user data holder
            if(attributeUserData == null)
                attributeUserData = new HashMap[getAttributeCount()];
            // lazily create the attribute user data
            if(attributeUserData[index] == null)
                attributeUserData[index] = new HashMap<Object, Object>();
            return attributeUserData[index];
        }

        public Object getValue() {
            return getAttribute(index);
        }

        public boolean isNillable() {
            return getDescriptor().isNillable();
        }

        public void setValue(Object newValue) {
            setAttribute(index, newValue);
        }
        
        public void validate() {
            Types.validate(getDescriptor(), getAttribute(index));
        }
        
    }
	
}
