/*
 * #%L
 * JSQLParser library
 * %%
 * Copyright (C) 2004 - 2017 JSQLParser
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 2.1 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-2.1.html>.
 * #L%
 */
package net.sf.jsqlparser.util.deparser;

import java.util.Iterator;

import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.ExpressionVisitor;
import net.sf.jsqlparser.expression.operators.relational.ExpressionList;
import net.sf.jsqlparser.expression.operators.relational.ItemsListVisitor;
import net.sf.jsqlparser.expression.operators.relational.MultiExpressionList;
import net.sf.jsqlparser.schema.Column;
import net.sf.jsqlparser.statement.select.SelectVisitor;
import net.sf.jsqlparser.statement.select.SubSelect;
import net.sf.jsqlparser.statement.select.WithItem;
import net.sf.jsqlparser.statement.upsert.Upsert;

/**
 * A class to de-parse (that is, tranform from JSqlParser hierarchy into a string)
 * @author messfish
 *
 */
public class UpsertDeParser implements ItemsListVisitor {

    private StringBuilder buffer;
    private ExpressionVisitor expressionVisitor;
    private SelectVisitor selectVisitor;
    
    /**
     * Constructor: this constructor is used to assign the values coming from the
     * arguments to their global values.
     * @param expressionVisitor a {@link ExpressionVisitor} to de-parse
     * {@link net.sf.jsqlparser.expression.Expression}s. It has to share the same<br>
     * StringBuilder (buffer parameter) as this object in order to work
     * @param selectVisitor a {@link SelectVisitor} to de-parse
     * {@link net.sf.jsqlparser.statement.select.Select}s. It has to share the same<br>
     * StringBuilder (buffer parameter) as this object in order to work
     * @param buffer the buffer that will be filled with the insert
     */
    public UpsertDeParser(ExpressionVisitor expressionVisitor, SelectVisitor selectVisitor, StringBuilder buffer) {
        this.buffer = buffer;
        this.expressionVisitor = expressionVisitor;
        this.selectVisitor = selectVisitor;
    }

    public StringBuilder getBuffer() {
        return buffer;
    }

    public void setBuffer(StringBuilder buffer) {
        this.buffer = buffer;
    }

    public void deParse(Upsert upsert) {
        buffer.append("UPSERT INTO ");

        buffer.append(upsert.getTable().getFullyQualifiedName());
        if (upsert.getColumns() != null) {
            appendColumns(upsert);
        }

        if (upsert.getItemsList() != null) {
            upsert.getItemsList().accept(this);
        }

        if (upsert.getSelect() != null) {
            appendSelect(upsert);
        }

        if (upsert.isUseDuplicate()) {
            appendDuplicate(upsert);
        }

    }
    
    private void appendColumns(Upsert upsert) {
        buffer.append(" (");
        for (Iterator<Column> iter = upsert.getColumns().iterator(); iter.hasNext();) {
            Column column = iter.next();
            buffer.append(column.getColumnName());
            if (iter.hasNext()) {
                buffer.append(", ");
            }
        }
        buffer.append(")");
    }
    
    private void appendSelect(Upsert upsert) {
        buffer.append(" ");
        if (upsert.isUseSelectBrackets()) {
            buffer.append("(");
        }
        if (upsert.getSelect().getWithItemsList() != null) {
            buffer.append("WITH ");
            for (WithItem with : upsert.getSelect().getWithItemsList()) {
                with.accept(selectVisitor);
            }
            buffer.append(" ");
        }
        upsert.getSelect().getSelectBody().accept(selectVisitor);
        if (upsert.isUseSelectBrackets()) {
            buffer.append(")");
        }
    }
    
    private void appendDuplicate(Upsert upsert) {
        buffer.append(" ON DUPLICATE KEY UPDATE ");
        for (int i = 0; i < upsert.getDuplicateUpdateColumns().size(); i++) {
            Column column = upsert.getDuplicateUpdateColumns().get(i);
            buffer.append(column.getFullyQualifiedName()).append(" = ");

            Expression expression = upsert.getDuplicateUpdateExpressionList().get(i);
            expression.accept(expressionVisitor);
            if (i < upsert.getDuplicateUpdateColumns().size() - 1) {
                buffer.append(", ");
            }
        }
    }

    @Override
    public void visit(ExpressionList expressionList) {
        buffer.append(" VALUES (");
        for (Iterator<Expression> iter = expressionList.getExpressions().iterator(); iter.hasNext();) {
            Expression expression = iter.next();
            expression.accept(expressionVisitor);
            if (iter.hasNext()) {
                buffer.append(", ");
            }
        }
        buffer.append(")");
    }

    @Override
    public void visit(MultiExpressionList multiExprList) {
        buffer.append(" VALUES ");
        for (Iterator<ExpressionList> it = multiExprList.getExprList().iterator(); it.hasNext();) {
            buffer.append("(");
            for (Iterator<Expression> iter = it.next().getExpressions().iterator(); iter.hasNext();) {
                Expression expression = iter.next();
                expression.accept(expressionVisitor);
                if (iter.hasNext()) {
                    buffer.append(", ");
                }
            }
            buffer.append(")");
            if (it.hasNext()) {
                buffer.append(", ");
            }
        }
    }

    @Override
    public void visit(SubSelect subSelect) {
        subSelect.getSelectBody().accept(selectVisitor);
    }

    public ExpressionVisitor getExpressionVisitor() {
        return expressionVisitor;
    }

    public SelectVisitor getSelectVisitor() {
        return selectVisitor;
    }

    public void setExpressionVisitor(ExpressionVisitor visitor) {
        expressionVisitor = visitor;
    }

    public void setSelectVisitor(SelectVisitor visitor) {
        selectVisitor = visitor;
    }

}
