package io.quarkus.arc;

import java.lang.annotation.Annotation;

/**
 * Superclass of all annotation literals generated by ArC. Redeclares some {@link Annotation}
 * methods to be able to replace interface type checks with class type checks. There are
 * two typical use cases:
 * <ol>
 * <li>checking against {@code Annotation} can be replaced with checking against
 * {@code AbstractAnnotationLiteral}, when the caller knows that the annotation instance
 * is likely to come from ArC;</li>
 * <li>checking against some concrete annotation type can be replaced with checking
 * equality to {@code annotationType()}.</li>
 * </ol>
 *
 * <p>
 * Replacing interface type checks with class type checks is useful on the HotSpot JVM.
 * Its type check algorithm has a fast path for classes and a slow path for interfaces [1].
 * That slow path is fronted with a single-element cache, containing the last successfully
 * tested type. This often improves the slow path performance, but in presence of repeated
 * successful type checks against <em>different</em> interfaces, the cache never hits and
 * keeps changing all the time.
 * <p>
 * Moreover, if such pathologic type checks happen concurrently on multiple CPU cores,
 * those CPU cores will compete for the ability to write to the CPU cache line containing
 * the single-element type cache. This leads to significant increase in CPU cache coherence
 * traffic and worse performance.
 * <p>
 * [1] This is a simplification. Fast path is only possible for classes that don't have
 * too many superclasses, which is fortunately the common case. Type checks against classes
 * that are deep in an inheritance hierarchy have to go through the slow path too.
 * </p>
 * References:
 * <ul>
 * <li><a href="http://cr.openjdk.java.net/~jrose/jvm/checktype-2001.txt">John Rose & Cliff Click:
 * Fast and Decisive Class Checking in the Hotspot VM</a></li>
 * <li><a href="https://bugs.openjdk.org/browse/JDK-8180450">JDK-8180450</a></li>
 * <li><a href="https://github.com/RedHatPerf/type-pollution-agent">Java agent to diagnose JDK-8180450</a></li>
 * </ul>
 * <p>
 * Note: replacing interface type checks with class type checks as described above may easily
 * lead to non-idiomatic, harder to understand code. Demonstrating measurable performance
 * improvements should be a necessary prerequisite for using this technique.
 */
public abstract class AbstractAnnotationLiteral {
    /**
     * Same as {@link Annotation#annotationType()}.
     *
     * @return the annotation type of this annotation literal
     */
    public abstract Class<? extends Annotation> annotationType();
}
