/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.internal.primitive;

import java.util.Collection;

import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.collection.primitive.MutableCharCollection;
import com.gs.collections.api.iterator.CharIterator;

/**
 * The CharIteratorIterate class provides implementations of the various iteration patterns for use with the {@link CharIterator}.
 * This file was automatically generated from template file primitiveIteratorIterate.stg.
 *
 * @since 5.0
 */
public final class CharIteratorIterate
{
    private CharIteratorIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    public static void forEach(CharIterator iterator, CharProcedure procedure)
    {
        while (iterator.hasNext())
        {
            procedure.value(iterator.next());
        }
    }

    public static <R extends MutableCharCollection> R select(
            CharIterator iterator,
            CharPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            char item = iterator.next();
            if (predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <R extends MutableCharCollection> R reject(
            CharIterator iterator,
            CharPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            char item = iterator.next();
            if (!predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <V, R extends Collection<V>> R collect(
            CharIterator iterator,
            CharToObjectFunction<? extends V> function,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            char item = iterator.next();
            targetCollection.add(function.valueOf(item));
        }
        return targetCollection;
    }

    public static char detectIfNone(CharIterator iterator, CharPredicate predicate, char ifNone)
    {
        while (iterator.hasNext())
        {
            char item = iterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public static int count(CharIterator iterator, CharPredicate predicate)
    {
        int count = 0;
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                count++;
            }
        }
        return count;
    }

    public static boolean anySatisfy(CharIterator iterator, CharPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean allSatisfy(CharIterator iterator, CharPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (!predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static boolean noneSatisfy(CharIterator iterator, CharPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static <T> T injectInto(CharIterator iterator, T injectedValue, ObjectCharToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        while (iterator.hasNext())
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }


    public static long sum(CharIterator iterator)
    {
        long sum = 0L;
        while (iterator.hasNext())
        {
            sum += iterator.next();
        }
        return sum;
    }

    public static char max(CharIterator iterator)
    {
        char max = iterator.next();
        while (iterator.hasNext())
        {
            char next = iterator.next();
            if (max < next)
            {
                max = next;
            }
        }
        return max;
    }

    public static char min(CharIterator iterator)
    {
        char min = iterator.next();
        while (iterator.hasNext())
        {
            char next = iterator.next();
            if (next < min)
            {
                min = next;
            }
        }
        return min;
    }
}
