/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.internal.primitive;

import java.util.Collection;

import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortProcedure;
import com.gs.collections.api.collection.primitive.MutableShortCollection;
import com.gs.collections.api.iterator.ShortIterator;

/**
 * The ShortIteratorIterate class provides implementations of the various iteration patterns for use with the {@link ShortIterator}.
 * This file was automatically generated from template file primitiveIteratorIterate.stg.
 *
 * @since 5.0
 */
public final class ShortIteratorIterate
{
    private ShortIteratorIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    public static void forEach(ShortIterator iterator, ShortProcedure procedure)
    {
        while (iterator.hasNext())
        {
            procedure.value(iterator.next());
        }
    }

    public static <R extends MutableShortCollection> R select(
            ShortIterator iterator,
            ShortPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            short item = iterator.next();
            if (predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <R extends MutableShortCollection> R reject(
            ShortIterator iterator,
            ShortPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            short item = iterator.next();
            if (!predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <V, R extends Collection<V>> R collect(
            ShortIterator iterator,
            ShortToObjectFunction<? extends V> function,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            short item = iterator.next();
            targetCollection.add(function.valueOf(item));
        }
        return targetCollection;
    }

    public static short detectIfNone(ShortIterator iterator, ShortPredicate predicate, short ifNone)
    {
        while (iterator.hasNext())
        {
            short item = iterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public static int count(ShortIterator iterator, ShortPredicate predicate)
    {
        int count = 0;
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                count++;
            }
        }
        return count;
    }

    public static boolean anySatisfy(ShortIterator iterator, ShortPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean allSatisfy(ShortIterator iterator, ShortPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (!predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static boolean noneSatisfy(ShortIterator iterator, ShortPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static <T> T injectInto(ShortIterator iterator, T injectedValue, ObjectShortToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        while (iterator.hasNext())
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }


    public static long sum(ShortIterator iterator)
    {
        long sum = 0L;
        while (iterator.hasNext())
        {
            sum += iterator.next();
        }
        return sum;
    }

    public static short max(ShortIterator iterator)
    {
        short max = iterator.next();
        while (iterator.hasNext())
        {
            short next = iterator.next();
            if (max < next)
            {
                max = next;
            }
        }
        return max;
    }

    public static short min(ShortIterator iterator)
    {
        short min = iterator.next();
        while (iterator.hasNext())
        {
            short next = iterator.next();
            if (next < min)
            {
                min = next;
            }
        }
        return min;
    }
}
