/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.impl.factory.primitive.ShortLists;
import com.gs.collections.impl.lazy.primitive.CollectShortToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectShortIterable;

/**
 * LazyShortIterate is a factory class which creates "deferred" short iterables around the specified short iterables. A "deferred"
 * short iterable performs some operation, such as filtering or transforming, when the result short iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyShortIterate
{
    private static final LazyShortIterable EMPTY_ITERABLE = ShortLists.immutable.of().asLazy();

    private LazyShortIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred short iterable for the specified short iterable.
     */
    public static LazyShortIterable adapt(ShortIterable iterable)
    {
        return new LazyShortIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering short iterable for the specified short iterable.
     */
    public static LazyShortIterable select(ShortIterable iterable, ShortPredicate predicate)
    {
        return new SelectShortIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming short iterable for the specified short iterable.
     */
    public static <V> LazyIterable<V> collect(
            ShortIterable iterable,
            ShortToObjectFunction<? extends V> function)
    {
        return new CollectShortToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming short iterable for the specified short iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            ShortIterable iterable,
            ShortPredicate predicate,
            ShortToObjectFunction<? extends V> function)
    {
        return LazyShortIterate.select(iterable, predicate).collect(function);
    }

    public static LazyShortIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
