package csbase.client.algorithms.commands.newview;

import java.awt.Window;
import java.util.concurrent.atomic.AtomicBoolean;

import tecgraf.javautils.gui.StatusBar;
import csbase.client.util.event.EventListener;
import csbase.client.util.gui.log.AutoReloadable;
import csbase.client.util.gui.log.LogPanelReloader;
import csbase.client.util.gui.log.LogPanelReloader.StartEvent;
import csbase.client.util.gui.log.LogPanelReloader.StopEvent;

/**
 * Base para implementao de uma aba de logs no visualizador de comandos.
 */
abstract class AbstractLogTab extends AbstractTab implements ReloadableTab,
  AutoReloadable {

  /**
   * Indica se o painel no foi selecionado ainda.
   */
  protected final AtomicBoolean wasNeverSelected;
  /**
   * Indica se o evento de atualizao de status deve ser ignorado.
   */
  protected final AtomicBoolean updateAutoReloadStatus;
  /**
   * Indica se o painel deve ser recarredao automaticamente.
   */
  protected final AtomicBoolean autoReload;

  /**
   * Controlador de recarga do painel de modo temporizado.
   */
  protected final LogPanelReloader reloader;
  /**
   * Barra de status utilizada para mostrar mensagens e erros da aba.
   */
  protected final StatusBar statusBar;

  /**
   * Construtor.
   * 
   * @param title o ttulo da aba de log.
   * @param statusBar barra de status para apresentar mensagens da aba.
   * @param owner janela dona da aba.
   * @param reload indica se a aba deve ser recarregada automaticamente de modo
   *        temporizado.
   */
  public AbstractLogTab(String title, StatusBar statusBar, Window owner,
    boolean reload) {
    super(TabType.RELOADABLE, title, owner);
    initialize();
    this.statusBar = statusBar;
    this.reloader = createReloader();

    this.wasNeverSelected = new AtomicBoolean(true);

    this.updateAutoReloadStatus = new AtomicBoolean(false);

    this.autoReload = new AtomicBoolean(reload);

    /*
     * Precisamos de listeners especficos para cada evento ao invs de um que
     * s avisa quando o estado mudou. S assim conseguimos atualizar o estado
     * do indicador de auto-recarregamento mesmo quando o timer 
     * ligado/desligado programaticamente enquanto a aba no est selecionada.
     */
    this.reloader.addStopEventListener(new EventListener<StopEvent>() {
      @Override
      public void eventFired(StopEvent event) {
        setAutoReload(false);
      }
    });

    this.reloader.addStartEventListener(new EventListener<StartEvent>() {
      @Override
      public void eventFired(StartEvent event) {
        setAutoReload(true);
      }
    });
  }

  /**
   * Atribui um novo estado ao indicador de auto-recarregamento, se essa opo
   * estiver habilitada.
   * 
   * @param reload o novo estado.
   */
  private void setAutoReload(boolean reload) {
    if (!updateAutoReloadStatus.compareAndSet(true, false)) {
      this.autoReload.set(reload);
    }
  }

  /**
   * Mtodo que possibilita classes filhas a inicializarem atributos antes da
   * inicializao dos componentes grficos da classe.
   */
  protected void initialize() {
    /*
     * Implementao "dummy" para no obrigar todas as subclasses a
     * implementarem o mtodo.
     */
  }

  /**
   * Cria o controlador de recargas do painel.
   * 
   * @return o controlador de recargas do painel.
   */
  protected LogPanelReloader createReloader() {
    return new LogPanelReloader(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSelected(boolean selected) {
    super.setSelected(selected);
    /*
     * Se a tab de log est sendo selecionada pela primeira vez e o comando
     * ainda no terminou, a recarga de log deve estar ligada. Se este no for o
     * caso, a recarga deve ficar desligada se esta tab estiver sendo
     * desselecionada ou deve ficar no ltimo estado em que o usurio a deixou.
     * <br>
     * 
     * O updateAutoReloadStatus permite que ao sair desta aba, o reloader seja
     * desligado, mas que o estado escolhido pelo usurio atravs do boto de
     * reload ou pelo fato do comando estar em execuo, seja armazenado no
     * wasRunningStatus. Assim, os eventos gerados pela ativao ou desativao
     * do reload a partir da seleo de abas so ignorados e os demais - usurio
     * escolhendo o estado atravs do boto de recarga - so pegos e o estado da
     * recarga  salvo. Porm, como a lgica da 1a seleo  diferente, este
     * evento no deve ser ignorado.
     */

    boolean ignoreReloader = !wasNeverSelected.get();

    wasNeverSelected.compareAndSet(true, !selected);

    if (!autoReload.get()) {
      // A recarga estava desligada, ento deve continuar assim.
      return;
    }

    updateAutoReloadStatus.set(ignoreReloader);

    if (selected) {
      reloader.start();
    }
    else {
      reloader.stop();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LogPanelReloader getReloader() {
    return reloader;
  }

}