package csbase.client.algorithms.commands.newview;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.algorithms.commands.newview.AbstractTab.TabType;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.logic.CommandInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;

/**
 * Fbrica de visualizaes de comando.
 */
public class CommandViewFactory {

  /**
   * Template do identificador de viso de comandos.
   */
  private static final String VIEW_INDEX_TEMPLATE = "|%s|";

  /**
   * Template do identificador de viso de ns de fluxo.
   */
  private static final String NODE_INDEX_TEMPLATE = VIEW_INDEX_TEMPLATE
    + VIEW_INDEX_TEMPLATE;

  /**
   * Cria a visualizao de um nico algoritmo (n) pertencente a um fluxo.
   * 
   * @param command O comando a ser visualizado (No aceita {@code null}).
   * @param owner Componente dono da viso (No aceita {@code null}).
   * @param tabType O tipo ({@link TabType}) da aba que deve ser selecionada ao
   *        abrir aa visualizao.
   * @param node n do fluxo a ser visualizado (No aceita {@code null}).
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  public static void showNodeView(CommandInfo command,
    DesktopComponentFrame owner, TabType tabType, GraphNode node)
    throws ClientException {
    if (null == command) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    if (null == owner) {
      throw new IllegalArgumentException("O parmetro owner est nulo.");
    }

    String index =
      String.format(NODE_INDEX_TEMPLATE, owner.getIndex(), node.getId());

    showView(CommandViewType.NODE, command, node.getAlgorithmConfiguratorView()
      .getConfigurator(), owner, tabType, index, node.getId());
  }

  /**
   * Cria a visualizao de um comando.
   * 
   * @param command O comando a ser visualizado (No aceita {@code null}).
   * @param owner Componente dono da viso (No aceita {@code null}).
   * @param tabType O tipo ({@link TabType}) da aba que deve ser selecionada ao
   *        abrir a visualizao.
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  public static void showCommandView(CommandInfo command,
    DesktopComponentFrame owner, TabType tabType) throws ClientException {
    if (null == command) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    if (null == owner) {
      throw new IllegalArgumentException("O parmetro owner est nulo.");
    }

    String index = String.format(VIEW_INDEX_TEMPLATE, command.getId());
    AlgorithmConfigurator configurator = getConfigurator(command, owner);

    CommandViewType type;
    if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
      type = CommandViewType.FLOW;
    }
    else {
      type = CommandViewType.SIMPLE;
    }
    showView(type, command, configurator, owner, tabType, index);
  }

  /**
   * Cria a visualizao de um comando.
   * 
   * @param viewType O tipo ({@link CommandViewType}) de viso do comando a ser
   *        mostrada.
   * @param command O comando a ser visualizado (No aceita {@code null}).
   * @param configurator O configurador do algoritmo executado pelo comando.
   * @param owner Componente dono da viso (No aceita {@code null}).
   * @param tabType O tipo ({@link TabType}) da aba que deve ser selecionada ao
   *        abrir a visualizao.
   * @param index Identificador da visualizao. Usado para garantir a
   *        unicidade.
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  private static void showView(CommandViewType viewType, CommandInfo command,
    AlgorithmConfigurator configurator, DesktopComponentFrame owner,
    TabType tabType, String index) throws ClientException {
    showView(viewType, command, configurator, owner, tabType, index, null);
  }

  /**
   * Cria a visualizao de um comando.
   * 
   * @param viewType O tipo ({@link CommandViewType}) de viso do comando a ser
   *        mostrada.
   * @param command O comando a ser visualizado (No aceita {@code null}).
   * @param configurator O configurador do algoritmo executado pelo comando.
   * @param owner Componente dono da viso (No aceita {@code null}).
   * @param tabType O tipo ({@link TabType}) da aba que deve ser selecionada ao
   *        abrir a visualizao.
   * @param index Identificador da visualizao. Usado para garantir a
   *        unicidade.
   * @param nodeId n do fluxo a ser visualizado (se for o caso).
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  private static void showView(CommandViewType viewType, CommandInfo command,
    AlgorithmConfigurator configurator, DesktopComponentFrame owner,
    TabType tabType, String index, Integer nodeId) throws ClientException {

    CommandViewFrame view =
      (CommandViewFrame) DesktopComponentFrame.getDesktopComponentFrame(index);

    if (view == null) {
      CommandView panel;
      switch (viewType) {
        case FLOW:
          panel =
            new FlowCommandView(command,
              (FlowAlgorithmConfigurator) configurator);
          break;
        case SIMPLE:
          panel = new SimpleCommandView(command, configurator);
          break;
        case NODE:
          if (nodeId == null) {
            throw new IllegalArgumentException("O parmetro node est nulo.");
          }
          panel = new NodeCommandView(nodeId, command, configurator);
          break;
        default:
          throw new ClientException(LNG.get(CommandViewFactory.class
            .getSimpleName() + ".error.view.unknown"));
      }
      view = new CommandViewFrame(panel, index, owner, command);
    }

    view.showTab(tabType);
  }

  /**
   * Obtm o configurador do comando especificado.
   * 
   * @param command O comando a ser visualizado (No aceita {@code null}).
   * @param owner Componente dono da viso (No aceita {@code null}).
   * @return o configurador.
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  private static AlgorithmConfigurator getConfigurator(
    final CommandInfo command, DesktopComponentFrame owner)
    throws ClientException {
    RemoteTask<AlgorithmConfigurator> task =
      new RemoteTask<AlgorithmConfigurator>() {
        @Override
        public void performTask() throws Exception {
          AlgorithmConfigurator configurator = command.getConfigurator();
          setResult(configurator);
        }
      };
    task.setProgressDialogDelay(1);
    String msg =
      LNG.get(CommandViewFactory.class.getSimpleName()
        + ".message.loading.configurator");
    String title =
      LNG.get(CommandViewFactory.class.getSimpleName() + ".window.title");
    if (!task.execute(owner, title, msg)) {
      throw new ClientException(task.getError());
    }
    AlgorithmConfigurator configurator = task.getResult();
    return configurator;
  }
}
