package csbase.client.algorithms.parameters;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.gui.GBC;
import csbase.client.util.ClientUtilities;
import csbase.logic.algorithms.parameters.ListParameter;

/**
 * Base para a viso de parmetros do tipo lista.
 * 
 * @param <E> Tipo do elemento contido na lista.
 * @author Tecgraf / PUC-Rio
 */
public abstract class ListParameterView<E> extends SimpleParameterView<List<E>> {

  /**
   * Cria a viso em modo {@link ParameterView.Mode#CONFIGURATION}.
   * 
   * @param window NO EST SENDO UTILIZADO. Existe somente para manter
   *        compatibilidade com o WebSintesi.
   * @param parameter O parmetro que  manipulado por esta viso (No aceita
   * @param parser o analisador de itens da lista.
   * @param formatter o formatador de itens da lista. {@code null}).
   * @deprecated para manter compatibilidade com o WebSintesi
   */
  @Deprecated
  protected ListParameterView(Window window, ListParameter<E> parameter,
    Parser<E> parser, Formatter<E> formatter) {
    this(parameter, Mode.CONFIGURATION, parser, formatter);
  }

  /**
   * Cria a viso.
   * 
   * @param parameter O parmetro que  manipulado por esta viso (No aceita
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   * @param parser o analisador de itens da lista.
   * @param formatter o formatador de itens da lista. {@code null}).
   *        {@link ParameterView.Mode#REPORT}.
   */
  protected ListParameterView(ListParameter<E> parameter, Mode mode,
    Parser<E> parser, Formatter<E> formatter) {
    this(parameter, mode, parser, formatter, null);
  }

  /**
   * Cria a viso em modo {@link ParameterView.Mode#CONFIGURATION}.
   * 
   * @param window NO EST SENDO UTILIZADO. Existe somente para manter
   *        compatibilidade com o WebSintesi.
   * @param parameter O parmetro que  manipulado por esta viso (No aceita
   * @param parser o analisador de itens da lista.
   * @param formatter o formatador de itens da lista.
   * @param textField o campo de texto utilizado na viso. {@code null}).
   * @deprecated para manter compatibilidade com o WebSintesi
   */
  @Deprecated
  protected ListParameterView(Window window, ListParameter<E> parameter,
    Parser<E> parser, Formatter<E> formatter, JTextField textField) {
    this(parameter, Mode.CONFIGURATION, parser, formatter, textField);
  }

  /**
   * Cria a viso.
   * 
   * @param parameter O parmetro que  manipulado por esta viso (No aceita
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   * @param parser o analisador de itens da lista.
   * @param formatter o formatador de itens da lista.
   * @param textField o campo de texto utilizado na viso. {@code null}).
   *        {@link ParameterView.Mode#REPORT}.
   */
  protected ListParameterView(ListParameter<E> parameter, Mode mode,
    Parser<E> parser, Formatter<E> formatter, JTextField textField) {
    super(parameter, mode, parser, formatter, textField);
    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ListParameter<E> getParameter() {
    return (ListParameter<E>) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    Parser<E> parser = (Parser<E>) componentArgs[0];
    Formatter<E> formatter = (Formatter<E>) componentArgs[1];
    JTextField textField = (JTextField) componentArgs[2];
    return new ListConfigurationParameter(parser, formatter, textField);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    Formatter<E> formatter = (Formatter<E>) componentArgs[1];
    return new ListReportParameter(formatter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateViewContents() {
    ((IListParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * Interface para o formatador de elementos da lista, que representa o
   * elemento de forma textual.
   * 
   * @param <E> Tipo do elemento contido na lista.
   */
  protected static interface Formatter<E> {

    /**
     * Representa o elemento indicado de forma textual.
     * 
     * @param element o elemento.
     * @return a string que representa o elemento.
     */
    String format(E element);
  }

  /**
   * A interface do analisador de elementos da lista, que recria o elemento a
   * partir de sua representao textual.
   * 
   * @param <E> the element type
   */
  protected static interface Parser<E> {

    /**
     * Analisa o texto para recriar o elemento da lista.
     * 
     * @param text o texto a ser analisado.
     * @return o elemento recriado.
     */
    E parse(String text);
  }

  /**
   * Interface comum s diferentes vises do parmetro.
   */
  private interface IListParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#REPORT}.
   */
  private final class ListReportParameter extends JPanel implements
    IListParameterComponent {

    /** O formatador de itens da lista. */
    private final Formatter<E> formatter;

    /** A rea de texto utilizada na viso. */
    private JTextArea textArea;

    /**
     * Construtor.
     * 
     * @param formatter o formatador de itens da lista.
     */
    ListReportParameter(Formatter<E> formatter) {
      this.formatter = formatter;

      setLayout(new GridLayout());

      textArea = new JTextArea();
      textArea.setToolTipText(getParameter().getDescription());
      ComponentProperties.setProperties(textArea, Mode.REPORT, false);

      JScrollPane scroll = new JScrollPane(textArea);
      scroll
        .setBorder(ComponentProperties.getInstance(Mode.REPORT).getBorder());
      updateViewContents();
      add(scroll, new GBC(0, 0).both());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      List<E> values = getParameter().getValue();
      if (values == null) {
        textArea.setRows(1);
      }
      else {
        textArea.setRows(Math.max(1, Math.min(values.size(), 6)));

        StringBuffer sb = new StringBuffer();
        for (E value : values) {
          sb.append(formatter.format(value)).append('\n');
        }
        // remove o ltimo breakLine
        String text = sb.substring(0, sb.length() - 1);
        textArea.setText(text);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean enabled) {
      super.setEnabled(enabled);
      textArea.setEnabled(enabled);
    }
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#CONFIGURATION}.
   */
  private final class ListConfigurationParameter extends JPanel implements
    IListParameterComponent {

    /** O analisador de itens da lista. */
    private Parser<E> parser;

    /** O formatador de itens da lista. */
    private Formatter<E> formatter;

    /** O boto de adicionar itens na lista. */
    private final JButton addButton;

    /** O boto de editar itens na lista. */
    private final JButton editButton;

    /** O componente visual que representa a lista. */
    private final JList list;

    /** O modelo do componente visual. */
    private final DefaultListModel model;

    /** O boto de remover itens na lista. */
    private final JButton removeButton;

    /** O campo de texto para utilizado para adicionar itens na lista. */
    private final JTextField textField;

    /**
     * Construtor.
     * 
     * @param parser O analisador de itens da lista.
     * @param formatter O formatador de itens da lista.
     * @param textField O campo de texto para utilizado para adicionar itens na
     *        lista.
     */
    ListConfigurationParameter(Parser<E> parser, Formatter<E> formatter,
      JTextField textField) {
      setParser(parser);
      setFormatter(formatter);

      setLayout(new BorderLayout());

      this.textField = null == textField ? new JTextField(50) : textField;
      this.textField.addActionListener(new AddActionListener());
      this.textField.getDocument().addDocumentListener(
        new AddDocumentListener());
      this.textField.setToolTipText(getParameter().getDescription());

      this.model = new DefaultListModel();

      this.list = new JList(this.model);
      this.list.getSelectionModel().setSelectionMode(
        ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
      this.list.addKeyListener(new RemoveKeyListener());
      this.list.addListSelectionListener(new EditListSelectionListener());
      Dimension preferredSize = this.textField.getPreferredSize();
      this.list.setFixedCellWidth((int) preferredSize.getWidth());
      this.list.setFixedCellHeight((int) preferredSize.getHeight());
      this.list.setToolTipText(getParameter().getDescription());

      JPanel editPanel = new JPanel();
      editPanel.setLayout(new BorderLayout());
      editPanel.add(this.textField, BorderLayout.NORTH);
      editPanel.add(new JScrollPane(this.list), BorderLayout.CENTER);
      add(editPanel, BorderLayout.CENTER);

      this.addButton = new JButton("Adicionar");
      this.addButton.addActionListener(new AddActionListener());
      String description = getParameter().getDescription();
      this.addButton.setToolTipText(description);
      this.addButton.setEnabled(false);
      this.editButton = new JButton("Editar...");
      this.editButton.addActionListener(new EditActionListener());
      this.editButton.setToolTipText(description);
      this.editButton.setEnabled(false);
      this.removeButton = new JButton("Remover");
      this.removeButton.addActionListener(new RemoveActionListener());
      this.removeButton.setToolTipText(description);
      this.removeButton.setEnabled(false);
      ClientUtilities.adjustEqualSizes(this.addButton, this.editButton,
        this.removeButton);
      JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout(new GridBagLayout());
      GridBagConstraints constraints = new GridBagConstraints();
      constraints.anchor = GridBagConstraints.CENTER;
      constraints.fill = GridBagConstraints.NONE;
      constraints.gridheight = 1;
      constraints.gridwidth = 1;
      constraints.gridx = 1;
      constraints.gridy = 1;
      constraints.insets = new Insets(2, 2, 2, 2);
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      buttonPanel.add(this.addButton, constraints);
      constraints.gridy++;
      buttonPanel.add(this.editButton, constraints);
      constraints.gridy++;
      buttonPanel.add(this.removeButton, constraints);
      add(buttonPanel, BorderLayout.EAST);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      super.setEnabled(isEnabled);
      this.textField.setEnabled(isEnabled);
      this.list.setEnabled(isEnabled);
      updateAddButton();
      updateEditButton();
      updateRemoveButton();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      this.model.removeAllElements();
      List<E> values = getParameter().getValue();
      if (values != null) {
        for (E value : values) {
          this.model.addElement(getFormatter().format(value));
        }
      }
    }

    /**
     * Obtm o formatador de itens da lista.
     * 
     * @return o formatador.
     */
    private Formatter<E> getFormatter() {
      return this.formatter;
    }

    /**
     * Obtm o analisador de itens da lista.
     * 
     * @return o analisador.
     */
    private Parser<E> getParser() {
      return this.parser;
    }

    /**
     * Atribui o formatador de itens da lista..
     * 
     * @param formatter o formatador.
     */
    private void setFormatter(Formatter<E> formatter) {
      if (formatter == null) {
        throw new IllegalArgumentException("O parmetro formatter est nulo.");
      }
      this.formatter = formatter;
    }

    /**
     * Atribui o analisador de itens da lista.
     * 
     * @param parser o analisador.
     */
    private void setParser(Parser<E> parser) {
      if (parser == null) {
        throw new IllegalArgumentException("O parmetro parser est nulo.");
      }
      this.parser = parser;
    }

    /**
     * Obtm o modelo do componente visual da lista.
     * 
     * @return o modelo.
     */
    private DefaultListModel getModel() {
      return this.model;
    }

    /**
     * Remove os itens selecionados.
     */
    private void removeSelection() {
      int[] indices = this.list.getSelectedIndices();
      for (int i = 0; i < indices.length; i++) {
        getParameter().removeElement(indices[i] - i);
      }
    }

    /**
     * Restaura o campo de texto.
     */
    private void restoreText() {
      this.textField.setText("");
      this.textField.requestFocus();
    }

    /**
     * Atualiza o estado do boto de adicionar novos elementos.
     */
    private void updateAddButton() {
      String rawText = this.textField.getText();
      Object object = getParser().parse(rawText);
      this.addButton.setEnabled(isEnabled() && object != null);
    }

    /**
     * Atualiza o estado do boto de editar elementos.
     */
    private void updateEditButton() {
      this.editButton.setEnabled(isEnabled() && !this.list.isSelectionEmpty());
    }

    /**
     * Atualiza o estado do boto de remover elementos.
     */
    private void updateRemoveButton() {
      this.removeButton
        .setEnabled(isEnabled() && !this.list.isSelectionEmpty());
    }

    /**
     * Ao de adicionar novos elementos na lista.
     */
    private final class AddActionListener implements ActionListener {

      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        if (addButton.isEnabled()) {
          String text = textField.getText().trim();
          E element = getParser().parse(text);
          if (element != null) {
            getModel().addElement(getFormatter().format(element));
            ListParameter<E> parameter = getParameter();
            parameter.addElement(element);
            restoreText();
          }
        }
      }
    }

    /**
     * Observador de mudanas da caixa de texto utilizada para insero na
     * lista.
     */
    private final class AddDocumentListener implements DocumentListener {

      /**
       * {@inheritDoc}
       */
      @Override
      public void changedUpdate(DocumentEvent e) {
        updateAddButton();
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void insertUpdate(DocumentEvent e) {
        updateAddButton();
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void removeUpdate(DocumentEvent e) {
        updateAddButton();
      }
    }

    /**
     * Ao de editar um elemento j existente na lista.
     */
    private final class EditActionListener implements ActionListener {

      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        String text =
          JOptionPane.showInputDialog("Editar", list.getSelectedValue());
        if (text != null) {
          E element = getParser().parse(text);
          if (element != null) {
            int index = list.getSelectedIndex();
            getModel().removeElementAt(index);
            getModel().add(index, getFormatter().format(element));
            getParameter().setElement(index, element);
          }
        }
        restoreText();
      }
    }

    /**
     * Observador de mudanas na seleo de elementos da lista.
     */
    private final class EditListSelectionListener implements
      ListSelectionListener {

      /**
       * {@inheritDoc}
       */
      @Override
      public void valueChanged(ListSelectionEvent e) {
        if (list.isSelectionEmpty() || list.getSelectedIndices().length > 1) {
          editButton.setEnabled(false);
        }
        else {
          editButton.setEnabled(true);
        }
        if (list.isSelectionEmpty()) {
          removeButton.setEnabled(false);
        }
        else {
          removeButton.setEnabled(true);
        }
      }
    }

    /**
     * Ao de remover um item da lista.
     */
    private final class RemoveActionListener implements ActionListener {

      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        removeSelection();
        restoreText();
      }
    }

    /**
     * Observador de atalhos de teclado para remoo de itens da lista.
     */
    private final class RemoveKeyListener implements KeyListener {

      /**
       * {@inheritDoc}
       */
      @Override
      public void keyPressed(KeyEvent e) {
        int keyCode = e.getKeyCode();
        switch (keyCode) {
          case KeyEvent.VK_BACK_SPACE:
          case KeyEvent.VK_DELETE:
            removeSelection();
        }
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void keyReleased(KeyEvent e) {
        // Ignora o evento
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void keyTyped(KeyEvent e) {
        // Ignora o evento
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return true;
  }
}
