package csbase.client.applicationmanager;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.rmi.RemoteException;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.exception.MaxClientsReachedException;
import tecgraf.ftc.common.exception.PermissionException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.desktop.DesktopPref;
import csbase.client.desktop.RemoteTask;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.types.PVBoolean;
import csbase.client.preferences.types.PVString;
import csbase.logic.FileInfo;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.Version;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.remote.ApplicationServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * Controla um diretrio de cache de arquivos das aplicaes CSDK no cliente. O
 * cache pode ser desabilitado atravs da preferncia
 * {@link DesktopPref#CSDK_CACHE_ENABLED}. O diretrio local utilizado como base
 * pelo cache pode ser configurado pela preferncia
 * {@link DesktopPref#CSDK_CACHE_LOCATION}.
 */
public class ApplicationCache {

  /**
   * Indica se a verso da aplicao j foi carregada no cache.
   * 
   * @param registry o registro da aplicao.
   * @return <code>true</code> se a aplicao j est em cache ou
   *         <code>false</code> caso contrrio.
   */
  public static boolean isApplicationVersionInCache(ApplicationRegistry registry) {
    File cacheDir = getCachePathForApplicationVersion(registry);
    if (!cacheDir.exists()) {
      return false;
    }
    FileInfo[] libInfos = registry.getApplicationLibs();
    for (FileInfo libInfo : libInfos) {
      String fileName = libInfo.getName();
      File localFile = new File(cacheDir, fileName);
      if (!localFile.exists()) {
        return false;
      }
    }
    return true;
  }

  /**
   * Limpa o diretrio local de cache das bibliotecas da aplicao especificada.
   * 
   * @param registry o registro da aplicao.
   * 
   * @throws ApplicationException em caso de erro ao limpar o cache.
   */
  public static void clearCacheForApplication(ApplicationRegistry registry)
    throws ApplicationException {
    File cacheDir = getCacheBasePathForApplication(registry);
    if (cacheDir.exists()) {
      try {
        if (!FileUtils.delete(cacheDir)) {
          String errorMsg =
            LNG.get("application.manager.error.clear.app.cache");
          throw new ApplicationException(errorMsg);
        }
      }
      catch (Exception e) {
        String errorMsg =
          LNG.get("application.manager.error.clear.app.cache",
            new Object[] { registry.getId() });
        throw new ApplicationException(errorMsg);
      }
    }
  }

  /**
   * Limpa o diretrio local de cache das bibliotecas de todas as aplicaes.
   * 
   * @throws ApplicationException em caso de erro ao limpar o cache.
   */
  public static void clearCache() throws ApplicationException {
    File cacheDir = getCacheBasePath();
    if (cacheDir.exists()) {
      try {
        if (!FileUtils.delete(cacheDir)) {
          String errorMsg = LNG.get("application.manager.error.clear.cache");
          throw new ApplicationException(errorMsg);
        }
      }
      catch (Exception e) {
        String errorMsg = LNG.get("application.manager.error.clear.cache");
        throw new ApplicationException(errorMsg);
      }
    }
  }

  /**
   * Obtm o diretrio de cache da aplicao, j preenchido com os arquivos
   * necessrios para a aplicao.
   * 
   * @param registry o registro da aplicao.
   * @return o diretrio de cache.
   * @throws ApplicationUpdatedException caso a aplicao tenha sido atualizada
   *         no servidor e no esteja no cache.
   * @throws ApplicationException caso haja erro na preparao do cache.
   */
  public static File getApplicationCache(ApplicationRegistry registry)
    throws ApplicationUpdatedException, ApplicationException {
    FileInfo[] libs = registry.getApplicationLibs();
    if (libs == null) {
      return null;
    }
    File cacheDir = getCachePathForApplicationVersion(registry);
    if (!isCacheEnabled() || !isApplicationVersionInCache(registry)) {
      boolean updated = wasApplicationUpdated(registry);
      if (updated) {
        throw new ApplicationUpdatedException("Application updated in server");
      }
      fillApplicationCache(registry, cacheDir, libs);
    }
    return cacheDir;
  }

  /**
   * Indica se o cache de aplicaes est habilitado.
   * 
   * @return <code>true</code> caso o cache esteja habilitado ou
   *         <code>false</code> caso contrrio.
   */
  private static boolean isCacheEnabled() {
    PreferenceManager manager = PreferenceManager.getInstance();
    PreferenceCategory allPrefs = manager.loadPreferences();
    PreferenceCategory category = allPrefs.getCategory(DesktopPref.class);
    PVBoolean enabledProperty =
      category.getPVBoolean(DesktopPref.CSDK_CACHE_ENABLED);
    Boolean cacheEnabled = enabledProperty.getValue();
    return cacheEnabled;
  }

  /**
   * Obtm o diretrio de cache de uma verso especfica da aplicao. A verso
   * nesse caso  definida pelo timestamp do registro e no pela verso
   * configurada nas propriedades da aplicao.
   * 
   * @param registry o registro da aplicao.
   * 
   * @return o diretrio de cache da verso.
   */
  private static File getCachePathForApplicationVersion(
    ApplicationRegistry registry) {
    File applicationDir = getCacheBasePathForApplication(registry);
    return new File(applicationDir, String.valueOf(registry.getTimestamp()));
  }

  /**
   * Obtm o diretrio base de cache onde so armazenados os arquivos de uma
   * aplicao especfica localmente.
   * 
   * @param registry o registro da aplicao.
   * 
   * @return o diretrio base.
   */
  private static File getCacheBasePathForApplication(
    ApplicationRegistry registry) {
    File baseDir = getCacheBasePath();
    return new File(baseDir, registry.getId());
  }

  /**
   * Obtm o diretrio base de cache onde so armazenados os arquivos das
   * aplicaes localmente.
   * 
   * @return o diretrio base.
   */
  private static File getCacheBasePath() {
    PreferenceManager manager = PreferenceManager.getInstance();
    PreferenceCategory allPrefs = manager.loadPreferences();
    PreferenceCategory category = allPrefs.getCategory(DesktopPref.class);
    PVString cacheLocationPref =
      category.getPVString(DesktopPref.CSDK_CACHE_LOCATION);
    String cacheLocationValue = cacheLocationPref.getValue();
    CSDKCacheLocation cacheLocation;
    try {
      cacheLocation = CSDKCacheLocation.valueOf(cacheLocationValue);
    }
    catch (Exception e) {
      String defaultValue = cacheLocationPref.getDefaultValue();
      try {
        cacheLocation = CSDKCacheLocation.valueOf(defaultValue);
      }
      catch (Exception e1) {
        cacheLocation = CSDKCacheLocation.JAVA_IO_TMPDIR;
      }
    }
    File baseDir = cacheLocation.getPath();
    if (baseDir == null || !baseDir.exists()) {
      baseDir = CSDKCacheLocation.JAVA_IO_TMPDIR.getPath();
    }

    Client client = Client.getInstance();
    String systemName = client.getSystemName();
    String systemDirName = "." + FileUtils.fixDirectoryName(systemName);
    File systemDir = new File(baseDir, systemDirName);
    Version v = Version.getInstance();
    String version = v.getVersion();
    String versionDirName = FileUtils.fixDirectoryName(version);
    File versionDir = new File(systemDir, versionDirName);
    User user = User.getLoggedUser();
    String login = user.getLogin();
    String userDirName = FileUtils.fixDirectoryName(login);
    File userDir = new File(versionDir, userDirName);
    return userDir;
  }

  /**
   * Preenche o cache da aplicao, baixando do servidor os arquivos utilizadas
   * pela aplicao especificada.
   * 
   * @param registry o registro da aplicao.
   * @param cacheDir o diretrio de cache.
   * @param libInfos dados das biliotecas.
   * @throws ApplicationException em caso de erro ao salvar as bibliotecas.
   */
  private static void fillApplicationCache(ApplicationRegistry registry,
    final File cacheDir, final FileInfo[] libInfos) throws ApplicationException {
    final ApplicationServiceInterface applicationService =
      ClientRemoteLocator.applicationService;
    if (applicationService == null) {
      return;
    }
    try {
      clearCacheForApplication(registry);
    }
    catch (Exception e) {
      /*
       * Se o diretrio de cache no tiver sido apagado com sucesso, pode ser
       * que tenham ficado arquivos .nfs pendentes. Podemos seguir e tentar
       * repopular a cache no diretrio existente.
       */
    }
    if (!cacheDir.exists()) {
      if (!cacheDir.mkdirs()) {
        String errorMsg =
          LNG.get("application.manager.error.lib.dir", new Object[] { cacheDir
            .getPath() });
        throw new ApplicationException(errorMsg);
      }
    }
    final String appId = registry.getId();
    final RemoteTask<Void> loadLibTask = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws RemoteException {
        for (FileInfo libInfo : libInfos) {
          String[] resoursePath = libInfo.getPathAsArray();
          RemoteFileChannelInfo fileChannel =
            applicationService.getApplicationResource(appId, resoursePath);
          if (fileChannel == null) {
            throw new RemoteException(
              "Application resource not found in server: " + libInfo.getPath());
          }
          try {
            String fileName = libInfo.getName();
            File localFile = new File(cacheDir, fileName);
            downloadFile(fileChannel, localFile);
          }
          catch (final Exception e) {
            throw new RemoteException("Error downloading application resource "
              + libInfo.getPath(), e);
          }
        }
      }

      private void downloadFile(RemoteFileChannelInfo info, File targetFile)
        throws IOException, PermissionException, FailureException,
        MaxClientsReachedException {
        @SuppressWarnings("resource")
        OutputStream target = null;
        SyncRemoteFileChannel channel = null;
        try {
          channel = new SyncRemoteFileChannel(info);
          channel.open(true);
          long size = channel.getSize();
          target = new FileOutputStream(targetFile);
          channel.syncTransferTo(0, size, target);
          target.flush();
        }
        finally {
          FileUtils.close(target);
          if (channel != null) {
            channel.close();
          }
        }
      }
    };
    String message =
      LNG.get("application.manager.task.lib.download", new Object[] { appId });
    if (!loadLibTask.execute(null, null, message)) {
      String errorMsg =
        LNG.get("application.manager.error.lib.download",
          new Object[] { appId });
      throw new ApplicationException(errorMsg, loadLibTask.getError());
    }
  }

  /**
   * Indica se a aplicao foi atualizada no servidor desde que o registro local
   * foi obtido.
   * 
   * @param registry o registro local da aplicao.
   * @return <code>true</code> se a aplicao foi atualiada ou
   *         <code>false</code> caso contrrio.
   */
  private static boolean wasApplicationUpdated(ApplicationRegistry registry) {
    final ApplicationServiceInterface appService =
      ClientRemoteLocator.applicationService;
    if (appService == null) {
      return false;
    }
    ApplicationRegistry serverRegistry =
      appService.getApplicationRegistry(registry.getId());
    if (serverRegistry == null) {
      return true;
    }
    Long serverTimestamp = serverRegistry.getTimestamp();
    if (serverTimestamp != null) {
      long localTimestamp = registry.getTimestamp();
      return serverTimestamp != localTimestamp;
    }
    else {
      return false;
    }
  }

}
