/*
 * $Id: ApplicationProject.java 153877 2014-06-30 17:35:42Z karla $
 */

package csbase.client.applications;

import java.awt.Window;
import java.rmi.RemoteException;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.kernel.ClientException;
import csbase.client.project.ProjectFileChooser;
import csbase.client.project.ProjectFileChooserOpen;
import csbase.client.project.ProjectFileChooserSave;
import csbase.client.project.ProjectTreePath;
import csbase.client.project.tasks.CreateFileTask;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.project.tasks.GetChildFromPathTask;
import csbase.client.project.tasks.SetFileTypeTask;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil.OperationResult;
import csbase.exception.CSBaseRuntimeException;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Classe que modela uma aplicao CSBASE que necessita de projeto aberto; ou
 * seja, manipula arquivos de um projeto corrente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class ApplicationProject extends Application {

  /**
   * Construtor padro.
   * 
   * @param id o identificador da aplicao.
   */
  protected ApplicationProject(final String id) {
    super(id);
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param project projeto sobre o qual deseja-se abrir um arquivo.
   * @param fileTypes Array com os tipos de arquivos permitidos
   * @param defaultType Tipo default que vai aparecer selecionado na lista
   * @param frame janela-me do navegador.
   * 
   * @return Arquivo selecionado
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileOpen(String[],String,Window) mtodo de convenincia para
   *      uso do projeto corrente da aplicao.
   */
  public final ClientProjectFile browseFileOpenFromProject(
    final CommonClientProject project, final String[] fileTypes,
    final String defaultType, final Window frame) throws ApplicationException {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseFileOpenFromProject(project, fileTypes, defaultType,
        frame);
    }
    return oldBrowseFileOpenFromProject(project, fileTypes, defaultType, frame);
  }

  /**
   * Indica uso de novo chooser.
   * 
   * @return indicativo
   */
  private boolean useNewChooser() {
    try {
      final ConfigurationManager cnfManager =
        ConfigurationManager.getInstance();
      final Configuration cnf =
        cnfManager.getConfiguration(ApplicationProject.class);
      final String propName = "new.chooser";
      final Boolean newChooser =
        cnf.getOptionalBooleanProperty(propName, false);
      return newChooser;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param project projeto sobre o qual deseja-se abrir um arquivo.
   * @param fileTypes Array com os tipos de arquivos permitidos
   * @param defaultType Tipo default que vai aparecer selecionado na lista
   * @param window janela-me do navegador.
   * 
   * @return arquivo selecionado
   * 
   * @see #browseFileOpen(String[],String,Window) mtodo de convenincia para
   *      uso do projeto corrente da aplicao.
   */
  private ClientProjectFile newBrowseFileOpenFromProject(
    final CommonClientProject project, final String[] fileTypes,
    final String defaultType, final Window window) {
    final List<String> types = new ArrayList<String>();
    for (String fileType : fileTypes) {
      types.add(fileType);
    }
    final OperationResult result =
      ClientProjectFileChooserUtil.browseSingleFileInOpenMode(window, types,
        "", true);
    if (result == null) {
      return null;
    }

    final ClientProjectFile clientProjectFile = result.getClientProjectFile();
    return clientProjectFile;
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param project projeto sobre o qual deseja-se abrir um arquivo.
   * @param fileTypes Array com os tipos de arquivos permitidos
   * @param defaultType Tipo default que vai aparecer selecionado na lista
   * @param frame janela-me do navegador.
   * 
   * @return Arquivo selecionado
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileOpen(String[],String,Window) mtodo de convenincia para
   *      uso do projeto corrente da aplicao.
   */
  private ClientProjectFile oldBrowseFileOpenFromProject(
    final CommonClientProject project, final String[] fileTypes,
    final String defaultType, final Window frame) throws ApplicationException {
    ProjectFileChooserOpen selection;
    try {
      selection =
        new ProjectFileChooserOpen(frame, project, false, fileTypes,
          defaultType);
    }
    catch (ClientException e) {
      throw new ApplicationException(e);
    }
    final ProjectTreePath treePath = selection.getSelectedPath();
    if (treePath == null) {
      return null;
    }
    return treePath.getFile();
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param fileTypes Array com os tipos de arquivos permitidos
   * @param defaultType Tipo default que vai aparecer selecionado na lista
   * @param frame janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileOpen(String[],String) mtodo de convenincia para uso do
   *      dilogo principal da aplicao como janela-me do navegador.
   */
  public final ClientProjectFile browseFileOpen(final String[] fileTypes,
    final String defaultType, final Window frame) throws ApplicationException {
    final CommonClientProject project = getApplicationProject();
    return browseFileOpenFromProject(project, fileTypes, defaultType, frame);
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param fileTypes Array com os tipos de arquivos permitidos
   * @param defaultType Tipo default que vai aparecer selecionado na lista
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileOpen(String) mtodo de convenincia para uso de um nico
   *      tipo de arquivo na navegao.
   */
  public final ClientProjectFile browseFileOpen(final String[] fileTypes,
    final String defaultType) throws ApplicationException {
    return browseFileOpen(fileTypes, defaultType, getApplicationFrame());
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param fileType tipos de arquivo permitido
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileOpen(String,Window) mtodo de convenincia para seleo de
   *      uma janela-me.
   */
  public final ClientProjectFile browseFileOpen(final String fileType)
    throws ApplicationException {
    return browseFileOpen(new String[] { fileType }, fileType);
  }

  /**
   * Navegador para abertura de mltiplos tipos de arquivos.
   * 
   * @param fileType tipos de arquivos permitidos.
   * @param frame .
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileOpen(String) mtodo de convenincia para uso do dilogo
   *      principal da aplicao.
   */
  public final ClientProjectFile browseFileOpen(final String fileType,
    final Window frame) throws ApplicationException {
    return browseFileOpen(new String[] { fileType }, fileType, frame);
  }

  /**
   * Navegador para gravao de arquivos (modo <i>save</i>), onde o nome do
   * arquivo pode ser sugerido inicialmente. Diretrio corrente ser obtido do
   * nome do arquivo ou (caso este seja null) ser a raiz do projeto.
   * 
   * @param project o projeto de busca do arquivo.
   * @param fileType o tipo de arquivos permitidos
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param window janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String,Window) mtodo de convenincia para usar o
   *      projeto corrente da aplicao.
   */
  public final ClientProjectFile browseFileSaveFromProject(
    final CommonClientProject project, final String fileType,
    final String fileExtension, String defaultName, final Window window)
    throws ApplicationException {
    return this.browseFileSaveFromProject(project, fileType, fileExtension,
      defaultName, null, window);
  }

  /**
   * Navegador para gravao de arquivos (modo <i>save</i>), onde o nome do
   * arquivo e diretrio corrente podem ser sugeridos inicialmente.
   * 
   * @param project o projeto de busca do arquivo.
   * @param fileType o tipo de arquivos permitidos
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param currentDirectory diretrio corrente para abertura do chooser, pode
   *        ser null, faz sentido usar apenas se no houver defaultName
   * @param window janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String,Window) mtodo de convenincia para usar o
   *      projeto corrente da aplicao.
   */
  public final ClientProjectFile browseFileSaveFromProject(
    final CommonClientProject project, final String fileType,
    final String fileExtension, String defaultName, String[] currentDirectory,
    final Window window) throws ApplicationException {
    if (useNewChooser()) {
      return newBrowseFileSaveFromProject(project, fileType, fileExtension,
        defaultName, currentDirectory, window);
    }
    return oldBrowseFileSaveFromProject(project, fileType, fileExtension,
      defaultName, currentDirectory, window);
  }

  /**
   * Navegador para gravao de arquivos (modo <i>save</i>), onde o nome do
   * arquivo e diretrio corrente podem ser sugeridos inicialmente.
   * 
   * @param project o projeto de busca do arquivo.
   * @param fileType o tipo de arquivos permitidos
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param currentDirectory diretrio corrente para abertura do chooser, pode
   *        ser null, faz sentido usar apenas se no houver defaultName. FIXME
   *        este atributo ainda no est sendo tratado por
   *        ClientProjectFileChooserUtil
   * @param window janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @see #browseFileSave(String,Window) mtodo de convenincia para usar o
   *      projeto corrente da aplicao.
   */
  private ClientProjectFile newBrowseFileSaveFromProject(
    final CommonClientProject project, final String fileType,
    final String fileExtension, String defaultName, String[] currentDirectory,
    final Window window) {
    final List<String> types = new ArrayList<String>();
    types.add(fileType);
    final OperationResult result =
      ClientProjectFileChooserUtil.browseSingleFileInSaveMode(window, types,
        fileType, "", true, true);
    if (result == null) {
      return null;
    }

    final ClientProjectFile clientProjectFile = result.getClientProjectFile();
    return clientProjectFile;
  }

  /**
   * Navegador para gravao de arquivos (modo <i>save</i>), onde o nome do
   * arquivo e diretrio corrente podem ser sugeridos inicialmente.
   * 
   * @param project o projeto de busca do arquivo.
   * @param fileType o tipo de arquivos permitidos
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param currentDirectory diretrio corrente para abertura do chooser, pode
   *        ser null, faz sentido usar apenas se no houver defaultName
   * @param frame janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String,Window) mtodo de convenincia para usar o
   *      projeto corrente da aplicao.
   */
  private ClientProjectFile oldBrowseFileSaveFromProject(
    final CommonClientProject project, final String fileType,
    final String fileExtension, String defaultName, String[] currentDirectory,
    final Window frame) throws ApplicationException {
    final ProjectFileChooserSave selection;
    try {
      selection =
        new ProjectFileChooserSave(frame, project,
          ProjectFileChooser.FILE_ONLY, fileType, defaultName,
          currentDirectory, fileExtension, false);
    }
    catch (ClientException e) {
      throw new ApplicationException(e);
    }
    return getFile(selection, project, fileExtension, frame);
  }

  /**
   * Retorna o arquivo selecionado com o tipo selecionado. Se no existir, cria.
   * 
   * @param saveSelection a seleo
   * @param project o projeto
   * @param fileExtension a extenso do arquivo
   * @param frame a janela me
   * @return retorna o resultado da tarefa de criao do arquivo
   * @throws ApplicationException em caso de falha.
   */
  private ClientProjectFile getFile(ProjectFileChooserSave saveSelection,
    final CommonClientProject project, final String fileExtension,
    final Window frame) throws ApplicationException {

    final ProjectTreePath treePath = saveSelection.getSelectedPath();
    if (treePath == null) {
      return null;
    }

    final String[] path = treePath.getPath();
    String name = path[path.length - 1];

    if (fileExtension != null) {
      // desconsideramos os "." no incio da extenso, caso existam
      final String fileExt = fileExtension.replaceAll("^\\.*", "");
      String[] nameParts = name.split("\\.");
      if (nameParts.length > 1) {
        String ext = nameParts[nameParts.length - 1];
        if (!ext.toUpperCase().equals(fileExt.toUpperCase())) {
          name = name + "." + fileExt;
        }
      }
      else {
        name = name + "." + fileExt;
      }
    }

    ClientProjectFile parentDir = treePath.getParent();
    final ClientProjectFile child =
      GetChildFromNameTask.runTask(parentDir, name);
    if (child != null) {
      /*
       * Se o arquivo j existir e o tipo for diferente do esperado, altera o
       * tipo.
       */
      final String type = saveSelection.getFileType();
      if (!child.getType().equalsIgnoreCase(type)) {
        final SetFileTypeTask fileTypeSetter = new SetFileTypeTask(child, type);
        fileTypeSetter.execute(frame, getName(), String
          .format(LNG
            .get("applications.ApplicationProject.info.updating_file_type"),
            name));
      }
      return child;
    }

    final CreateFileTask task =
      new CreateFileTask(project, treePath.getParent(), name, saveSelection
        .getFileType());

    final boolean succeeded =
      task.execute(frame, getName(), String.format(LNG
        .get("applications.ApplicationProject.info.creating_file"), name));

    /*
     * Lana ApplicationException apenas para erros que no foram tratados na
     * RemoteTask
     */
    Exception exception = task.getError();
    if (!succeeded && (!(exception instanceof CSBaseRuntimeException))
      && (!(exception instanceof RemoteException))) {
      throw new ApplicationException(exception);
    }
    return task.getResult();
  }

  /**
   * Navegador para gravao de arquivos.
   * 
   * @param fileType tipos de arquivos permitidos.
   * @param frame janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String) mtodo de convenincia para usar, como
   *      janela-me, o dilogo principal da aplicao.
   */
  public final ClientProjectFile browseFileSave(final String fileType,
    final Window frame) throws ApplicationException {
    final CommonClientProject project = getApplicationProject();
    return browseFileSaveFromProject(project, fileType, null, null, frame);
  }

  /**
   * Navegador para gravao de arquivos.
   * 
   * @param fileType tipos de arquivos permitidos.
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param frame janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String) mtodo de convenincia para usar, como
   *      janela-me, o dilogo principal da aplicao.
   */
  public final ClientProjectFile browseFileSave(final String fileType,
    final String fileExtension, final Window frame) throws ApplicationException {
    final CommonClientProject project = getApplicationProject();
    return browseFileSaveFromProject(project, fileType, fileExtension, null,
      null, frame);
  }

  /**
   * Navegador para gravao de arquivos, onde o nome do arquivo pode ser
   * sugerido inicialmente. Diretrio corrente ser obtido do nome do arquivo ou
   * (caso este seja null) ser a raiz do projeto.
   * 
   * @param fileType tipos de arquivos permitidos.
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param frame janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String) mtodo de convenincia para usar, como
   *      janela-me, o dilogo principal da aplicao.
   */
  public final ClientProjectFile browseFileSave(final String fileType,
    final String fileExtension, String defaultName, final Window frame)
    throws ApplicationException {
    return browseFileSave(fileType, fileExtension, defaultName, null, frame);
  }

  /**
   * Navegador para gravao de arquivos, onde o nome do arquivo e diretrio
   * corrente podem ser sugeridos inicialmente.
   * 
   * @param fileType tipos de arquivos permitidos.
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param currentDirectory diretrio corrente para abertura do chooser, pode
   *        ser null, faz sentido usar apenas se no houver defaultName
   * @param frame janela-me do navegador.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String) mtodo de convenincia para usar, como
   *      janela-me, o dilogo principal da aplicao.
   */
  public final ClientProjectFile browseFileSave(final String fileType,
    final String fileExtension, String defaultName, String[] currentDirectory,
    final Window frame) throws ApplicationException {
    final CommonClientProject project = getApplicationProject();
    return browseFileSaveFromProject(project, fileType, fileExtension,
      defaultName, currentDirectory, frame);
  }

  /**
   * Navegador para gravao de arquivos.
   * 
   * @param fileType tipo de arquivos permitidos.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String,Window) mtodo de convenincia para ajustar a
   *      janela-me do navegador.
   */
  public final ClientProjectFile browseFileSave(final String fileType)
    throws ApplicationException {
    return browseFileSave(fileType, getApplicationFrame());
  }

  /**
   * Navegador para gravao de arquivos.
   * 
   * @param fileType tipo de arquivos permitidos.
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String,Window) mtodo de convenincia para ajustar a
   *      janela-me do navegador.
   */
  public final ClientProjectFile browseFileSave(final String fileType,
    final String fileExtension) throws ApplicationException {
    return browseFileSave(fileType, fileExtension, getApplicationFrame());
  }

  /**
   * Navegador para gravao de arquivos, em que pode-se especificar um valor
   * default para o nome do arquivo a ser salvo.
   * 
   * @param fileType tipos de arquivos permitidos.
   * @param fileExtension extenso a ser inserida no nome do arquivo,
   *        independente do seu tipo.
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * 
   * @return arquivo selecionado ou <code>null</code>.
   * 
   * @throws ApplicationException em caso de falha.
   * 
   * @see #browseFileSave(String) mtodo de convenincia para usar, como
   *      janela-me, o dilogo principal da aplicao.
   */
  public final ClientProjectFile browseFileSave(final String fileType,
    final String fileExtension, String defaultName) throws ApplicationException {
    return browseFileSave(fileType, fileExtension, defaultName,
      getApplicationFrame());
  }

  /**
   * Mtodo utilitrio para converso um arquivo no projeto corrente em uma
   * string que represente seu caminho lgico no projeto.
   * 
   * @param file o arquivo.
   * 
   * @return o <i>path</i>.
   */
  public final String fileToString(final ClientFile file) {
    if (file == null) {
      return "";
    }
    return file.getStringPath();
  }

  /**
   * Mtodo utilitrio para converso de um string que representa um <i>path</i>
   * em um arquivo no projeto corrente; caso no exista o arquivo, retorna-se
   * <code>null</code>. O nome do projeto no deve estar no <i>path</i>.
   * 
   * @param path o texto com o <i>path</i>.
   * 
   * @return o objeto que representa o arquivo ou <code>null</code>.
   * 
   * @see ClientProjectFile
   */
  public final ClientProjectFile stringToFile(final String path) {
    if (path == null) {
      throw new InvalidParameterException("Erro interno.\n"
        + "Parmetro invlido em ApplicationProject.stringToFile");
    }
    final String[] splitPath = ClientProjectFile.splitPath(path.trim());
    return stringArrayToFile(splitPath);
  }

  /**
   * Mtodo utilitrio para converso de um string que representa um <i>path</i>
   * em um arquivo no projeto corrente; caso no exista o arquivo, retorna-se
   * <code>null</code>. O nome do projeto no deve estar no <i>path</i>.
   * 
   * @param path o texto com o <i>path</i>.
   * @param type o tipo (caso o arquivo seja criado)
   * 
   * @return o objeto que representa o arquivo ou <code>null</code>.
   * 
   * @see ClientProjectFile
   */
  public final ClientProjectFile stringToFileWithCreation(final String path,
    final String type) {
    if (path == null) {
      throw new InvalidParameterException("Erro interno.\n"
        + "Parmetro invlido em ApplicationProject.stringToFileWithCreation");
    }
    final String[] splitPath = ClientProjectFile.splitPath(path.trim());
    return stringArrayToFileWithCreation(splitPath, type);
  }

  /**
   * Mtodo utilitrio para converso de um array de string que representa um
   * <i>path</i> em um arquivo no projeto corrente; caso no exista o arquivo,
   * retorna-se <code>null</code>. O nome do projeto no deve estar no
   * <i>path</i>.
   * 
   * @param path o array de String com o <i>path</i>.
   * 
   * @return o objeto que representa o arquivo ou <code>null</code>.
   * 
   * @see ClientProjectFile
   */
  public final ClientProjectFile stringArrayToFile(final String[] path) {
    if (path == null || (path.length == 0)) {
      throw new InvalidParameterException("Erro interno.\n"
        + "Parmetro invlido em ApplicationProject.stringArrayToFile");
    }
    CommonClientProject project = getApplicationProject();
    // Obtm a raiz do projeto
    ClientProjectFile root = project.getRoot();

    // Insere o nome do projeto no path
    String[] rootPath = root.getPath();
    String[] fullPath = new String[rootPath.length + path.length];
    System.arraycopy(rootPath, 0, fullPath, 0, rootPath.length);
    System.arraycopy(path, 0, fullPath, rootPath.length, path.length);

    final ClientProjectFile file = GetChildFromPathTask.runTask(root, fullPath);
    return file;
  }

  /**
   * Mtodo utilitrio para converso de um array de string que representa um
   * <i>path</i> em um arquivo no projeto corrente; caso no exista o arquivo,
   * retorna-se <code>null</code>. O nome do projeto no deve estar no
   * <i>path</i>.
   * 
   * @param path o array de String com o <i>path</i>.
   * @param type o tipo a ser criado.
   * 
   * @return o objeto que representa o arquivo ou <code>null</code>.
   * 
   * @see ClientProjectFile
   */
  public final ClientProjectFile stringArrayToFileWithCreation(
    final String[] path, final String type) {
    final ClientProjectFile file = stringArrayToFile(path);
    if (file != null) {
      return file;
    }

    final CommonClientProject project = getApplicationProject();
    final ClientProjectFile root = project.getRoot();

    final String name = path[path.length - 1];
    final String[] rootPath = root.getPath();
    final String[] dirPath = new String[rootPath.length + path.length - 1];
    System.arraycopy(rootPath, 0, dirPath, 0, rootPath.length);
    System.arraycopy(path, 0, dirPath, rootPath.length, path.length - 1);

    final ClientProjectFile dir;
    if (dirPath.length == 0) {
      dir = root;
    }
    else {
      dir = GetChildFromPathTask.runTask(root, dirPath);
    }

    final String msg = "[" + name + "]";
    final ImageIcon icon = ApplicationImages.ICON_ANIMATEDENGINES_32;
    final ApplicationTask<ClientProjectFile> tsk =
      new ApplicationTask<ClientProjectFile>(this, msg) {
        @Override
        public void performTask() throws Exception {
          final ClientProjectFile newFile =
            project.createAndWaitForFile(dir, name, type);
          setResult(newFile);
        }
      };
    final boolean ok = tsk.execute(this.getApplicationFrame(), "", msg, icon);
    if (!ok) {
      return null;
    }
    return tsk.getResult();
  }

}
