package csbase.client.applications.flowapplication.actions;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNGKeys;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.flowapplication.FlowApplication;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.configurator.FlowAlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphElement;
import csbase.client.applications.flowapplication.graph.GraphFileDescriptor;
import csbase.client.applications.flowapplication.graph.GraphLayout;
import csbase.client.applications.flowapplication.graph.GraphLink;
import csbase.client.desktop.RemoteTask;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * Ao que executa o fluxo de algoritmos corrente.
 */
public abstract class GraphLayoutAction extends FlowApplicationAction {

  /**
   * Tipo de layout a ser aplicado.
   */
  private GraphLayout layout;

  /**
   * Cria a ao.
   * 
   * @param application A aplicao construtura de fluxos.
   * @param icon O cone da ao.
   * @param layout Tipo de layout a ser aplicado.
   */
  public GraphLayoutAction(FlowApplication application, ImageIcon icon,
    GraphLayout layout) {
    super(application, icon);
    this.layout = layout;
    setEnabled(false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleActionPerformed(ActionEvent e) {
    FlowApplication application = getApplication();
    ApplicationFrame applicationFrame = application.getApplicationFrame();
    if (usePreview()) {
      Component layoutPreview = createLayoutPreview();
      if (layoutPreview != null) {
        JPanel panel = new JPanel(new BorderLayout());
        String sufix = GraphLayoutAction.class.getSimpleName();
        String message = application.getString(sufix + ".preview.message");
        panel.add(new JLabel(message), BorderLayout.NORTH);
        panel.add(layoutPreview, BorderLayout.CENTER);
        final Object[] options =
          { LNG.get(LNGKeys.YES), LNG.get(LNGKeys.CANCEL) };
        int result =
          JOptionPane.showOptionDialog(applicationFrame, panel,
            application.getName(), JOptionPane.YES_NO_OPTION,
            JOptionPane.PLAIN_MESSAGE, null, options, options[0]);
        if (result != 0) {
          return;
        }
      }
    }
    doLayout(application.getWorkspace());
  }

  /**
   * Aplica o layout no fluxo corrente.
   * 
   * @param workspace workspace com o fluxo.
   */
  protected void doLayout(Workspace workspace) {
    Graph graph = workspace.getGraph();
    layout.doLayout(graph);
    workspace.repaint();
    graph.notifyChangedWorkspace();
  }

  /**
   * Indica se antes da ao deve ser mostrada uma pr-visualizao do layout.
   * 
   * @return verdadeiro se a pr-visualizao deve ser feita ou falso, caso
   *         contrrio.
   */
  protected abstract boolean usePreview();

  /**
   * Cria a pr-visualizao do fluxo com o layout aplicado.
   * 
   * @return o componente de pr-visualizao.
   */
  private Component createLayoutPreview() {
    final FlowApplication application = getApplication();
    final ApplicationFrame applicationFrame = application.getApplicationFrame();
    RemoteTask<FlowAlgorithmConfiguratorView> task =
      new RemoteTask<FlowAlgorithmConfiguratorView>() {
        @Override
        protected void performTask() throws Exception {
          Workspace workspace = application.getWorkspace();
          Graph graph = workspace.getGraph();
          FlowAlgorithmConfiguratorView configuratorView =
            new FlowAlgorithmConfiguratorView(applicationFrame, graph,
              Mode.REPORT, ValidationMode.FULL);
          setResult(configuratorView);
        }
      };
    String prefix = GraphLayoutAction.class.getSimpleName();
    final String msg = application.getString(prefix + ".task.message");
    if (task.execute(applicationFrame, application.getName(), msg)) {
      FlowAlgorithmConfiguratorView configurator = task.getResult();
      Workspace workspace = configurator.getWorkspace();
      doLayout(workspace);
      JScrollPane pane = new JScrollPane(workspace);
      final double factor = 0.5;
      Dimension screenSize = GUIUtils.getScreenDimension();
      Dimension currentSize = workspace.getPreferredSize();
      double usuableWidth = screenSize.getWidth() * factor;
      double newWidth = Math.min(currentSize.getWidth(), usuableWidth);
      double usuableHeight = screenSize.getHeight() * factor;
      double newHeight = Math.min(currentSize.getHeight(), usuableHeight);
      Dimension newSize = new Dimension();
      newSize.setSize(newWidth, newHeight);
      pane.setPreferredSize(newSize);
      return pane;
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasChangedWorkspace(Graph graph) {
    updateEnabledStatus(graph);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasElementCreated(Graph graph, GraphElement element) {
    updateEnabledStatus(graph);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasElementRemoved(Graph graph, GraphElement element) {
    updateEnabledStatus(graph);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasLinkAnchored(Graph graph, GraphLink link,
    GraphFileDescriptor fileDescriptor) {
    updateEnabledStatus(graph);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasLinkUnanchored(Graph graph, GraphLink link,
    GraphFileDescriptor fileDescriptor) {
    updateEnabledStatus(graph);
  }

  /**
   * Habilita/desabilita a ao, dependendo do estado do grafo.
   * 
   * @param graph O grafo.
   */
  private void updateEnabledStatus(Graph graph) {
    setEnabled(graph.hasElements());
  }

  /**
   * Retorna o layout da ao.
   * 
   * @return o layout.
   */
  protected GraphLayout getLayout() {
    return layout;
  }
}
