package csbase.client.applications.projectsmanager;

import java.util.ArrayList;
import java.util.List;
import java.util.Observer;

import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.ReloadSelectedProjectsTask;
import csbase.client.remote.ProjectAdminObserver;
import csbase.logic.CommonProjectInfo;

/**
 * Observador remoto abstrato, para ser usado pelo {@link ProjectsManager}.
 * 
 * @author Tecgraf
 */
public abstract class AbstractProjectsManagerObserver implements Observer {

  /**
   * Referncia para a aplicao de gerenciamento de projetos.
   */
  protected ProjectsManager projectsManager;

  /**
   * Construtor.
   * 
   * @param projectsManager aplicao de gerenciamento de projetos
   */
  public AbstractProjectsManagerObserver(final ProjectsManager projectsManager) {
    this.projectsManager = projectsManager;
    ProjectAdminObserver.getInstance().addObserver(this);
  }

  /**
   * Preenche a estrutura ProjectsManagerData com o contedo de um projeto
   * recm-criado por fora do ProjectsManager.
   * 
   * @param projectId Identificador do projeto.
   * @param projectName Nome do projeto.
   * @param userId Id do dono do projeto.
   * @return Projeto criado na estrutura ProjectsManagerData.
   */
  private ProjectsManagerData gatherProjectInfo(Object projectId,
    String projectName, Object userId) {
    final ProjectsManagerData pmd =
      new ProjectsManagerData(projectId, projectName, userId);
    final List<ProjectsManagerData> pmds = new ArrayList<ProjectsManagerData>();
    pmds.add(pmd);
    final ReloadSelectedProjectsTask rspt =
      new ReloadSelectedProjectsTask(projectsManager, pmds);
    rspt.execute(projectsManager.getApplicationFrame(), ProjectsManagerUI
      .getString("AbstractProjectsManagerObserver.title"), ProjectsManagerUI
      .getString("AbstractProjectsManagerObserver.message"));
    if (rspt.wasCancelled()) {
      final String err =
        ProjectsManagerUI
          .getString("AbstractProjectsManagerObserver.cancelled.message");
      rspt.showError(err);
      return null;
    }
    if (rspt.getStatus() != true) {
      final Exception exception = rspt.getError();
      ProjectsManagerUI.handleException(projectsManager, exception);
      return null;
    }
    final List<ProjectsManagerData> projectData = rspt.getResult();
    if (projectData == null) {
      return null;
    }
    return projectData.get(0);
  }

  /**
   * Remove-se da lista de observadores de projetos.
   */
  public void removeItselfAsObserver() {
    ProjectAdminObserver.getInstance().deleteObserver(this);
  }

  /**
   * Adiciona um novo projeto  tabela, levando em considerao os filtros em
   * questo (i.e. o projeto no ser exibido se no passar pelos filtros).
   * 
   * @param prjInfo informaes do projeto ({@link CommonProjectInfo})
   */
  protected void addProject(final CommonProjectInfo prjInfo) {
    final ProjectsManagerData pmd =
      gatherProjectInfo(prjInfo.projectId, prjInfo.name, prjInfo.userId);
    projectsManager.addProject(pmd);
  }

}