package csbase.client.applications.projectsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.table.DefaultTableModel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.ChangeServerOwnerProjectTask;
import csbase.client.applications.projectsmanager.proxy.ReloadSelectedProjectsTask;
import csbase.client.desktop.RemoteTask;
import csbase.remote.ClientRemoteLocator;

/**
 * Dilogo de incluso de comentrios no histrico de projetos.
 * 
 * @author jnlopes
 */
public class ChangeServerOwnerProjectDialog extends ProjectsManagerDialog {

  /**
   * Caixa de texto para o nome do novo servidor dono do projeto.
   */
  private JTextField newOwnerServerNameText;

  /**
   * Lista de projetos selecionados.
   */
  private List<ProjectsManagerData> projectsManagerData;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao.
   * @param projectsManagerData Lista de projetos selecionados.
   */
  public ChangeServerOwnerProjectDialog(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projectsManagerData) {
    super(projectsManager);
    this.projectsManagerData = projectsManagerData;
    buildPanel();
  }

  /**
   * Monta o painel com os botes.
   * 
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {
    JButton changeButton = new JButton();
    JButton cancelButton = new JButton();
    final ChangeServerOwnerProjectDialog phd = this;
    changeButton.setAction(new AbstractAction() {
      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectsManager projectsManager = getProjectsManager();
        /*
         * Executa a task para alterar o nome do servidor dono do projeto
         */
        ChangeServerOwnerProjectTask task =
          new ChangeServerOwnerProjectTask(projectsManager,
            projectsManagerData, newOwnerServerNameText.getText());

        ApplicationFrame appFrame = projectsManager.getApplicationFrame();
        String taskTitle =
          getString("ChangeServerOwnerProjectDialog.running.title");
        String taskMsg =
          getString("ChangeServerOwnerProjectDialog.running.message");

        task.execute(appFrame, taskTitle, taskMsg);

        if (task.wasCancelled()) {
          String err =
            getString("ChangeServerOwnerProjectDialog.cancelled.message");
          task.showError(err);
          return;
        }

        if (task.getStatus() != true) {
          Exception exception = task.getError();
          String err =
            getString("ChangeServerOwnerProjectDialog.change.server.name.error.message");
          String msg = err + "\n" + exception.getMessage();
          StandardDialogs.showErrorDialog(appFrame, "", msg);
        }
        /*
         * Executa a tarefa e obtm seu retorno. O retorno  uma lista dos
         * projetos nos quais, por algum problema, no foi possvel incluir o
         * comentrio.
         */
        List<ProjectsManagerData> ret = task.getResult();

        // Verifica se houve falha para algum projeto. 
        if (ret != null && ret.size() > 0) {
          StringBuilder text = new StringBuilder();
          for (ProjectsManagerData prj : ret) {
            text.append(" - " + prj.getProjectName() + "\n");
          }
          String err =
            getString("ChangeServerOwnerProjectDialog.failure.change.some.server.name.message");
          String msg = err + "\n" + text.toString();
          //StandardDialogs.showWarningDialog(getParent(), "", msg);
          StandardDialogs.showErrorDialog(getApplication()
            .getApplicationFrame(), "", msg);
        }

        /*
         * Executa task que obtm dados atualizados dos projetos.
         */
        projectsManagerData = runUpdateProjectDataTask();

        projectsManager.refreshProjectsTable();

        phd.dispose();
      }
    });
    changeButton
      .setText(getString("ChangeServerOwnerProjectDialog.change.button"));

    cancelButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        phd.dispose();
      }
    });
    cancelButton
      .setText(getString("ChangeServerOwnerProjectDialog.cancel.button"));

    JComponent[] buttons = { changeButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);
    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(changeButton);
    buttonsPanel.add(cancelButton);

    /*
     * Definio do boto "Atualizar" como padro deste dilogo
     */
    getRootPane().setDefaultButton(changeButton);

    return buttonsPanel;
  }

  /**
   * Monta o componente de insero do texto.
   * 
   * @return o componente
   */
  private JPanel buildTextPanel() {
    final JPanel panel = new JPanel(new GridBagLayout());

    GBC gbc = new GBC(0, 0).west().insets(10, 15, 10, 0);

    final JLabel newServerNameLabel =
      new JLabel(getString("ChangeServerOwnerProjectDialog.new.owner.label"));
    panel.add(newServerNameLabel, gbc);

    newOwnerServerNameText = new JTextField(15);

    newOwnerServerNameText.setText(getRemoteSystemName());

    gbc = new GBC(1, 0).insets(10).horizontal();
    panel.add(newOwnerServerNameText, gbc);

    /* Ajusta o foco para o campo de edit */
    addWindowFocusListener(new WindowAdapter() {
      @Override
      public void windowGainedFocus(WindowEvent e) {
        newOwnerServerNameText.requestFocusInWindow();
      }
    });

    return panel;
  }

  /**
   * Obtm o nome do sistema que o cliente est conectado.
   * 
   * @return o nome do sistema.
   */
  private String getRemoteSystemName() {

    String serverName = null;
    final RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.server.getSystemName());
      }
    };
    // Esse dilogo no deve aparecer rapidamente pois dar uma impresso
    // ruim ao usurio no start-up do desktop. Ele somente aparecer caso
    // haja alguma demora real na conexo com o servidor.
    task.setProgressDialogDelay(10);
    task.execute(this.getOwner(), "", "");
    if (!task.getStatus()) {
      serverName = "";
    }
    serverName = task.getResult();
    if (serverName == null) {
      serverName = "";
    }

    return serverName;
  }

  /**
   * Monta o painel.
   */
  private void buildPanel() {

    //    mainPanel = new JPanel();
    //    mainPanel.setLayout(new GridBagLayout());

    final JPanel buttonsPanel = buildButtonsPanel();
    final JPanel textPanel = buildTextPanel();

    setLayout(new GridBagLayout());
    SortableTable table = new SortableTable(new ChangeServerModel());

    //table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
    //    final TableColumnModel columnModel = table.getColumnModel();
    //    final TableColumn column0 = columnModel.getColumn(0);
    //    column0.setCellRenderer(new TableScopeRenderer());
    //    column0.setResizable(false);
    //    column0.setPreferredWidth(20);

    add(new JScrollPane(table), new GBC(0, 0).both().insets(10));
    add(textPanel, new GBC(0, 1).both());
    add(buttonsPanel, new GBC(0, 2).horizontal());
  }

  /**
   * Executa task que obtm dados atualizados dos projetos.
   * 
   * @return Projetos atualizados.
   */
  private List<ProjectsManagerData> runUpdateProjectDataTask() {

    final ProjectsManager projectsManager = getProjectsManager();

    final ReloadSelectedProjectsTask rspt =
      new ReloadSelectedProjectsTask(projectsManager, projectsManagerData);

    rspt.execute(projectsManager.getApplicationFrame(),
      getString("ChangeServerOwnerProjectDialog.title"),
      getString("ChangeServerOwnerProjectDialog.message"));
    if (rspt.wasCancelled()) {
      final String err =
        getString("ChangeServerOwnerProjectDialog.cancelled.message");
      rspt.showError(err);
      return null;
    }

    if (rspt.getStatus() != true) {
      final Exception exception = rspt.getError();
      ProjectsManagerUI.handleException(this.getProjectsManager(), exception);
      return null;
    }

    final List<ProjectsManagerData> projectList = rspt.getResult();
    if (projectList == null) {
      final String errTitle =
        getString("ChangeServerOwnerProjectDialog.modification.date.title");
      final String errMsg =
        getString("ChangeServerOwnerProjectDialog.modification.date.message");
      StandardDialogs.showErrorDialog(getProjectsManager()
        .getApplicationFrame(), errTitle, errMsg);
      return null;
    }

    return projectList;
  }

  /**
   * Modelo para a tabela.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private class ChangeServerModel extends DefaultTableModel {

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName(int column) {
      return getString("ChangeServerModel.model.column.name." + column);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValueAt(int row, int column) {
      ProjectsManagerData elem = projectsManagerData.get(row);
      if (column == 0) {
        return elem.getScope().getIcon();
      }
      else if (column == 1) {
        return elem.getProjectName();
      }
      else {
        return elem.getServerOwnerName();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return projectsManagerData.size();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getColumnCount() {
      return 3;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
      final Object v = getValueAt(0, columnIndex);

      if (v == null) {
        return null;
      }

      return v.getClass();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCellEditable(int row, int column) {
      return false;
    }
  }
}
