package csbase.client.applications.statsviewer.graphic;

import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.io.File;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Observer;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.border.TitledBorder;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.title.LegendTitle;
import org.jfree.chart.title.TextTitle;
import org.jfree.chart.title.Title;
import org.jfree.data.category.DefaultCategoryDataset;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.statsviewer.StatsViewer;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.logic.ClientFile;
import csbase.logic.eventlogservice.LogsInfo;

/**
 * Painel que modela a funcionalidade comum aos painis para exibio de
 * grficos de estatsticas.
 */
public abstract class GraphicPanel extends JPanel implements Observer {

  /** Painel que contm o grfico */
  protected JFreeChart graph;

  /** Ttulo para o grfico */
  protected String title;

  /** Formato para exibir perodo de tempo associado a busca no sub-ttulo */
  private final DateFormat dfTitle;

  /** Referncia para a Aplicao */
  protected StatsViewer app;

  /**
   * Construtor.
   *
   * @param viewer referncia para aplicao
   * @param title ttulo do grfico
   */
  public GraphicPanel(final StatsViewer viewer, String title) {
    this.app = viewer;
    this.title = title;
    this.dfTitle =
      DateFormat.getDateInstance(DateFormat.MEDIUM, LNG.getLocale());
    viewer.addObserver(this);
  }

  /**
   * Cria um grfico barra 3D.
   *
   * @param xAxisLabel rtulo do eixo x
   * @param yAxisLabel rtulo do eixo y
   * @param showLegend exibir legenda
   */
  protected void newBarChart3DPanel(String xAxisLabel, String yAxisLabel,
    boolean showLegend) {

    graph =
      ChartFactory.createBarChart3D(title, xAxisLabel, yAxisLabel, null,
        PlotOrientation.VERTICAL, showLegend, true, false);

    CategoryPlot p = graph.getCategoryPlot();
    p.setBackgroundPaint(Color.LIGHT_GRAY);

    setLayout(new GridBagLayout());

    final ChartPanel chartPanel = new ChartPanel(graph);
    chartPanel.setPopupMenu(null);
    chartPanel.setBorder(new TitledBorder(""));
    chartPanel.setLayout(new GridBagLayout());

    JButton button =
      new JButton(new AbstractAction(app.getString("export.image")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          final ClientFile imageFile =
            ClientLocalFileChooserUtil.browseSingleFileInSaveMode(app
              .getApplicationFrame(), null, "png", null, false, null);
          if (imageFile == null) {
            return;
          }
          try {
            if (imageFile.exists()) {
              if (StandardDialogs.showYesNoDialog(app.getApplicationFrame(),
                app.getString("warning.title"), app
                .getString("override.question")) == StandardDialogs.NO_OPTION) {
                // no sobrescrever, cancelar operao
                return;
              }
            }
            ChartUtilities.saveChartAsJPEG(new File(imageFile.getStringPath()),
              graph, 800, 600);
          }
          catch (Exception exception) {
            app.showException(app.getString("exportFileError"), exception);
          }
        }
      });

    GBC gbc = new GBC(0, 0).insets(10, 5, 5, 5).both();
    add(chartPanel, gbc);

    gbc = new GBC(0, 1).insets(10, 5, 10, 5).east();
    add(button, gbc);

  }

  /**
   * Atualiza o subttulo para o grfico, conforme as informaes fornecidas.
   *
   * @param info informaes da busca e resultado corrente
   */
  public void updateSubTitle(LogsInfo info) {
    List<Title> list = new ArrayList<Title>();
    LegendTitle legendTitle = graph.getLegend();
    list.add(legendTitle);
    if (info != null) {
      Date initialDate = info.getStartDate();
      Date finalDate = info.getEndDate();
      if (initialDate != null && finalDate != null) {
        list.add(new TextTitle(String.format(app.getString("graphic.subtitle"),
          dfTitle.format(initialDate), dfTitle.format(finalDate))));
      }
    }
    graph.setSubtitles(list);
  }

  /**
   * Atualiza DataSet com os dados do mapa fornecido. Cada valor da chave do
   * mapa fornecido contm outro mapa associado as chaves do tipo data.
   *
   * @param map mapa [chave, [mapa<chave,valor>]]
   */
  protected void updateMapDataSet(Map<String, Map<String, Integer>> map) {
    DefaultCategoryDataset dataset = new DefaultCategoryDataset();
    for (Iterator<String> keyIterator = map.keySet().iterator(); keyIterator
      .hasNext();) {
      String date = keyIterator.next();
      Map<String, Integer> valueMap = map.get(date);
      for (Iterator<String> iterator = valueMap.keySet().iterator(); iterator
        .hasNext();) {
        String key = iterator.next();
        dataset.setValue(valueMap.get(key), key, date);
      }
    }
    graph.getCategoryPlot().setDataset(dataset);
  }

  /**
   * Atualiza DataSet com os dados fornecidos. Cada valor da chave do mapa
   * fornecido contm o nmero de ocorrncias deste valor.
   *
   * @param map mapa [chave, valor]
   * @param rowkey identificador dos valores no dataSet
   */
  protected void updateSimpleDataSet(Map<String, Integer> map, String rowkey) {
    DefaultCategoryDataset dataset = new DefaultCategoryDataset();
    for (Iterator<String> keyIterator = map.keySet().iterator(); keyIterator
      .hasNext();) {
      String key = keyIterator.next();
      Integer number = map.get(key);
      dataset.setValue(number, rowkey, key);
    }
    graph.getCategoryPlot().setDataset(dataset);
  }

}
