package csbase.client.preferences;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.types.PVBoolean;
import csbase.client.preferences.types.PVInteger;
import csbase.client.preferences.types.PVString;
import csbase.client.preferences.util.PreferenceBundle;

/**
 * Classe que disponibiliza a API que permite consultarmos as preferncias do
 * usurio.
 * 
 * A rvore de preferncias do usurio, como o prprio nome j diz,  uma
 * estrutura recursiva, onde cada n  uma instncia de
 * {@link PreferenceCategory}. Um objeto {@link PreferenceCategory}  apenas um
 * agrupador de preferncias. Logo, este armazena um conjunto de preferncias e
 * tambm possui categorias filhas.
 * 
 * Podemos consultar o valor de uma preferncia da seguinte forma:
 * 
 * <code>
 * PreferenceManager pm = PreferenceManager.getInstance();
 * PreferenceCategory root = pm.loadPreferences();
 * 
 * PreferenceCategory desktopCategory = root.getCategory(DesktopPref.class);
 * boolean value = desktopCategory.getPreferenceAsBoolean(DesktopPref.SHOW_HIDDEN_FILES);
 * </code>
 * 
 * O cdigo abaixo mostra como consultamos as preferncias dentro da classe da
 * aplicao:
 * 
 * <code>
 * PreferenceCategory appPref = getPreferences();
 * String value = appPref.getPreferenceAsString(NotepadPref.FONT);
 * </code>
 * 
 * Cada categoria est associada a uma enumerao que a define. Nesta
 * enumerao, cada constante define uma preferncia. Para mais informaes de
 * como definir preferncias atravs de enumeraes, veja a interface
 * {@link PreferenceDefinition}.
 * 
 * @see PreferenceManager PreferenceValue
 * 
 * @author Tecgraf
 */
public class PreferenceCategory {

  /** Identificador da raz de preferncias. */
  static String ROOT_ID = "root";

  /** Identificador da categoria de preferncias */
  private String id;

  /** Objeto usado na internacionalizao. */
  private PreferenceBundle preferenceBundle;

  /** Referncia para categoria original quando esta instncia  uma cpia. */
  private PreferenceCategory original;

  /** Mapeia o valor de preferncia do usurio a constante que o define. */
  private Map<String, PreferenceValue<?>> preferences;

  /** Mapeia as categorias filhas a seus respectivos identificadores. */
  private Map<String, PreferenceCategory> children;

  /**
   * Construtor padro.
   * 
   * @param id identificador da categoria.
   * @param preferenceBundle objeto usado na internacionalizao.
   */
  PreferenceCategory(String id, PreferenceBundle preferenceBundle) {
    this(id, preferenceBundle, null);
  }

  /**
   * Construtor padro.
   * 
   * @param id identificador da categoria.
   * @param preferenceBundle objeto usado na internacionalizao.
   * @param original referncia para a categoria original se est for uma cpia.
   */
  PreferenceCategory(String id, PreferenceBundle preferenceBundle,
    PreferenceCategory original) {
    if (id == null) {
      throw new PreferenceException(
        "Identificador de uma categoria no pode ser nulo.");
    }
    if (preferenceBundle == null) {
      throw new PreferenceException("bundle no pode ser nulo.");
    }
    this.id = id;
    this.preferenceBundle = preferenceBundle;
    this.original = original;
    preferences = new LinkedHashMap<String, PreferenceValue<?>>();
    children = new LinkedHashMap<String, PreferenceCategory>();
  }

  /**
   * Identificador da categoria.
   * 
   * @return identificador da categoria.
   */
  public String getId() {
    return id;
  }

  /**
   * True se est categoria for raz da rvore de preferncias, false caso
   * contrrio.
   * 
   * @return true se est categoria for raz da rvore de preferncias, false
   *         caso contrrio.
   */
  public boolean isRoot() {
    return ROOT_ID.equals(id);
  }

  /**
   * Retorna true se esta categoria for uma cpia, false caso contrrio.
   * 
   * @return retorna true se esta categoria for uma cpia, false caso contrrio.
   */
  public boolean isCopy() {
    return original != null;
  }

  /**
   * Referncia para categoria original quando esta instncia  uma cpia.
   * 
   * @return referncia para categoria original quando esta instncia  uma
   *         cpia.
   */
  public PreferenceCategory getOriginal() {
    return original;
  }

  /**
   * Rtulo da categoria.
   * 
   * @return rtulo da categoria.
   */
  public String getLabel() {
    int lastDot = Math.max(getId().lastIndexOf("."), getId().lastIndexOf("$"));
    int length = getId().length();
    String key = getId().substring(lastDot + 1, length) + ".label";
    return preferenceBundle.get(key);
  }

  /**
   * Retorna a categoria associada a classe da enumerao que a define. Caso no
   * haj uma categoria associada a enumerao, uma exceo, do tipo
   * {@link PreferenceException}  lanada.
   * 
   * @param enumClass classe da enumerao que encapsula as preferncias.
   * @return categoria equivalente.
   */
  public PreferenceCategory getCategory(
    Class<? extends PreferenceDefinition> enumClass) {
    if (enumClass == null) {
      throw new IllegalArgumentException(
        "Classe da enumerao no pode ser nulo.");
    }

    PreferenceCategory child = children.get(enumClass.getName());

    if (child == null) {
      throw new PreferenceException("Categoria " + getId()
        + " no possui categoria filha " + enumClass.getName());
    }

    return child;
  }

  /**
   * Retorna a lista com todas as categorias filhas deste n.
   * 
   * @return lista com todas as categorias filhas.
   */
  public List<PreferenceCategory> getCategories() {
    return new ArrayList<PreferenceCategory>(children.values());
  }

  /**
   * Verifica se existe a preferncia com o dado nome.
   * 
   * @param name constante que define a preferncia.
   * @return true se existir uma preferncia com o dado nome, false caso
   *         contrrio.
   */
  public boolean hasPreference(PreferenceDefinition name) {
    return getPreference(name) != null;
  }

  /**
   * Retorna o valor da preferncia.
   * 
   * @param name constante que define a preferncia.
   * @return valor da preferncia.
   */
  public PreferenceValue<?> getPreference(PreferenceDefinition name) {
    if (name == null) {
      throw new IllegalArgumentException("name no pode ser nulo.");
    }
    return preferences.get(name.toString());
  }

  /**
   * Retorna o objeto que encapsula um valor de preferncia que  do tipo
   * {@link String} correspondente a constante dada.
   * 
   * @param name constante que define a preferncia.
   * @return valor de preferncia {@link String}.
   */
  public PVString getPVString(PreferenceDefinition name) {
    return (PVString) getPreference(name);
  }

  /**
   * Retorna o objeto que encapsula um valor de preferncia boleano
   * correspondente a constante dada.
   * 
   * @param name constante que define a preferncia.
   * @return valor de preferncia boleano.
   */
  public PVBoolean getPVBoolean(PreferenceDefinition name) {
    return (PVBoolean) getPreference(name);
  }

  /**
   * Retorna o objeto que encapsula um valor de preferncia que  do tipo
   * {@link Integer} correspondente a constante dada.
   * 
   * @param name constante que define a preferncia.
   * @return valor de preferncia {@link Integer}.
   */
  public PVInteger getPVInteger(PreferenceDefinition name) {
    return (PVInteger) getPreference(name);
  }

  /**
   * Retorna o valor da preferncia como {@link String}. Para isso, o valor da
   * preferncia deve ser uma instncia de {@link PVString}, caso contrrio ser
   * lanado uma exceo do tipo {@link PreferenceException}.
   * 
   * @param name constante que define a preferncia.
   * @return valor em string.
   */
  public String getPreferenceAsString(PreferenceDefinition name) {
    try {
      return getPVString(name).getValue();
    }
    catch (Exception e) {
      throw new PreferenceException("Valor da preferncia " + name
        + " no  String.");
    }
  }

  /**
   * Retorna o valor da preferncia como {@link Boolean}. Para isso, o valor da
   * preferncia deve ser uma instncia de {@link PVBoolean}, caso contrrio
   * ser lanado uma exceo do tipo {@link PreferenceException}.
   * 
   * @param name constante que define a preferncia.
   * @return valor booleano.
   */
  public Boolean getPreferenceAsBoolean(PreferenceDefinition name) {
    try {
      return getPVBoolean(name).getValue();
    }
    catch (Exception e) {
      throw new PreferenceException("Valor da preferncia " + name
        + " no  boleano.");
    }
  }

  /**
   * Retorna o valor da preferncia como {@link Integer}. Para isso, o valor da
   * preferncia deve ser uma instncia de {@link PVInteger}, caso contrrio
   * ser lanado uma exceo do tipo {@link PreferenceException}.
   * 
   * @param name constante que define a preferncia.
   * @return valor inteiro.
   */
  public Integer getPreferenceAsInt(PreferenceDefinition name) {
    try {
      return getPVInteger(name).getValue();
    }
    catch (Exception e) {
      throw new PreferenceException("Valor da preferncia " + name
        + " no  inteiro.");
    }
  }

  /**
   * Retorna lista com os valores de preferncias que atendem as polticas de
   * visibilidade.
   * 
   * @param policies array com as polticas de visibilidade.
   * 
   * @return lista com todos os valores de preferncias.
   */
  public List<PreferenceValue<?>> getPreferences(PreferencePolicy... policies) {
    List<PreferenceValue<?>> result = new ArrayList<PreferenceValue<?>>();
    if (policies != null) {
      for (PreferenceValue<?> value : preferences.values()) {
        for (PreferencePolicy policy : policies) {
          if (value.getPolicy().equals(policy)) {
            result.add(value);
            break;
          }
        }
      }
    }
    return result;
  }

  /**
   * Retorna true se h pelo menos uma preferncia com uma das polticas de
   * visibilidade passadas por parmetro, false caso contrrio.
   * 
   * @param policies array com as polticas de visibilidade.
   * 
   * @return true se h pelo menos uma preferncia visvel, false caso
   *         contrrio.
   */
  public boolean hasPreferences(PreferencePolicy... policies) {
    return !getPreferences(policies).isEmpty();
  }

  /**
   * Retorna uma cpia desta categoria. A cpia de uma categoria contm cpias
   * de todos os valores de preferncias. Nota: os ouvintes dos valores de
   * preferncias no so copiados.
   * 
   * @return objeto que representa a cpia desta categoria.
   */
  public PreferenceCategory copy() {
    if (isCopy()) {
      throw new PreferenceException(
        "No  permitido criar cpias de categorias que j so cpias.");
    }

    PreferenceCategory copy =
      new PreferenceCategory(id, preferenceBundle, this);

    for (Entry<String, PreferenceValue<?>> entry : preferences.entrySet()) {
      String name = entry.getKey();
      PreferenceValue<?> value = entry.getValue();

      copy.addPreference(name, value.clone());
    }

    for (PreferenceCategory child : children.values()) {
      copy.addChild(child.getId(), child.copy());
    }

    return copy;
  }

  /**
   * Cria uma categoria filha dado um identificador {@link String}.
   * 
   * @param categoryId identificador da nova categoria.
   * @param preferenceBundle objeto usado na internacionalizao.
   * @return Categoria criada.
   */
  PreferenceCategory createCategory(String categoryId,
    PreferenceBundle preferenceBundle) {
    PreferenceCategory child =
      new PreferenceCategory(categoryId, preferenceBundle);

    addChild(categoryId, child);

    return child;
  }

  /**
   * Adiciona categoria filha.
   * 
   * @param categoryId identificador da categoria.
   * @param child categoria filha.
   */
  void addChild(String categoryId, PreferenceCategory child) {
    children.put(categoryId, child);
  }

  /**
   * Retorna a categoria associada ao determinado identificador.
   * 
   * @param categoryId identificador da categoria.
   * @return objeto que encapsula uma categoria de preferncias.
   */
  PreferenceCategory getCategory(String categoryId) {
    return children.get(categoryId);
  }

  /**
   * Adiciona nova preferncia.
   * 
   * @param name String que define uma preferncia.
   * @param value valor da preferncia.
   */
  void addPreference(String name, PreferenceValue<?> value) {
    if (name == null || value == null) {
      throw new IllegalArgumentException("name/value no pode ser null.");
    }

    preferences.put(name, value);
  }

  /**
   * Retorna um valor de preferncia dado um string que representa seu nome.
   * 
   * @param name nome da preferncia.
   * @return valor da preferncia.
   */
  PreferenceValue<?> getPreference(String name) {
    return preferences.get(name);
  }

  /**
   * Retorna mapa com todas as preferncias desta categoria.
   * 
   * @return todas as preferncias desta categoria.
   */
  Map<String, PreferenceValue<?>> getPreferencesMap() {
    return preferences;
  }

  /**
   * Intncias dessa classe s podem ser criadas pelo {@link PreferenceManager},
   * logo, as implementaes de {@link PreferenceCategory#equals(Object)},
   * {@link PreferenceCategory#hashCode()} e
   * {@link PreferenceCategory#toString()} so muito simples e devem levar em
   * conta apenas se a instncia for o mesma.
   * 
   * Sobrescrito apenas para fins de documentao.
   */
  @Override
  public boolean equals(Object obj) {
    return super.equals(obj);
  }

  /**
   * Intncias dessa classe s podem ser criadas pelo {@link PreferenceManager},
   * logo, as implementaes de {@link PreferenceCategory#equals(Object)},
   * {@link PreferenceCategory#hashCode()} e
   * {@link PreferenceCategory#toString()} so muito simples e devem levar em
   * conta apenas se a instncia for o mesma.
   * 
   * Sobrescrito apenas para fins de documentao.
   */
  @Override
  public int hashCode() {
    return super.hashCode();
  }

  /**
   * Intncias dessa classe s podem ser criadas pelo {@link PreferenceManager},
   * logo, as implementaes de {@link PreferenceCategory#equals(Object)},
   * {@link PreferenceCategory#hashCode()} e
   * {@link PreferenceCategory#toString()} so muito simples e devem levar em
   * conta apenas se a instncia for o mesma.
   * 
   * Sobrescrito apenas para fins de documentao.
   */
  @Override
  public String toString() {
    return super.toString();
  }
}
