package csbase.client.util.table;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

/**
 * Clula de <i>ComboBoxes</i>: a clula que exibe uma {@link JComboBox}.
 */
public final class ComboBoxCellView extends AbstractCellView {

  /**
   * Os tens disponveis para seleo.
   */
  private List<ComboBoxCellViewItem> availableItems;
  /**
   * A <i>combobox</i>.
   */
  private JComboBox comboBox;

  /**
   * O tem selecionado.
   */
  private ComboBoxCellViewItem selectedItem;

  /**
   * Cria uma clula de <i>combobox</i>.
   * 
   * @param availableItems A relao de tens disponveis para seleo (No
   *        aceita {@code null} e no pode estar vazia).
   */
  public ComboBoxCellView(List<ComboBoxCellViewItem> availableItems) {
    setAvailableItems(availableItems);
    createComboBox();
  }

  @Override
  public String getTip() {
    if (selectedItem == null) {
      return null;
    }
    return selectedItem.getTip();
  }

  public Object getValue() {
    ComboBoxCellViewItem item = (ComboBoxCellViewItem) comboBox
      .getSelectedItem();
    if (item == null) {
      return null;
    }
    return item.getValue();

  }

  public JComponent getView() {
    return comboBox;
  }

  @Override
  protected void keepValue(Object value) {
    for (ComboBoxCellViewItem item : availableItems) {
      if (item.getValue().equals(value)) {
        comboBox.setSelectedItem(item);
        return;
      }
    }
    throw new IllegalArgumentException(String.format(
      "O item da combobox cujo valor  %s no foi encontrado.", value));
  }

  /**
   * Cria a <i>combobox</i>.
   */
  private void createComboBox() {
    comboBox = new JComboBox();
    comboBox.setModel(createComboBoxModel());
    comboBox.setRenderer(createListCellRenderer());
    comboBox.getModel().addListDataListener(createListDataListener());
  }

  /**
   * Cria um {@link ListDataListener}.
   * 
   * @return .
   */
  private ListDataListener createListDataListener() {
    return new ListDataListener() {

      public void intervalRemoved(ListDataEvent e) {
      }

      public void intervalAdded(ListDataEvent e) {
      }

      public void contentsChanged(ListDataEvent e) {
        if (selectedItem != null) {
          comboBox.setToolTipText(selectedItem.getTip());
        }
        else {
          comboBox.setToolTipText(null);
        }
        fireValueWasChanged();
      }
    };
  }

  /**
   * Cria o modelo para a <i>combobox</i>.
   * 
   * @return O modelo.
   */
  private ComboBoxModel createComboBoxModel() {
    return new ComboBoxModel() {

      private List<ListDataListener> listeners = new LinkedList<ListDataListener>();

      public void addListDataListener(ListDataListener l) {
        listeners.add(l);
      }

      public Object getElementAt(int index) {
        return availableItems.get(index);
      }

      public Object getSelectedItem() {
        return selectedItem;
      }

      public int getSize() {
        return availableItems.size();
      }

      public void removeListDataListener(ListDataListener l) {
        listeners.remove(l);
      }

      public void setSelectedItem(Object anItem) {
        selectedItem = (ComboBoxCellViewItem) anItem;
        for (ListDataListener listener : listeners) {
          listener.contentsChanged(new ListDataEvent(comboBox,
            ListDataEvent.CONTENTS_CHANGED, -1, 1));
        }
      }
    };
  }

  /**
   * Cria o <i>renderer</i> da <i>combobox</i>.
   * 
   * @return O renderer.
   */
  private DefaultListCellRenderer createListCellRenderer() {
    return new DefaultListCellRenderer() {

      @Override
      public Component getListCellRendererComponent(JList list, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
        Component component = super.getListCellRendererComponent(list, value,
          index, isSelected, cellHasFocus);
        ComboBoxCellViewItem item = (ComboBoxCellViewItem) value;
        if (item != null) {
          setText(item.getLabel());
          if (item.getTip() != null) {
            setToolTipText(item.getTip());
          }
        }
        return component;
      }
    };
  }

  /**
   * Atribui os tens disponveis para seleo nesta clula.
   * 
   * @param availableItems Os tens disponveis (No aceita {@code null} e no
   *        pode estar vazio).
   */
  private void setAvailableItems(List<ComboBoxCellViewItem> availableItems) {
    if (availableItems == null) {
      throw new IllegalArgumentException(
        "O parmetro availableItems est nulo.");
    }

    if (availableItems.isEmpty()) {
      throw new IllegalArgumentException(
        "O parmetro availableItems est vazio.");
    }

    this.availableItems = new ArrayList<ComboBoxCellViewItem>(availableItems);
    Collections.sort(this.availableItems);
  }
}
