/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.actions;

import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.commandsmonitor.CommandsMonitor;
import csbase.client.applications.executor.ExecutorFrame;
import csbase.client.applications.flowapplication.FlowApplication;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.CommonClientProject;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;

/**
 * Inicializa a aplicao de execuo de comando adequada, com uma cpia de um
 * nico comando selecionado.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class ExecuteCommandAction extends AbstractCommandsAction {

  /**
   * Construtor.
   * 
   * @param application Aplicao que detm essa ao.
   */
  public ExecuteCommandAction(CommandsMonitor application) {
    super(application, false, ApplicationImages.ICON_EXECUTE_CMD_16);
  }

  /**
   * Filtro indicando que esta ao aceita qualquer comando que no esteja no
   * estado {@link CommandStatus#SCHEDULED agendado} ou
   * {@link CommandStatus#SYSTEM_FAILURE falha no sistema}. {@inheritDoc}
   */
  @Override
  protected boolean accept(CommandInfo command) {
    return command.getStatus() != CommandStatus.SCHEDULED
      && command.getStatus() != CommandStatus.SYSTEM_FAILURE;
  }

  /**
   * Trata esta ao obtendo o comando selecionado e iniciando a aplicao de
   * execuo com uma cpia dele. {@inheritDoc}
   */
  @Override
  protected void handleActionPerformed(ActionEvent ae) throws Exception {
    final CommandInfo command = getSelectedCommand();
    executeCommand(command);
  }

  /**
   * Inicializa a aplicao de execuo de comando com uma cpia do comando
   * passado, simulando assim uma re-execuo do comando
   * 
   * @param command Comando a ter uma cpia executada.
   * 
   * @return <tt>true</tt> se foi bem sucedido a abrir a janela de execuo com
   *         uma cpia do comando passado.
   * 
   * @throws ApplicationException se no foi possvel abrir a aplicao de
   *         execuo de comando - {@link FlowApplication} ou
   *         {@link ExecutorFrame}.
   * @throws ParameterNotFoundException No foi possvel obter o valor de um
   *         parmetro no configurador do comando, para que este pudesse ser
   *         atribuido ao configurado de sua cpia. Possivelmente o configurador
   *         do comando foi alterado sem que tenha sido criada uma nova verso
   *         para este configurador.
   */
  private boolean executeCommand(final CommandInfo command)
    throws ApplicationException, ParameterNotFoundException {
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    if (null == project) {
      ApplicationFrame frame = getApplication().getApplicationFrame();
      String msg = getString("ExecuteCommandAction.project.missing.error");
      StandardErrorDialogs.showErrorDialog(frame, frame.getTitle(), msg);
      return false;
    }
    if (!project.isWritable()) {
      ApplicationFrame frame = getApplication().getApplicationFrame();
      String msg =
        getString("ExecuteCommandAction.executing.permission.error", project
          .getName());
      StandardErrorDialogs.showErrorDialog(frame, frame.getTitle(), msg);
      return false;
    }

    RemoteTask<AlgorithmConfigurator> task =
      new RemoteTask<AlgorithmConfigurator>() {
        @Override
        public void performTask() throws Exception {
          AlgorithmConfigurator configurator = command.getConfigurator();
          setResult(configurator);
        }
      };
    String title = getApplication().getString("task.command.retrieve.title");
    String msg = getApplication().getString("task.command.retrieve.message");
    if (!task.execute(getApplication().getApplicationFrame(), title, msg)) {
      return false;
    }
    AlgorithmConfigurator configurator = task.getResult();
    if (null == configurator) {
      String error = getApplication().getString("task.command.retrieve.error");
      getApplication().showError(error);
      return false;
    }

    /*
     * Quando o configurador no tem nome nem verso, abre a aplicao de fluxo
     * para visualizar o comando. Note que um configurador de fluxo PODE ter
     * nome e verso, mas nesse caso ele deve ser aberto pelo ExecutorFrame!
     */
    if (null == configurator.getAlgorithmName()
      && null == configurator.getAlgorithmVersionId()
      && configurator.getConfiguratorType() == ConfiguratorType.FLOW) {

      ApplicationManager applicationManager = ApplicationManager.getInstance();
      FlowApplication application =
        applicationManager.runApplication(FlowApplication.class);
      return application.editConfigurator(
        (FlowAlgorithmConfigurator) configurator, command.getDescription());
    }
    String algName = configurator.getAlgorithmName();
    AlgorithmVersionId algVersionId = configurator.getAlgorithmVersionId();
    Map<String, String> parameters = new HashMap<String, String>();
    Set<String> names = configurator.getParameterNames();
    for (String name : names) {
      String value = configurator.getParameterValue(name);
      parameters.put(name, value);
    }

    ExecutorFrame executorFrame =
      (ExecutorFrame) ApplicationManager.getInstance().runApplication(
        "executor");
    executorFrame.showConfigurator(algName, algVersionId, parameters);
    return true;
  }
}
