/*
 * $Id: DirectoryContentsPanel.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.desktop.dircontents;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Window;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.ImageIcon;
import javax.swing.InputMap;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.SortOrder;
import javax.swing.TransferHandler;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectFileCellRenderer;
import csbase.client.project.ProjectFileContainer;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.project.ProjectTreeSelectionEvent;
import csbase.client.project.ProjectTreeSelectionListener;
import csbase.client.project.action.CommonDirectoryCreationAction;
import csbase.client.project.action.CommonDirectoryRefreshAction;
import csbase.client.project.action.CommonFileChangeStateAction;
import csbase.client.project.action.CommonFileCopyAction;
import csbase.client.project.action.CommonFileCutAction;
import csbase.client.project.action.CommonFileDeleteAction;
import csbase.client.project.action.CommonFileDownloadAction;
import csbase.client.project.action.CommonFileImportAction;
import csbase.client.project.action.CommonFileOpenAction;
import csbase.client.project.action.CommonFilePasteAction;
import csbase.client.project.action.CommonFilePropertiesShowAction;
import csbase.client.project.action.CommonFileRenameAction;
import csbase.client.project.action.CommonFileTypeChangeAction;
import csbase.client.project.action.CommonMultiSelectionFileTypeChangeAction;
import csbase.client.project.action.ExportFilesToDirAction;
import csbase.client.project.action.ProjectTreeUpdateAction;
import csbase.client.project.action.SingleFileExportAction;
import csbase.client.project.tasks.FileTypeIconCellRenderer;
import csbase.client.project.tasks.GetChildrenTask;
import csbase.client.util.DateTableCellRenderer;
import csbase.client.util.SizeTableCellRenderer;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientProjectFileComparator;
import csbase.logic.CommonClientProject;
import csbase.logic.FileTypeComparator;
import csbase.logic.NoHiddenFileFilter;
import csbase.logic.ProjectFileFilter;

/**
 * Cria um painel que exibe o contedo de um diretrio, toda vez que o usurio
 * selecionar algum na rvore de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DirectoryContentsPanel extends JPanel implements
  ProjectFileContainer {

  /**
   * Formatador para as colunas da tabela.
   */
  private static class MyObjectTableProvider implements ObjectTableProvider {
    /**
     * Classe para cada coluna.
     */
    private static final Class<?>[] COL_CLASSES = { ImageIcon.class,
        ClientProjectFile.class, Long.class, Date.class };
    /**
     * Nomes para cada coluna.
     */
    private static final String[] COL_NAMES = new String[] { "",
        LNG.get("DIR_CONTENTS_NAME"), LNG.get("DIR_CONTENTS_SIZE"),
        LNG.get("DIR_CONTENTS_DATE") };

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getCellValue(final Object row, final int col) {
      final ClientProjectFile file = (ClientProjectFile) row;
      switch (col) {
        case 0:
          //          ImageIcon icon = ClientProjectFileProperties.getImageIcon(file);
          //          icon.setDescription(getType(file));
          return file;

        case 1:
          return file;

        case 2:
          return getSize(file);

        case 3:
          return getDate(file);
      }
      return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      return MyObjectTableProvider.COL_CLASSES;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      return MyObjectTableProvider.COL_NAMES;
    }

    /**
     * Obtm a data da ltima modificao do arquivo.
     * 
     * @param file arquivo.
     * 
     * @return data da ltima modificao do arquivo.
     */
    private Date getDate(final ClientProjectFile file) {
      return new Date(file.getModificationDate());
    }

    /**
     * Obtm o tamanho do arquivo.
     * 
     * @param file arquivo.
     * 
     * @return tipo do arquivo.
     */
    private Long getSize(final ClientProjectFile file) {
      if (file.isDirectory()) {
        return (long) -1;
      }
      return file.size();
    }
  }

  /**
   * ndice da coluna de cones que representam o tipo de aqruivo.
   */
  static final int TYPE_COL_INDEX = 0;

  /**
   * ndice da coluna que contm nome de aqruivo.
   */
  static final int NAME_COL_INDEX = 1;

  /**
   * ndice da coluna que contm nome de aqruivo.
   */
  static final int SIZE_COL_INDEX = 2;

  /**
   * ndice da coluna que contm nome de aqruivo.
   */
  static final int DATE_COL_INDEX = 3;

  /**
   * Tabela que exibe lista de arquivos de um diretrio
   */
  private SortableTable dirContentsTable;

  /**
   * Modelo para a tabela de contedo de um diretrio
   */
  private ObjectTableModel<ClientProjectFile> dirModel;

  /**
   * Barra de status do diretrio
   */
  private JLabel statusBar;

  /**
   * ltimo texto exibido na barra de status
   */
  private String lastStatusBarText = " ";

  /**
   * Menu de popup de diretrio
   */
  private JPopupMenu popupDir;

  /**
   * Menu de popup de arquivo
   */
  private JPopupMenu popupFile;

  /**
   * Menu de popup de mltipla seleo (arquivos/diretrios)
   */
  private JPopupMenu popupMultSelection;

  /**
   * Item de menu "Colar" em diretrio
   */
  protected JMenuItem dirPasteFileMenuItem;

  /**
   * Controle de edio do nome do arquivo
   */
  private boolean currentEditingAllowed;

  /**
   * Seleo anterior de arquivo ou diretrio.  usado para controlar a
   * atualizao da SortableTable.
   */
  private ClientProjectFile previousSelectedFile;

  /**
   * Item de menu para diretrios: deleo
   */
  protected JMenuItem dirDeleteFileMenuItem;

  /**
   * Item de menu para diretrios: cut
   */
  protected JMenuItem dirCutFileMenuItem;

  /**
   * Item de menu para diretrios: rename
   */
  protected JMenuItem dirRenameFileMenuItem;

  /**
   * Item de menu para diretrios: criao de diretrio
   */
  protected JMenuItem dirCreateDirMenuItem;

  /**
   * Item de menu para diretrios: refresh
   */
  protected JMenuItem dirRefreshDirMenuItem;

  /**
   * Item de menu para diretrios: copy
   */
  protected JMenuItem dirCopyFileMenuItem;

  /**
   * Item de menu para diretrios: upload
   */
  protected JMenuItem dirUploadMenuItem;

  /**
   * Item de menu para diretrios: export
   */
  protected JMenuItem dirExportMenuItem;

  /**
   * Item de menu para arquivo: abrir arquivo com a aplicao associada
   */
  protected JMenuItem fileOpenMenuItem;

  /**
   * Item de menu para arquivo: abrir com bloco de notas
   */
  protected JMenuItem notepadFileOpenMenuItem;

  /**
   * Item de menu para arquivo: propriedaded
   */
  protected JMenuItem fileSeeFilePropertiesMenuItem;

  /**
   * Item de menu para arquivo: cut
   */
  protected JMenuItem fileCutFileMenuItem;

  /**
   * Item de menu para arquivo: copy
   */
  protected JMenuItem fileCopyFileMenuItem;

  /**
   * Item de menu para arquivo: renomear
   */
  protected JMenuItem fileRenameFileMenuItem;

  /**
   * Item de menu para arquivo: tipo
   */
  protected JMenuItem fileChangeFileTypeMenuItem;

  /**
   * Item de menu para arquivo: apagar
   */
  protected JMenuItem fileDeleteFileMenuItem;

  /**
   * Item de menu para arquivo: importao
   */
  protected JMenuItem fileDownloadFileMenuItem;

  /**
   * Item de menu para arquivo: exportao
   */
  protected JMenuItem fileExportFileMenuItem;

  /**
   * Item de menu para arquivo: avanado
   */
  protected JMenu advancedSubmenu;

  /**
   * Item de menu para arquivo: alterar estado
   */
  protected JMenuItem fileChangeStateFileMenuItem;

  /**
   * Item de menu para mltiplos arquivos/diretrios: tipo
   */
  protected JMenuItem msChangeFileTypeMenuItem;

  /**
   * Item de menu para mltiplos arquivos/diretrios: cut
   */
  protected JMenuItem msCutFileMenuItem;

  /**
   * Item de menu para mltiplos arquivos/diretrios: copy
   */
  protected JMenuItem msCopyFileMenuItem;

  /**
   * Item de menu para mltiplos arquivos/diretrios: deleo
   */
  protected JMenuItem msDeleteFileMenuItem;

  /**
   * Item de menu para mltiplos arquivos/diretrios: exportao
   */
  protected JMenuItem msExportFileMenuItem;

  /**
   * Janela me
   */
  private Window ownerWindow;

  /** */
  private String ownerTitle;

  /** */
  private ProjectTree projectTree;

  /** */
  private JScrollPane scrollPane;

  /**
   * Indica se a transferncia de arquivos est habilitada.
   */
  protected boolean transferEnabled;

  /**
   * Indica se a remoo de arquivos est habilitada.
   */
  protected boolean deleteEnabled;

  /**
   * Indica se a edio de nomes de arquivos est habilitada.
   */
  protected boolean renameEnabled;

  /**
   * Indica se a atualizao de diretrios e do projeto est habilitada.
   */
  protected boolean refreshEnabled;

  /**
   * Filtro para os arquivos exibidos na tabela do painel.
   */
  private ProjectFileFilter filter;

  /**
   * Limpa a tabela e a barra de status
   */
  public void clear() {
    clearModel();
    lastStatusBarText = " ";
    statusBar.setText(lastStatusBarText);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clearClipboard() {
    final DirectoryContentsTransferHandler handler =
      (DirectoryContentsTransferHandler) dirContentsTable.getTransferHandler();
    handler.clearClipboard();
  }

  /**
   * Limpa o modelo, na EDT.
   */
  private void clearModel() {
    SwingThreadDispatcher.invokeLater(new Runnable() {

      @Override
      public void run() {
        dirModel.clear();
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clearSelection() {
    dirContentsTable.clearSelection();
  }

  /**
   * Habilita/desabilita alguns itens do menu para diretrios, de acordo com o
   * estado atual da janela de detalhes.
   */
  public void configurePopupDir() {
    if (getSelectedFile().getParent() == null) {
      dirCutFileMenuItem.setEnabled(false);
      dirRenameFileMenuItem.setEnabled(false);
      dirDeleteFileMenuItem.setEnabled(false);
    }
    else {
      dirCutFileMenuItem.setEnabled(transferEnabled);
      dirRenameFileMenuItem.setEnabled(renameEnabled);
      dirDeleteFileMenuItem.setEnabled(deleteEnabled);
    }
    dirRefreshDirMenuItem.setEnabled(refreshEnabled);
    dirCopyFileMenuItem.setEnabled(transferEnabled);
    dirPasteFileMenuItem.setEnabled(transferEnabled && hasFileToPaste());
  }

  /**
   * Habilita/desabilita alguns itens do menu para arquivos, de acordo com o
   * estado atual da janela de detalhes.
   * 
   * @param file o arquivo selecionado.
   */
  public void configurePopupFile(final ClientProjectFile file) {
    fileCopyFileMenuItem.setEnabled(transferEnabled);
    fileCutFileMenuItem.setEnabled(transferEnabled);
    fileDeleteFileMenuItem.setEnabled(deleteEnabled);
    fileRenameFileMenuItem.setEnabled(renameEnabled);
    final boolean isUnderConstruction = file.isUnderConstruction();
    advancedSubmenu.setEnabled(isUnderConstruction);
  }

  /**
   * Habilita/desabilita alguns itens do menu para mltiplos
   * arquivos/diretrios, de acordo com o estado atual da janela de detalhes.
   */
  public void configurePopupMultiSelection() {
    if (hasDirectories(getSelectedFiles())) {
      msChangeFileTypeMenuItem.setEnabled(false);
    }
    else {
      msChangeFileTypeMenuItem.setEnabled(true);
    }
    msCutFileMenuItem.setEnabled(transferEnabled);
    msCopyFileMenuItem.setEnabled(transferEnabled);
    msDeleteFileMenuItem.setEnabled(deleteEnabled);
  }

  /**
   * Criao de menu do contexto para diretrios
   * 
   * @return o menu
   */
  protected JPopupMenu createPopupDir() {
    /* Cria os tens de menu */
    dirCreateDirMenuItem =
      new JMenuItem(new CommonDirectoryCreationAction(this));
    dirRefreshDirMenuItem =
      new JMenuItem(new CommonDirectoryRefreshAction(this));
    dirCutFileMenuItem = new JMenuItem(new CommonFileCutAction(this));
    dirCopyFileMenuItem = new JMenuItem(new CommonFileCopyAction(this));
    dirPasteFileMenuItem = new JMenuItem(new CommonFilePasteAction(this));
    dirRenameFileMenuItem = new JMenuItem(new CommonFileRenameAction(this));
    dirDeleteFileMenuItem = new JMenuItem(new CommonFileDeleteAction(this));
    dirUploadMenuItem = new JMenuItem(new CommonFileImportAction(this));
    dirExportMenuItem = new JMenuItem(new ExportFilesToDirAction(this));
    /* Cria o menu */
    final JPopupMenu menu = new JPopupMenu();
    menu.add(dirCreateDirMenuItem);
    menu.add(dirRefreshDirMenuItem);
    menu.addSeparator();
    menu.add(dirCutFileMenuItem);
    menu.add(dirCopyFileMenuItem);
    menu.add(dirPasteFileMenuItem);
    menu.addSeparator();
    menu.add(dirRenameFileMenuItem);
    menu.add(dirDeleteFileMenuItem);
    menu.addSeparator();
    menu.add(dirUploadMenuItem);
    menu.add(dirExportMenuItem);
    return menu;
  }

  /**
   * Criao de menu de contexto para arquivo
   * 
   * @return o menu
   */
  protected JPopupMenu createPopupFile() {
    /* Cria os tens de menu */
    fileOpenMenuItem = new JMenuItem(new CommonFileOpenAction(this));
    notepadFileOpenMenuItem =
      new JMenuItem(new CommonFileOpenAction(this, "notepad",
        LNG.get("BasicProjectTree.open.notepad")));
    fileSeeFilePropertiesMenuItem =
      new JMenuItem(new CommonFilePropertiesShowAction(this));
    fileCutFileMenuItem = new JMenuItem(new CommonFileCutAction(this));
    fileCopyFileMenuItem = new JMenuItem(new CommonFileCopyAction(this));
    fileRenameFileMenuItem = new JMenuItem(new CommonFileRenameAction(this));
    fileChangeFileTypeMenuItem =
      new JMenuItem(new CommonFileTypeChangeAction(this));
    fileDeleteFileMenuItem = new JMenuItem(new CommonFileDeleteAction(this));
    fileDownloadFileMenuItem =
      new JMenuItem(new CommonFileDownloadAction(this));
    fileExportFileMenuItem = new JMenuItem(new SingleFileExportAction(this));
    advancedSubmenu = new JMenu(LNG.get("BasicProjectTree.popupmenu.advanced"));
    fileChangeStateFileMenuItem =
      new JMenuItem(new CommonFileChangeStateAction(this));
    advancedSubmenu.add(fileChangeStateFileMenuItem);

    /* Cria o menu */
    final JPopupMenu menu = new JPopupMenu();
    menu.add(fileOpenMenuItem);
    menu.add(notepadFileOpenMenuItem);
    menu.addSeparator();
    menu.add(fileSeeFilePropertiesMenuItem);
    menu.addSeparator();
    menu.add(fileCutFileMenuItem);
    menu.add(fileCopyFileMenuItem);
    menu.addSeparator();
    menu.add(fileRenameFileMenuItem);
    menu.add(fileChangeFileTypeMenuItem);
    menu.add(fileDeleteFileMenuItem);
    menu.addSeparator();
    menu.add(fileDownloadFileMenuItem);
    menu.add(fileExportFileMenuItem);
    menu.addSeparator();
    menu.add(advancedSubmenu);
    return menu;
  }

  /**
   * Criao de menu de contexto para seleo mltipla
   * 
   * @return o menu
   */
  protected JPopupMenu createPopupMultSelection() {
    /* Cria os tens de menu */
    msCutFileMenuItem = new JMenuItem(new CommonFileCutAction(this));
    msCopyFileMenuItem = new JMenuItem(new CommonFileCopyAction(this));
    msDeleteFileMenuItem = new JMenuItem(new CommonFileDeleteAction(this));
    msChangeFileTypeMenuItem =
      new JMenuItem(new CommonMultiSelectionFileTypeChangeAction(this));
    msExportFileMenuItem = new JMenuItem(new ExportFilesToDirAction(this));
    /* Cria o menu */
    final JPopupMenu menu = new JPopupMenu();
    menu.add(msCutFileMenuItem);
    menu.add(msCopyFileMenuItem);
    menu.addSeparator();
    menu.add(msChangeFileTypeMenuItem);
    menu.add(msDeleteFileMenuItem);
    menu.addSeparator();
    menu.add(msExportFileMenuItem);
    return menu;
  }

  /**
   * Exibe o contedo do diretrio/arquivo selecionado pelo usurio no painel de
   * detalhes. S atualiza a tabela se o arquivo selecionado mudar.
   * 
   * @param file diretrio selecionado pelo usurio
   */
  public void display(final ClientProjectFile file) {
    if (file.isDirectory()) {
      displayDir(file);
    }
    else {
      displayFile(file);
    }
    statusBar.setText(lastStatusBarText);
  }

  /**
   * Exibe na status dados dos arquivso selecionados.
   * 
   * @param files lista de arquivos
   */
  public void display(final ClientProjectFile[] files) {
    final List<ClientProjectFile> fileList = new ArrayList<ClientProjectFile>();
    for (final ClientProjectFile file : files) {
      fileList.add(file);
    }
    updateModel(fileList);
    lastStatusBarText = getStatusBarTextForFiles(files);
    statusBar.setText(lastStatusBarText);
  }

  /**
   * Exibe o diretrio com o ".." e os arquivos do diretrio dir dado.
   * 
   * @param dir diretrio
   */
  private void displayDir(final ClientProjectFile dir) {
    final ArrayList<ClientProjectFile> dirList =
      new ArrayList<ClientProjectFile>();
    final ClientProjectFile[] children = GetChildrenTask.runTask(dir);
    final NoHiddenFileFilter noDotFileFilter = NoHiddenFileFilter.getInstance();
    final boolean showHiddenFiles =
      DesktopFrame.getInstance().shouldShowHiddenFiles();
    long size = 0L;
    for (final ClientProjectFile element : children) {
      final ClientProjectFile child = element;
      if (!showHiddenFiles && !noDotFileFilter.accept(child)) {
        continue;
      }
      if (filter == null || filter.accept(child)) {
        dirList.add(child);
        if (!child.isDirectory()) {
          size += child.size();
        }
      }
    }
    updateModel(dirList);
    lastStatusBarText = getStatusBarTextForFiles(dirList.size(), size);
  }

  /**
   * Exibe dado (na status) do arquivo
   * 
   * @param file o arquivo
   */
  private void displayFile(final ClientProjectFile file) {
    final List<ClientProjectFile> fileList = new ArrayList<ClientProjectFile>();
    fileList.add(file);
    updateModel(fileList);
    lastStatusBarText = getStatusBarTextForFile(file);
  }

  /**
   * Habilita/desabilita a remoo de arquivos usando a tecla DELETE.
   * 
   * @param enable se <code>true</code>, habilita; caso contrrio, desabilita.
   */
  @Override
  public void enableDeleteAction(final boolean enable) {
    final KeyStroke delKey = KeyStroke.getKeyStroke("DELETE");
    if (enable) {
      final Action deleteAction = new CommonFileDeleteAction(this);
      final String actionName = (String) deleteAction.getValue(Action.NAME);
      dirContentsTable.getInputMap().put(delKey, actionName);
      dirContentsTable.getActionMap().put(actionName, deleteAction);
    }
    else {
      final String actionName =
        (String) dirContentsTable.getInputMap().get(delKey);
      dirContentsTable.getActionMap().remove(actionName);
      dirContentsTable.getInputMap().remove(delKey);
    }
    deleteEnabled = enable;
  }

  /**
   * Habilita/desabilita a atualizao da rvore de projetos usando a tecla F5.
   * 
   * @param enable se <code>true</code>, habilita; caso contrrio, desabilita.
   */
  @Override
  public void enableRefreshAction(final boolean enable) {
    final KeyStroke refreshKey = KeyStroke.getKeyStroke("F5");
    if (enable) {
      final Action refreshAction = new ProjectTreeUpdateAction(projectTree);
      final String actionName = (String) refreshAction.getValue(Action.NAME);
      dirContentsTable.getInputMap().put(refreshKey, actionName);
      dirContentsTable.getActionMap().put(actionName, refreshAction);
    }
    else {
      final String actionName =
        (String) dirContentsTable.getInputMap().get(refreshKey);
      dirContentsTable.getActionMap().remove(actionName);
      dirContentsTable.getInputMap().remove(refreshKey);
    }
    refreshEnabled = enable;
  }

  /**
   * Habilita/desabilita edio dos nomes dos arquivos.
   * 
   * @param enable <code>true</code> habilitar a edio, <code>false</code>
   *        desabilita.
   */
  @Override
  public void enableRenameAction(final boolean enable) {
    final TableColumnModel columnModel = dirContentsTable.getColumnModel();
    final TableColumn column =
      columnModel.getColumn(DirectoryContentsPanel.NAME_COL_INDEX);
    if (enable) {
      column.setCellEditor(new DirectoryContentsCellEditor(ownerWindow,
        ownerTitle));
    }
    else {
      column.setCellEditor(null);
    }
    renameEnabled = enable;
  }

  /**
   * Habilita/desabilita a transferncia de arquivos por drag&drop e clipboard.
   * 
   * @param enable true, para habilitar a transferncia, ou false, caso
   *        contrrio.
   */
  @Override
  public void enableTransferAction(final boolean enable) {
    if (enable) {
      dirContentsTable.setDragEnabled(true);
      dirContentsTable.setTransferHandler(new DirectoryContentsTransferHandler(
        ownerWindow, ownerTitle));
    }
    else {
      dirContentsTable.setDragEnabled(false);
      dirContentsTable.setTransferHandler(null);
    }
    transferEnabled = enable;
    enableTransferKeyBindings(enable);
  }

  /**
   * Habilita/desabilita o mapeamento de teclas de atalho para transferncia de
   * arquivos (Ctrl-X para Recortar, Ctrl-C para Copiar e Ctrl-V para Colar).
   * 
   * @param enable true, para habilitar a transferncia, ou false, caso
   *        contrrio.
   */
  private void enableTransferKeyBindings(final boolean enable) {
    final KeyStroke cutKey = KeyStroke.getKeyStroke("ctrl X");
    final KeyStroke copyKey = KeyStroke.getKeyStroke("ctrl C");
    final KeyStroke pasteKey = KeyStroke.getKeyStroke("ctrl V");
    final InputMap inputMap = dirContentsTable.getInputMap();
    final ActionMap actionMap = dirContentsTable.getActionMap();
    if (enable) {
      final Action cutAction = new CommonFileCutAction(this);
      final Action copyAction = new CommonFileCopyAction(this);
      final Action pasteAction = new CommonFilePasteAction(this);
      inputMap.put(cutKey, cutAction.getValue(Action.NAME));
      inputMap.put(copyKey, copyAction.getValue(Action.NAME));
      inputMap.put(pasteKey, pasteAction.getValue(Action.NAME));
      actionMap.put(cutAction.getValue(Action.NAME), cutAction);
      actionMap.put(copyAction.getValue(Action.NAME), copyAction);
      actionMap.put(pasteAction.getValue(Action.NAME), pasteAction);
    }
    else {
      String actionName = (String) inputMap.get(cutKey);
      actionMap.remove(actionName);
      inputMap.remove(cutKey);
      actionName = (String) inputMap.get(copyKey);
      actionMap.remove(actionName);
      inputMap.remove(copyKey);
      actionName = (String) inputMap.get(pasteKey);
      actionMap.remove(actionName);
      inputMap.remove(pasteKey);
    }
  }

  /**
   * Consulta o ndice de arquivo
   * 
   * @return o ndice
   */
  public int getFileColIndex() {
    return DirectoryContentsPanel.NAME_COL_INDEX;
  }

  /**
   * Retorna o menu de diretrios
   * 
   * @return o menu
   */
  public JPopupMenu getPopupDir() {
    return popupDir;
  }

  /**
   * Retorna o menu de arquivo
   * 
   * @return o menu
   */
  public JPopupMenu getPopupFile() {
    return popupFile;
  }

  /**
   * Retorna o menu de seleo mltipla
   * 
   * @return o menu
   */
  public JPopupMenu getPopupMultSelection() {
    return popupMultSelection;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommonClientProject getProject() {
    return this.projectTree.getProject();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile getSelectedFile() {
    final int selectedRow = dirContentsTable.getSelectedRow();
    return (ClientProjectFile) dirContentsTable.getValueAt(selectedRow,
      DirectoryContentsPanel.NAME_COL_INDEX);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile[] getSelectedFiles() {
    final int[] selectedRows = dirContentsTable.getSelectedRows();
    if (selectedRows == null || selectedRows.length == 0) {
      return null;
    }
    final ClientProjectFile[] selectedFiles =
      new ClientProjectFile[selectedRows.length];
    for (int i = 0; i < selectedRows.length; i++) {
      selectedFiles[i] =
        (ClientProjectFile) dirContentsTable.getValueAt(selectedRows[i],
          DirectoryContentsPanel.NAME_COL_INDEX);
    }
    return selectedFiles;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getSelectionCount() {
    final int[] selectedRows = dirContentsTable.getSelectedRows();
    if (selectedRows == null) {
      return 0;
    }
    return selectedRows.length;
  }

  /**
   * Consulta string de seleo de arquivo
   * 
   * @param file o arquivo
   * @return o texto
   */
  private String getStatusBarTextForFile(final ClientProjectFile file) {
    final long fileSize = file.size();
    final String formattedFileSize = FormatUtils.formatSize(fileSize, 2);
    return MessageFormat.format(LNG.get("DIR_CONTENTS_STATUS_BAR"), 1,
      formattedFileSize);
  }

  /**
   * Consulta string de seleo de mltiplos arquivos
   * 
   * @param files a lista de arquivos
   * @return o texto
   */
  private String getStatusBarTextForFiles(final ClientProjectFile[] files) {
    final int fileCount = files.length;

    // Calcula o volume total do diretrio para exibio na barra de status
    try {
      long dirSize = 0;
      for (int i = 0; i < fileCount; i++) {
        final ClientProjectFile file = files[i];
        if (!file.isDirectory()) {
          dirSize += file.size();
        }
      }
      return getStatusBarTextForFiles(fileCount, dirSize);
    }
    catch (final Exception e) {
      e.printStackTrace();
      return LNG.get("DIR_CONTENTS_STATUS_BAR_ERROR");
    }
  }

  /**
   * Obtm o texto para a status bar para um determinado nmero de arquivos
   * totalizando um determinado tamanho.
   * 
   * @param fileCount nmero de arquivos
   * @param totalSize tamanho acumulado dos arquivos
   * @return texto para a status bar
   */
  private String getStatusBarTextForFiles(final int fileCount,
    final long totalSize) {
    final String formattedDirSize = FormatUtils.formatSize(totalSize, 2);
    return MessageFormat.format(LNG.get("DIR_CONTENTS_STATUS_BAR"), fileCount,
      formattedDirSize);
  }

  /**
   * Retorna a tabela da listagem
   * 
   * @return a tabela
   */
  public SortableTable getTable() {
    return dirContentsTable;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTitle() {
    return ownerTitle;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Window getWindow() {
    return ownerWindow;
  }

  /**
   * Consulta se h diretrios em uma lista de "arquivos"
   * 
   * @param projectFiles a lista de "arquivos"
   * @return indicativo
   */
  public boolean hasDirectories(final ClientProjectFile[] projectFiles) {
    for (final ClientProjectFile projectFile : projectFiles) {
      if (projectFile.isDirectory()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Indica se existem arquivos na rvore recortados.
   * 
   * @return indicativo
   */
  public boolean hasFileToPaste() {
    final DirectoryContentsTransferHandler transferHandler =
      (DirectoryContentsTransferHandler) dirContentsTable.getTransferHandler();
    return transferHandler != null
      && transferHandler.hasTransferableInClipboard();
  }

  /**
   * <p>
   * Atribui o filtro para a tabela de arquivos deste painel.
   * </p>
   * ATENO:  importante que este mtodo seja chamado antes da rvore ser
   * filtrada, pois ela ir imediatamente notificar esta classe para se
   * redesenhar, e no momento do redesenho o filtro j dever estar atribudo.
   * 
   * @param filter novo filtro.
   */
  public void setFilter(final ProjectFileFilter filter) {
    this.filter = filter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startCopyAction() {
    final TransferHandler handler = dirContentsTable.getTransferHandler();
    handler.exportToClipboard(dirContentsTable, null, TransferHandler.COPY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startCutAction() {
    final TransferHandler handler = dirContentsTable.getTransferHandler();
    handler.exportToClipboard(dirContentsTable, null, TransferHandler.MOVE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startPasteAction() {
    final TransferHandler handler = dirContentsTable.getTransferHandler();
    handler.importData(dirContentsTable, null);
    if (projectTree.getSelectionCount() == 1) {
      final ClientProjectFile selectedFile = projectTree.getSelectedFile();
      display(selectedFile);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startRenamingAction() {
    final int selectedRow = dirContentsTable.getSelectedRow();
    currentEditingAllowed = true;
    dirContentsTable.editCellAt(selectedRow,
      DirectoryContentsPanel.NAME_COL_INDEX);
    currentEditingAllowed = false;
  }

  /**
   * Sincroniza a seleo.
   * 
   * @param file o arquivo
   */
  public void synchronizeSelection(final ClientProjectFile file) {
    if (file == null) {
      return;
    }
    if (!file.equals(projectTree.getSelectedFile())) {
      projectTree.setSelectedFile(file);
    }
  }

  /**
   * Atualiza os arquivos selecionados.
   * 
   * @see #update(ClientProjectFile[], boolean)
   */
  private void update() {
    this.update(DirectoryContentsPanel.this.projectTree.getSelectedFiles(),
      true);
  }

  /**
   * Atualiza o contedo da tabela se a seleo mudou ou se recebeu um evento da
   * rvore sobre alterao do modelo.
   * 
   * @param selectedFiles a seleo
   * @param refresh indicativo de atualizao
   */
  private void update(final ClientProjectFile[] selectedFiles,
    final boolean refresh) {
    if (selectedFiles == null || selectedFiles.length == 0) {
      this.clear();
      previousSelectedFile = null;
    }
    else {
      if (selectedFiles.length == 1) {
        final ClientProjectFile selFile = selectedFiles[0];
        if (refresh || !selFile.equals(previousSelectedFile)) {
          display(selFile);
          previousSelectedFile = selFile;
        }
      }
      else {
        display(selectedFiles);
        previousSelectedFile = null;
      }
    }
  }

  /**
   * Atualiza o modelo da tabela, na EDT.
   * 
   * @param fileList lista de linhas
   */
  private void updateModel(final List<ClientProjectFile> fileList) {
    SwingThreadDispatcher.invokeLater(new Runnable() {

      @Override
      public void run() {
        dirModel.setRows(fileList);
      }
    });
  }

  /**
   * Construtor.
   * 
   * @param ownerWindow janela me
   * @param ownerTitle ttulo
   * @param projectTree tree
   */
  public DirectoryContentsPanel(final Window ownerWindow,
    final String ownerTitle, final ProjectTree projectTree) {
    this.ownerWindow = ownerWindow;
    this.ownerTitle = ownerTitle;
    this.filter = null;
    this.projectTree = projectTree;
    this.projectTree.addTreeModelListener(new TreeModelListener() {
      @Override
      public void treeNodesChanged(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void treeNodesInserted(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void treeNodesRemoved(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void treeStructureChanged(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }
    });
    this.projectTree
      .addTreeSelectionListener(new ProjectTreeSelectionListener() {
        @Override
        public void update(final ProjectTreeSelectionEvent event) {
          DirectoryContentsPanel.this.update(event.getSelectedFiles(), false);
        }
      });

    // Cria o formato da tabela
    final ObjectTableProvider provider = new MyObjectTableProvider();

    // Cria o modelo da tabela
    dirModel =
      new ObjectTableModel<ClientProjectFile>(
        new ArrayList<ClientProjectFile>(), provider) {
        @Override
        public boolean isCellEditable(final int rowIndex, final int columnIndex) {
          if (!DirectoryContentsPanel.this.currentEditingAllowed) {
            return false;
          }
          return columnIndex == DirectoryContentsPanel.NAME_COL_INDEX;
        }
      };
    dirContentsTable = new SortableTable(dirModel);
    dirContentsTable.setNoSortStateEnabled(true);

    // Seta o comparator para cone que representa o tipo do arquivo
    final FileTypeComparator iconComparator = new FileTypeComparator();
    dirContentsTable.setComparator(DirectoryContentsPanel.TYPE_COL_INDEX,
      iconComparator);

    // Seta o comparator para ClientProjectFile
    final ClientProjectFileComparator comparator =
      new ClientProjectFileComparator();
    dirContentsTable.setComparator(DirectoryContentsPanel.NAME_COL_INDEX,
      comparator);

    // Ordena por nome 
    dirContentsTable.sort(DirectoryContentsPanel.NAME_COL_INDEX,
      SortOrder.ASCENDING);

    // Renderizador da coluna "data"
    dirContentsTable
      .setDefaultRenderer(Date.class, new DateTableCellRenderer());

    // Renderizador da coluna "tamanho"
    dirContentsTable
      .setDefaultRenderer(Long.class, new SizeTableCellRenderer());

    // Renderizador da coluna "nome"
    final TableColumnModel columnModel = dirContentsTable.getColumnModel();
    final TableColumn nameColumn =
      columnModel.getColumn(DirectoryContentsPanel.NAME_COL_INDEX);
    nameColumn.setCellRenderer(new ProjectFileCellRenderer());

    // Renderizador da coluna de cones que representam o tipo do arquivo
    final TableColumn typeColumn =
      columnModel.getColumn(DirectoryContentsPanel.TYPE_COL_INDEX);
    typeColumn.setCellRenderer(new FileTypeIconCellRenderer());

    // Definio de tamanho fixo para a coluna de cones que representam o 
    // tipo do arquivo
    typeColumn.setMaxWidth(25);
    typeColumn.setMinWidth(25);
    typeColumn.setPreferredWidth(25);

    // Definio de tamanho fixo para a coluna de tamanho
    final TableColumn sizeColumn =
      columnModel.getColumn(DirectoryContentsPanel.SIZE_COL_INDEX);
    sizeColumn.setPreferredWidth(80);
    sizeColumn.setMaxWidth(80);

    // Definio de tamanho fixo para a coluna de data
    final TableColumn dateColumn =
      columnModel.getColumn(DirectoryContentsPanel.DATE_COL_INDEX);
    dateColumn.setPreferredWidth(150);
    dateColumn.setMaxWidth(150);

    // Remove as linhas da tabela
    dirContentsTable.setShowGrid(false);
    dirContentsTable.setRowMargin(0);
    //!CSBASE-1530 Remove temporariamente as funcionalidades do mouse.
    //    dirContentsTable.addMouseListener(new DirectoryContentsMouseAdapter(this));
    // dirContentsTable.keepUserMarginChanges(true);

    // Se o usurio selecionar um ou mais arquivos, a barra de status 
    // atualizada com o tamanho total do(s) arquivo(s) selecionado(s).

    //!CSBASE-1530 Remove temporariamente a funcionalidades de seleo da tabela.
    //    final ListSelectionModel selectModel = dirContentsTable.getSelectionModel();
    //    selectModel.addListSelectionListener(new ListSelectionListener() {
    //      @Override
    //      public void valueChanged(final ListSelectionEvent e) {
    //        if (dirContentsTable.getSelectedRowCount() > 0) {
    //          final int[] indexes = dirContentsTable.getSelectedRows();
    //          int i;
    //          long selectedFilesSize = 0;
    //          boolean isParentDirSelected = false;
    //          for (i = 0; i < indexes.length; i++) {
    //            final ClientProjectFile file =
    //              (ClientProjectFile) dirContentsTable.getValueAt(indexes[i],
    //                DirectoryContentsPanel.NAME_COL_INDEX);
    //            if (!file.isDirectory()) {
    //              selectedFilesSize += file.size();
    //            }
    //            if (file.isParentDir()) {
    //              isParentDirSelected = true;
    //            }
    //          }
    //          if (isParentDirSelected) {
    //            i--;
    //          }
    //          final String formattedSelectedFilesSize =
    //            FormatUtils.formatSize(selectedFilesSize, 2);
    //          lastStatusBarText =
    //            MessageFormat.format(LNG.get("DIR_CONTENTS_STATUS_BAR"), i,
    //              formattedSelectedFilesSize);
    //        }
    //        statusBar.setText(lastStatusBarText);
    //      }
    //    });
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(final CommonClientProject project) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void projectClosed(final CommonClientProject project) {
        /*
         * se o projeto fechado for o que estamos visualizando, no precisamos
         * atualizar o painel, basta o clear()
         */
        DirectoryContentsPanel.this.clear();
      }

      @Override
      public void projectInfoModified() {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void projectRemoved() {
        DirectoryContentsPanel.this.clear();
      }
    });
    scrollPane = new JScrollPane(dirContentsTable);
    scrollPane.getViewport().setBackground(Color.WHITE);
    statusBar = new JLabel();
    statusBar.setText(lastStatusBarText);
    setLayout(new BorderLayout());
    add(scrollPane, BorderLayout.CENTER);
    add(statusBar, BorderLayout.SOUTH);
    setPreferredSize(new Dimension(220, 100));

    // Criao dos menus de popup
    popupDir = createPopupDir();
    popupFile = createPopupFile();
    popupMultSelection = createPopupMultSelection();
    // Comportamento padro; os sistemas que desejarem podem desabilitar os
    // comportamentos conforme a necessidade chamando os mtodos abaixo com
    // parmetro "false".
    //!CSBASE-1530 Desabilita temporariamente qualquer ao da tabela.
    //    enableDeleteAction(true);
    //    enableRefreshAction(true);
    //    enableRenameAction(true);
    //    enableTransferAction(true);
  }

}
