package csbase.client.algorithms.parameters;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.ClientLocalFile;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.filechooser.ClientFileChooser;
import csbase.client.util.filechooser.ClientFileChooserSelectionMode;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil.OperationResult;
import csbase.client.util.filechooser.ClientSGAFileChooserUtil;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientSGAFile;
import csbase.logic.ProjectFileType;
import csbase.logic.SGASet;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.OutputURLListParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * Viso do parmetro lista de URLs de sada {@link OutputURLListParameter}.
 * 
 * @author Tecgraf
 */
public class OutputURLListParameterView extends URLListParameterView {

  /**
   * Construtor.
   * 
   * @param parameter parmetro (No aceita {@code null}).
   * @param mode modo de visualizao. No aceita {@code null}.
   */
  public OutputURLListParameterView(OutputURLListParameter parameter, Mode mode) {
    super(parameter, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputURLListParameter getParameter() {
    return (OutputURLListParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<FileURLValue> askForFile(URLProtocol fileProtocol, Object... args) {
    String fileType = getParameter().getFileType();
    List<String> fileTypes = new ArrayList<String>();
    if (fileType != null) {
      fileTypes.add(fileType);
    }
    ClientFileChooserSelectionMode selectionMode = getFileChooserMode();
    String title = getParameter().getLabel();
    switch (fileProtocol) {
      case PROJECT:
        return askForProjectFile(fileTypes, true, selectionMode, title);
      case LOCAL:
        return askForLocalFile(fileTypes, true, selectionMode, title);
      case SGA:
        String sgaName = (String) args[0];
        return askForSGAFile(fileTypes, true, selectionMode, title, sgaName);
    }
    return null;
  }

  /**
   * Abre um navegador de arquivos do projeto.
   * 
   * @param fileTypes os tipos de arquivo permitidos.
   * @param allowAll determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * 
   * @return a URL do arquivo selecionado.
   */
  private List<FileURLValue> askForProjectFile(List<String> fileTypes,
    boolean allowAll, ClientFileChooserSelectionMode selectionMode, String title) {
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        List<OperationResult> selectedDirs =
          ClientProjectFileChooserUtil.browseMultipleDirectoriesInOpenMode(
            getWindow(), title, allowAll);
        if (selectedDirs == null) {
          return null;
        }
        for (OperationResult selectedDir : selectedDirs) {
          ClientProjectFile dir = selectedDir.getClientProjectFile();
          urls.add(getURLFromFile(URLProtocol.PROJECT, dir, null));
        }
        break;
      case FILES_ONLY:
        List<OperationResult> selectedFiles =
          ClientProjectFileChooserUtil.browseMultipleFilesInOpenMode(
            getWindow(), fileTypes, title, allowAll);
        if (selectedFiles == null) {
          return null;
        }
        for (OperationResult selectedFile : selectedFiles) {
          ClientFile file = selectedFile.getClientProjectFile();
          urls.add(getURLFromFile(URLProtocol.PROJECT, file, null));
        }
        break;
      default:
        return null;
    }
    return urls;
  }

  /**
   * Abre um navegador de arquivos do cliente.
   * 
   * @param fileTypes os tipos de arquivo permitidos.
   * @param allowAll determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * 
   * @return a URL do arquivo selecionado.
   */
  private List<FileURLValue> askForLocalFile(List<String> fileTypes,
    boolean allowAll, ClientFileChooserSelectionMode selectionMode, String title) {
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        List<ClientLocalFile> selectedDirs =
          ClientLocalFileChooserUtil.browseMultipleDirectoriesInOpenMode(
            getWindow(), title, allowAll, ClientFileChooserUtils
              .getLocalHomeDirectory());
        if (selectedDirs == null) {
          return null;
        }
        for (ClientLocalFile selectedDir : selectedDirs) {
          urls.add(getURLFromFile(URLProtocol.LOCAL, selectedDir, null));
        }
        break;
      case FILES_ONLY:
        List<ClientLocalFile> selectedFiles =
          ClientLocalFileChooserUtil.browseMultipleFilesInOpenMode(getWindow(),
            fileTypes, title, allowAll, ClientFileChooserUtils
              .getLocalHomeDirectory());
        if (selectedFiles == null) {
          return null;
        }
        for (ClientLocalFile selectedFile : selectedFiles) {
          urls.add(getURLFromFile(URLProtocol.LOCAL, selectedFile, null));
        }
        break;
      default:
        return null;
    }
    return urls;
  }

  /**
   * Abre um navegador de arquivos do SGA.
   * 
   * @param fileTypes os tipos de arquivo permitidos.
   * @param allowAll determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * @param sgaName nome do SGA.
   * 
   * @return a URL do arquivo selecionado.
   */
  private List<FileURLValue> askForSGAFile(List<String> fileTypes,
    boolean allowAll, ClientFileChooserSelectionMode selectionMode,
    String title, final String sgaName) {
    if (sgaName == null) {
      return null;
    }

    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    Task<Boolean> sgaIsAliveTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        SGASet sga = sgaService.getSGASet(sgaName);
        setResult(sga.getAlive());
      }
    };
    boolean isAlive = false;
    String description =
      LNG.get(URLParameterView.class.getSimpleName() + ".sga_alive_task");
    if (sgaIsAliveTask.execute(getWindow(), title, description)) {
      isAlive = sgaIsAliveTask.getResult();
    }

    if (!isAlive) {
      String errorMessage =
        LNG.get(URLParameterView.class.getSimpleName() + ".sga_not_alive",
          new Object[] { sgaName });
      StandardErrorDialogs.showErrorDialog(getWindow(), errorMessage);
      return null;
    }

    ClientSGAFile root = new ClientSGAFile(sgaName, "/");
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        List<ClientSGAFile> selectedDirs =
          ClientSGAFileChooserUtil.browseMultipleDirectoryInOpenMode(
            getWindow(), title, fileTypes, root);
        if (selectedDirs == null) {
          return null;
        }
        for (ClientSGAFile selectedDir : selectedDirs) {
          urls.add(getURLFromFile(URLProtocol.SGA, selectedDir, sgaName));
        }
        break;
      case FILES_ONLY:
        List<ClientSGAFile> selectedFiles =
          ClientSGAFileChooserUtil.browseMultipleFilesInOpenMode(getWindow(),
            title, fileTypes, root);
        if (selectedFiles == null) {
          return null;
        }
        for (ClientSGAFile selectedFile : selectedFiles) {
          urls.add(getURLFromFile(URLProtocol.SGA, selectedFile, sgaName));
        }
        break;
      default:
        return null;
    }
    return urls;
  }

  /**
   * Obtm a URL a partir de um arquivo.
   * 
   * @param fileProtocol o protocolo de acesso ao arquivo.
   * @param file o arquivo.
   * @param host o host da URL.
   * @return URL.
   */
  private FileURLValue getURLFromFile(URLProtocol fileProtocol, ClientFile file,
    String host) {
    if (file == null) {
      return null;
    }
    String type;
    if (file.isDirectory()) {
      type = ProjectFileType.DIRECTORY_TYPE;
    }
    else {
      type = file.getType();
    }
    return new FileURLValue(file.getStringPath(), type, fileProtocol, host);
  }

  /**
   * Obtm o modo do {@link ClientFileChooser} que ser utilizado para solicitar
   * os arquivos ao usurio.
   * 
   * @return modo arquivo ou diretrio.
   */
  private ClientFileChooserSelectionMode getFileChooserMode() {
    FileParameterMode parameterMode = getParameter().getMode();
    switch (parameterMode) {
      case REGULAR_FILE:
        return ClientFileChooserSelectionMode.FILES_ONLY;
      case DIRECTORY:
        return ClientFileChooserSelectionMode.DIRECTORIES_ONLY;
      default:
        String errorMessage =
          MessageFormat.format("O modo {0} no  vlido.", parameterMode);
        throw new IllegalStateException(errorMessage);
    }
  }

}
