package csbase.client.algorithms.parameters;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.filechooser.ClientFileChooser;
import csbase.client.util.filechooser.ClientFileChooserSelectionMode;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil.OperationResult;
import csbase.client.util.filechooser.ClientSGAFileChooserUtil;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientSGAFile;
import csbase.logic.ProjectFileType;
import csbase.logic.SGASet;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * Viso do parmetro URL de sada {@link OutputURLParameter}.
 * 
 * @author Tecgraf
 */
public class OutputURLParameterView extends URLParameterView {

  /**
   * Construtor.
   * 
   * @param parameter parmetro (No aceita {@code null}).
   * @param mode modo de visualizao. No aceita {@code null}.
   */
  public OutputURLParameterView(OutputURLParameter parameter, Mode mode) {
    super(parameter, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputURLParameter getParameter() {
    return (OutputURLParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected FileURLValue askForFile(URLProtocol fileProtocol, Object... args) {
    String fileType = getParameter().getFileType();
    List<String> fileTypes = new ArrayList<String>();
    if (fileType != null) {
      fileTypes.add(fileType);
    }
    ClientFileChooserSelectionMode selectionMode = getFileChooserMode();
    String title = getParameter().getLabel();
    switch (fileProtocol) {
      case PROJECT:
        return askForProjectFile(fileTypes, true, selectionMode, title);
      case LOCAL:
        return askForLocalFile(fileTypes, true, selectionMode, title);
      case SGA:
        String sgaName = (String) args[0];
        return askForSGAFile(fileTypes, true, selectionMode, title, sgaName);
    }
    return null;
  }

  /**
   * Abre um navegador de arquivos do projeto.
   * 
   * @param fileTypes os tipos de arquivo permitidos.
   * @param allowAll determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * 
   * @return a URL do arquivo selecionado.
   */
  private FileURLValue askForProjectFile(List<String> fileTypes, boolean allowAll,
    ClientFileChooserSelectionMode selectionMode, String title) {
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        OperationResult dirResult =
          ClientProjectFileChooserUtil.browseSingleDirectoryInSaveMode(
            getWindow(), title, allowAll);
        if (dirResult != null) {
          ClientProjectFile dir = dirResult.getClientProjectFile();
          return getURLFromFile(URLProtocol.PROJECT, dir, null);
        }
        break;
      case FILES_ONLY:
        String fileType = null;
        if (fileTypes != null && !fileTypes.isEmpty()) {
          fileType = fileTypes.get(0);
        }
        OperationResult fileResult =
          ClientProjectFileChooserUtil.browseSingleFileInSaveMode(getWindow(),
            fileTypes, fileType, title, allowAll, true);
        if (fileResult != null) {
          ClientFile file = fileResult.getClientProjectFile();
          return getURLFromFile(URLProtocol.PROJECT, file, null);
        }
        break;
    }
    return null;
  }

  /**
   * Abre um navegador de arquivos do cliente.
   * 
   * @param fileTypes os tipos de arquivo permitidos.
   * @param allowAll determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * 
   * @return a URL do arquivo selecionado.
   */
  private FileURLValue askForLocalFile(List<String> fileTypes, boolean allowAll,
    ClientFileChooserSelectionMode selectionMode, String title) {
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        ClientFile localDir =
          ClientLocalFileChooserUtil.browseSingleDirectoryInSaveMode(
            getWindow(), fileTypes, null, title, allowAll,
            ClientFileChooserUtils.getLocalHomeDirectory());
        return getURLFromFile(URLProtocol.LOCAL, localDir, null);
      case FILES_ONLY:
        ClientFile localFile =
          ClientLocalFileChooserUtil.browseSingleFileInSaveMode(getWindow(),
            fileTypes, null, title, allowAll, ClientFileChooserUtils
              .getLocalHomeDirectory());
        return getURLFromFile(URLProtocol.LOCAL, localFile, null);
    }
    return null;
  }

  /**
   * Abre um navegador de arquivos do SGA.
   * 
   * @param fileTypes os tipos de arquivo permitidos.
   * @param allowAll determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * @param sgaName nome do SGA.
   * 
   * @return a URL do arquivo selecionado.
   */
  private FileURLValue askForSGAFile(List<String> fileTypes, boolean allowAll,
    ClientFileChooserSelectionMode selectionMode, String title,
    final String sgaName) {
    if (sgaName == null) {
      return null;
    }

    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    Task<Boolean> sgaIsAliveTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        SGASet sga = sgaService.getSGASet(sgaName);
        setResult(sga.getAlive());
      }
    };
    boolean isAlive = false;
    String description =
      LNG.get(URLParameterView.class.getSimpleName() + ".sga_alive_task");
    if (sgaIsAliveTask.execute(getWindow(), title, description)) {
      isAlive = sgaIsAliveTask.getResult();
    }

    if (!isAlive) {
      String errorMessage =
        LNG.get(URLParameterView.class.getSimpleName() + ".sga_not_alive",
          new Object[] { sgaName });
      StandardErrorDialogs.showErrorDialog(getWindow(), errorMessage);
      return null;
    }

    ClientSGAFile root = new ClientSGAFile(sgaName, "/");
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        ClientSGAFile dir =
          ClientSGAFileChooserUtil.browseSingleDirectoryInSaveMode(getWindow(),
            title, fileTypes, root);
        return getURLFromFile(URLProtocol.SGA, dir, sgaName);
      case FILES_ONLY:
        ClientSGAFile file =
          ClientSGAFileChooserUtil.browseSingleFileInSaveMode(getWindow(),
            title, fileTypes, root);
        return getURLFromFile(URLProtocol.SGA, file, sgaName);
      default:
        return null;
    }
  }

  /**
   * Obtm a URL a partir de um arquivo.
   * 
   * @param fileProtocol o protocolo de acesso ao arquivo.
   * @param file o arquivo.
   * @param host o host da URL.
   * @return URL.
   */
  private FileURLValue getURLFromFile(URLProtocol fileProtocol, ClientFile file,
    String host) {
    if (file == null) {
      return null;
    }
    String type;
    if (file.isDirectory()) {
      type = ProjectFileType.DIRECTORY_TYPE;
    }
    else {
      type = file.getType();
    }
    return new FileURLValue(file.getStringPath(), type, fileProtocol, host);
  }

  /**
   * Obtm o modo do {@link ClientFileChooser} que ser utilizado para solicitar
   * os arquivos ao usurio.
   * 
   * @return modo arquivo ou diretrio.
   */
  private ClientFileChooserSelectionMode getFileChooserMode() {
    FileParameterMode parameterMode = getParameter().getMode();
    switch (parameterMode) {
      case REGULAR_FILE:
        return ClientFileChooserSelectionMode.FILES_ONLY;
      case DIRECTORY:
        return ClientFileChooserSelectionMode.DIRECTORIES_ONLY;
      default:
        String errorMessage =
          MessageFormat.format("O modo {0} no  vlido.", parameterMode);
        throw new IllegalStateException(errorMessage);
    }
  }

}
