package csbase.client.algorithms.parameters;

import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JComponent;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.algorithms.tasks.ViewValidationTask;
import csbase.client.algorithms.validation.ValidationTranslator;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.ParseException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.parameters.ParameterLoader;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parsers.SimpleAlgorithmParser;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * Viso (boto) do carregador de parmetros. O carregador de parmetros executa
 * um algoritmo a partir da janela de um configurador. O algoritmo executado,
 * chamado Extrator, pode ter como entrada alguns dos parmetros definidos neste
 * configurador e tem como sada os valores de alguns parmetros deste
 * configurador.
 */
public class ParameterLoaderView implements ViewValidator {
  /** Boto que representa o carregador de parmtros */
  private JButton button;
  /** Carregador de parmetros. */
  private ParameterLoader parameterLoader;
  /** Viso do grupo */
  private ParameterGroupView groupView;
  /** Arquivo de sada temporrio, gerado pelo algoritmo extrator. */
  private String outputExtractFileName;
  /** Parmetros usados na execuo do algoritmo extrator. */
  private Map<String, String> extractParametersValues;

  /**
   * Construtor.
   * 
   * @param groupView Viso do grupo
   * @param parameterLoader Carregador de parmetros
   */
  public ParameterLoaderView(ParameterGroupView groupView,
    ParameterLoader parameterLoader) {
    this.parameterLoader = parameterLoader;
    this.groupView = groupView;
    this.extractParametersValues = new HashMap<String, String>();
    this.button = new JButton(LNG.get("ParameterLoaderView.button"));
    this.button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent event) {
        // O grupo passa a validar o carregador tambm, por isso, temos que
        // preencher os parmetros do extrator antes de chamar o validador.
        loadParameterValues();
        if (groupIsValid(ValidationMode.ALLOW_EMPY_VALUES, true)) {
          // Monta e executa o comando
          submitExtractCommand();
        }
      }
    });
  }

  /**
   * <p>
   * Validamos se o carregador de parmetros foi executado com as entradas que
   * esto selecionadas.
   * </p>
   * <p>
   * No basta apenas comparar o valor passado para o
   * {@link ParameterLoaderView} ao ser requisitada a extrao dos dados, com o
   * valor durante a execuo. O contedo do
   * {@link ParameterLoaderView#extractParametersValues mapa dos parmetros} do
   * algoritmo extrator no necessariamente vai bater com a
   * {@link ParameterLoader#getInputParameterNames() lista dos parmetros de
   * entrada} do {@link ParameterLoader modelo}.<br>
   * Para facilitar a explicao abaixo de como  feita a validao, ser
   * utilizada a palavra VISO para se referir ao
   * {@link ParameterLoaderView#extractParametersValues mapa dos parmetros do
   * algoritmo extrator} presente na {@link ParameterLoaderView viso} e a
   * palavra MODELO para indicar a
   * {@link ParameterLoader#getInputParameterNames() lista dos parmetros de
   * entrada} do {@link ParameterLoader modelo}.
   * </p>
   * <p>
   * Abaixo pode ser visto como fica o estado de cada a cada momento.
   * <ol>
   * <li><u>Momento: edio dos parmetros do comando.</u><br>
   * <br>
   * Condio: inicial.
   * <ul>
   * <li>VISO -> vazio</li>
   * <li>MODELO -> tem o nome dos parmetros de entrada.</li>
   * </ul>
   * <br>
   * Condio: aps os valores serem carregados pelo mtodo
   * {@link ParameterLoaderView#loadParameterValues()}.
   * <ul>
   * <li>VISO -> tem os parmetros de entrada.</li>
   * <li>MODELO -> tem o nome dos parmetros de entrada.</li>
   * </ul>
   * <br>
   * Condio: aps o comando de extrao ser submetido atravs do mtodo
   * {@link ParameterLoaderView#submitExtractCommand()}. <br>
   * A VISO ser usada como mapa de parmetros do algoritmo extrator. Devido a
   * isso, o parmetro de sada do algoritmo extrator ser includo nela.
   * <ul>
   * <li>VISO -> tem os parmetros de entrada mais o parmetro de sada.</li>
   * <li>MODELO -> tem o nome dos parmetros de entrada.</li>
   * </ul>
   * </li>
   * <br>
   * <li><u>Momento: executando a partir de um fluxo.</u><br>
   * <br>
   * Condio: inicial<br>
   * Ao ser requisitada uma execuo de fluxo, os valores dos parmetros so
   * previamente salvos, o modelo  recriado com novas instncias de
   * configuradores para cada n e os valores so atribudos aos parmetros
   * destas novas instncias. S ai o comando  executado. O MODELO faz parte do
   * configurador e a VISO da viso do configurador. Como o configurador e sua
   * viso so recriados e somente os dados do primeiro so salvos, a VISO
   * perde todos os seus dados. Ela ir permanecer assim por que o boto de
   * carregar nunca ser acionado durante o processo de execuo.
   * <ul>
   * <li>VISO -> vazio.</li>
   * <li>MODELO -> tem o nome dos parmetros de entrada.</li>
   * </ul>
   * </li>
   * <br>
   * <li><u>Momento: executando como um comando simples.</u><br>
   * <br>
   * Condio: inicial<br>
   * O mesmo configurador e viso utilizado no processo de edio de parmetros
   * ser utilizado na execuo de um comando simples. Sendo assim, o estado
   * final da VISO e do MODELO da etapa anterior, ser o estado inicial deles
   * na etapa de execuo.
   * <ul>
   * <li>VISO -> tem os parmetros de entrada mais o parmetro.</li>
   * <li>MODELO -> tem o nome dos parmetros de entrada.</li>
   * </ul>
   * </li>
   * </ol>
   * </p>
   * <p>
   * Devido a estas discrepncias entre o MODELO e a VISO, para ser aceito como
   * vlida uma instncia de um {@link ParameterLoaderView}, todos os parmetros
   * presentes na VISO que estiverem sendo referenciados pelo MODELO, devem ser
   * iguais aos valores do configurador no momento da validao. Assim, quando a
   * VISO tiver o parmetro de sada que no aparece no MODELO, este ser
   * ignorado. Quando o MODELO tiver parmetros e a VISO no eles s podem ter
   * sido inseridos atravs do mtodo
   * {@link ParameterLoader#addInputParameterName(String)}. Sendo assim no h
   * como saber qual o valor que eles tinham durante a extrao dos dados.
   * Supe-se que eles foram adicionados durante a execuo de um fluxo que
   * reconstruiu o modelo e por isso estariam corretos. Somente pode-se garantir
   * os parmetros existentes tanto na VISO quanto no MODELO e estes quando
   * existirem  que devem ser validados.
   * </p>
   * 
   * @return O resultado da validao.
   */
  @Override
  public ViewValidationResult validate(ValidationMode mode) {

    // Se o extrator estiver em execuo, no valida.
    if (getComponent().isEnabled()) {

      boolean isValid = true;

      if (parameterLoader.isInputValidationEnabled()) {
        // Obtm os parmetros que so usados pelo algoritmo extrator.
        Map<String, String> allParametersValues =
          getConfigurator().getParameterValuesByName();

        for (String parameterName : extractParametersValues.keySet()) {
          if (!parameterLoader.getInputParameterNames().contains(parameterName)) {
            continue;
          }
          String parameterValue = allParametersValues.get(parameterName);
          String extractorValue = extractParametersValues.get(parameterName);
          if (extractorValue == null || !extractorValue.equals(parameterValue)) {
            isValid = false;
            break;
          }
        }
      }

      if (isValid) {
        return new ViewValidationResult(this);
      }
    }

    LocalizedMessage message =
      new LocalizedMessage(ParameterLoaderView.class, "error.validationError");
    ViewValidationResult result = new ViewValidationResult(message, this);
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean highlightValidationResult(ViewValidationResult result) {
    if (!result.isWellSucceded()) {
      Toolkit.getDefaultToolkit().beep();
      String message = ValidationTranslator.translateMessage(result);
      StandardErrorDialogs.showErrorDialog(groupView.getWindow(),
        LNG.get("ParameterLoaderView.error.title"), message);
      getComponent().requestFocus();
    }
    return true;
  }

  /**
   * Obtm os parmetros que sero enviados para o algoritmo extrator.
   */
  private void loadParameterValues() {
    clearParameterValues();
    Map<String, String> allParametersValues =
      getConfigurator().getParameterValuesByName();
    Set<String> parametersNames = parameterLoader.getInputParameterNames();
    for (String parameterName : parametersNames) {
      String parameterValue = allParametersValues.get(parameterName);
      extractParametersValues.put(parameterName, parameterValue);
    }
  }

  /**
   * Validamos os parmetros do grupo do carregador de parmetros antes de
   * executar o algoritmo extrator.
   * 
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   * @param complain indica se deve exibir um dilogo em caso de erro.
   * 
   * @return O resultado da validao.
   */
  private boolean groupIsValid(ValidationMode mode, boolean complain) {
    Validation validation =
      ViewValidationTask.runTask(groupView.getWindow(), groupView, mode,
        complain);
    if (validation.isWellSucceded()) {
      return true;
    }
    clearParameterValues();
    return false;
  }

  /**
   * Obtm o configurador do algoritmo extrator de parmetros e executa o
   * extrator.
   */
  private void submitExtractCommand() {
    /**
     * O retorno de {@link ParameterGroupView#getWindow() no tem perigo de ser
     * {@code null} neste ponto, por que este mtodo  chamado atravs de uma
     * ao do usurio em cima do boto de carregar parmetros. Sendo assim, o
     * boto deve estar visvel em uma janela.
     */
    Window window = groupView.getWindow();
    // Desabilita o componente at que a notificao de fim da execuo
    // seja recebida.
    setEnabled(false);
    outputExtractFileName = "outputExtract_" + System.currentTimeMillis();
    ParameterLoaderCommandExecutionTask task =
      new ParameterLoaderCommandExecutionTask(window, this);
    if (task.execute(window, LNG.get("ParameterLoaderView.executing.title"),
      LNG.get("ParameterLoaderView.executing.msg"))) {
      if (task.getResult()) {
        // Exibe os dados extrados
        extractParameter();
      }
      else {
        // Exibe o dilogo de falha de execuo
        StandardErrorDialogs.showErrorDialog(window, LNG
          .get("ParameterLoaderView.executing.title"), String.format(
          LNG.get("ParameterLoaderView.executing.error"),
          "Extrator de Parmetros"));
        clearParameterValues();
      }
    }
    else {
      clearParameterValues();
    }
    // Habilita o componente.
    setEnabled(true);
  }

  /**
   * Obtm o configurador do algoritmo.
   * 
   * @return o configurador do algoritmo
   */
  private AlgorithmConfigurator getConfigurator() {
    return groupView.getConfigurator();
  }

  /**
   * Exibe mensagem de erro com exceo.
   * 
   * @param msg Mensagem de erro.
   * 
   * @param e exceo
   */
  private void showErrorDialog(String msg, Exception e) {
    StandardErrorDialogs.showErrorDialog(groupView.getWindow(),
      LNG.get("ParameterLoaderView.error.title"), msg, e);
  }

  /**
   * Apaga os parmetros usados na execuo do extrator.
   */
  private void clearParameterValues() {
    extractParametersValues.clear();
  }

  /**
   * Carrega os parmetros no configurador. Este mtodo  chamado aps a
   * execuo do Extrator. O Extrator coloca os valores extrados em um arquivo
   * de sada. Este arquivo  lido e seus valores so colocados no configurador
   * que chamou o extrator.
   */
  private void extractParameter() {
    SimpleAlgorithmConfigurator configurator =
      (SimpleAlgorithmConfigurator) getConfigurator();
    SimpleAlgorithmParser parser = new SimpleAlgorithmParser();
    ParameterLoaderTask task =
      new ParameterLoaderTask(groupView.getWindow(), this);
    if (!task.execute(groupView.getWindow(),
      LNG.get("ParameterLoaderView.loading.title"),
      LNG.get("ParameterLoaderView.loading.msg"))) {
      return;
    }
    String fileContents = task.getResult();
    try {
      parser.loadModifications(configurator, new StringReader(fileContents));
    }
    catch (ParseException pe) {
      showErrorDialog(LNG.get("ParameterLoaderView.error.parseError"), pe);
    }
  }

  /**
   * Habilita/desabilita o componente da viso.
   * 
   * @param enabled que indica se habilita a viso ou no.
   */
  private void setEnabled(boolean enabled) {
    button.setEnabled(enabled);
  }

  /**
   * Obtm o componente da viso.
   * 
   * @return o componente da viso
   */
  public JComponent getComponent() {
    return button;
  }

  /**
   * Obtm os parmetros usados na execuo do algoritmo extrator.
   * 
   * @return .
   */
  public Map<String, String> getExtractParametersValues() {
    return extractParametersValues;
  }

  /**
   * Obtm o nome do arquivo de sada gerado pelo Extrator.
   * 
   * @return o nome do arquivo de sada gerado pelo Extrator.
   */
  public String getOutputExtractFileName() {
    return outputExtractFileName;
  }

  /**
   * Obtm o carregador de parmetros
   * 
   * @return .
   */
  public ParameterLoader getParameterLoader() {
    return parameterLoader;
  }
}
