package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.Application;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.versiontree.VersionNodeInterface;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTreeNode;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTreeRootNode;
import csbase.logic.algorithms.AlgorithmInfo;

/**
 * Representa uma viso da rvore de verses de um algoritmo e seus respectivos
 * filhos. Para inserir essa viso grfica voc deve obter o painel
 * correspondente.
 * 
 */
public class VersionTreeView {

  /** Painel que criou essa viso de rvore de verses */
  private AlgorithmVersionInfoPanel versionInfoPanel;

  /** rvore com as verses do algoritmo */
  private VersionTree versionTree;

  /** rvore de dados, que pode exibir ou no os filhos dos ns principais */
  private JPanel versionTreePanel;

  /**
   * Indica que as informaes j foram atualizadas. Foi necessrio para tratar
   * os casos de seleo de um item da JList em que o evento de mudana chega,
   * mas  somente para o item aparecer selecionado e no precisa inicializar
   * esse elemento
   */
  private boolean infoAlreadyUpdated;

  /**
   * Linha do n de verso correntemente selecionado na rvore
   */
  private int currentVersionRow;

  /** Nome da raiz da rvore de verses do algoritmo */
  public static String ROOT_NAME;

  /**
   * Constri o painel da rvore de dados.
   * 
   * @param versionInfoPanel painel que criou essa viso
   * @param algorithmInfo informaes do algoritmo selecionado
   */
  public VersionTreeView(final AlgorithmVersionInfoPanel versionInfoPanel,
    AlgorithmInfo algorithmInfo) {
    this.versionInfoPanel = versionInfoPanel;
    currentVersionRow = -1;
    ROOT_NAME = getApplication().getString(
      "VersionTreeView.root.label.versions");
    getVersionTree().getSelectionModel().addTreeSelectionListener(
      new TreeSelectionListener() {
        @Override
        public void valueChanged(TreeSelectionEvent e) {
          Object node = e.getPath().getLastPathComponent();
          TreeNode treeNode = (TreeNode) node;
          initVersionSelection(treeNode);
        }
      });
  }

  /**
   * Obtm a aplicao.
   * 
   * @return a aplicao
   */
  private Application getApplication() {
    return versionInfoPanel.getApplication();
  }

  /**
   * Inicializa a seleo de uma nova verso na rvore de verses do algoritmo.
   * 
   * @param treeNode n de verso selecionado na rvore
   */
  private void initVersionSelection(TreeNode treeNode) {
    if (infoAlreadyUpdated) {
      versionInfoPanel.verifyAndChangeButtonsState();
    }
    else {
      if (changedSelectedVersion()) {
        if (!versionInfoPanel.confirmSelectionChanged()) {
          setCurrentNode();
          return;
        }
        currentVersionRow = getSelectedRow();
      }
      VersionTreeNode versionNode = getVersionNode(treeNode);
      versionInfoPanel.initVersionNodeEdition(versionNode);
    }
    versionInfoPanel
      .enableVersionEditPanel(treeNode instanceof VersionTreeNode);
  }

  /**
   * Obtm o painel correspondente a viso da rvore de verses.
   * 
   * @return o painel criado
   */
  public JPanel getVersionTreePanel() {
    if (versionTreePanel == null) {
      versionTreePanel = new JPanel(new GridBagLayout());
      versionTreePanel.add(new JScrollPane(getVersionTree()), new GBC(0, 0)
        .both().west().insets(0, 0, 0, 0));
    }
    return versionTreePanel;
  }

  /**
   * Obtm o primeiro n selecionado na rvore de dados.
   * 
   * @return o primeiro n selecionado na rvore de dados
   */
  public VersionNodeInterface getFirstSelectedNode() {
    VersionNodeInterface node = null;
    int selectedCount = versionTree.getSelectionCount();
    if (selectedCount == 1) {
      Object selectionPath = versionTree.getSelectionPath()
        .getLastPathComponent();
      node = (VersionNodeInterface) selectionPath;
    }
    return node;
  }

  /**
   * Obtm o primeiro n selecionado na rvore de dados.
   * 
   * @return o primeiro n selecionado na rvore de dados
   */
  public int getSelectedRow() {
    int[] selectionRows = versionTree.getSelectionRows();
    if (selectionRows != null && selectionRows.length > 0) {
      return selectionRows[0];
    }
    return 0;
  }

  /**
   * Obtm a lista dos dados selecionados na rvore de dados.
   * 
   * @return retorna a lista de dados selecionados na rvore de dados. Se a
   *         seleo
   */
  public List<VersionNodeInterface> getSelectedDataList() {
    List<VersionNodeInterface> selectedDataList = new ArrayList<VersionNodeInterface>();
    TreePath[] selectionPaths = versionTree.getSelectionPaths();
    if (selectionPaths != null) {
      for (TreePath treePath : selectionPaths) {
        Object treeNode = treePath.getLastPathComponent();
        if (treeNode instanceof VersionNodeInterface) {
          selectedDataList.add((VersionNodeInterface) treeNode);
        }
      }
    }
    return selectedDataList;
  }

  /**
   * Obtm os ns referentes ao caminho completo do n selecionado.
   * 
   * @return uma lista com os ns referentes ao caminho completo do n
   *         selecionado
   */
  public List<VersionNodeInterface> getFullPathSelectedNode() {
    List<VersionNodeInterface> fullPath = new ArrayList<VersionNodeInterface>();
    int selectedCount = versionTree.getSelectionCount();
    if (selectedCount == 1) {
      Object[] paths = versionTree.getSelectionPath().getPath();
      VersionTreeRootNode rootNode = (VersionTreeRootNode) paths[0];
      fullPath.add(rootNode);
      for (Object path : paths) {
        if (path instanceof VersionTreeNode) {
          VersionTreeNode node = (VersionTreeNode) path;
          fullPath.add(node);
        }
      }
    }
    return fullPath;
  }

  /**
   * Obtm o n raiz da rvore de dados.
   * 
   * @return retorna o n raiz da rvore de dados, caso contrrio, retorna null
   * 
   */
  public VersionTreeRootNode getRootNode() {
    TreePath rootPath = versionTree.getPathForRow(0);
    Object treeNode = rootPath.getLastPathComponent();
    if (treeNode instanceof VersionTreeRootNode) {
      return (VersionTreeRootNode) treeNode;
    }
    return null;
  }

  /**
   * Obtm um n a partir de um caminho da rvore de dados.
   * 
   * @param path caminho completo de um n da rvore
   * @return o n correspondente a esse caminho
   */
  protected VersionNodeInterface getNode(TreePath path) {
    if (path != null) {
      Object lastPath = path.getLastPathComponent();
      if (VersionNodeInterface.class.isAssignableFrom(lastPath.getClass())) {
        return (VersionNodeInterface) lastPath;
      }
    }
    return null;
  }

  /**
   * Verifica se o n raiz da rvore de dados est selecionado.
   * 
   * @return retorna true, se o n raiz da rvore de dados estiver selecionado,
   *         caso contrrio, retorna false
   * 
   */
  public boolean isRootNodeSelected() {
    return versionTree.isRowSelected(0);
  }

  /**
   * Obtm o nmero de itens selecionados na rvore de dados.
   * 
   * @return o nmero de itens selecionados na rvore de dados
   */
  public int getSelectionCount() {
    return versionTree.getSelectionCount();
  }

  /**
   * Obtm a rvore de verses do algoritmo selecionado.
   * 
   * @return a rvore de verses do algoritmo
   */
  public VersionTree getVersionTree() {
    if (versionTree == null) {
      AlgorithmListItem selectedAlgorithmItem = versionInfoPanel
        .getSelectedAlgorithmItem();
      AlgorithmInfo algoInfo = (selectedAlgorithmItem == null) ? null
        : selectedAlgorithmItem.getItem();
      versionTree = new VersionTree(getApplication().getApplicationFrame(),
        algoInfo);
    }
    return versionTree;
  }

  /**
   * Inicializa a rvore de verso para um novo algoritmo.
   * 
   * @param algoInfo informaes do algoritmo
   */
  public void initializeVersionTree(AlgorithmInfo algoInfo) {
    getVersionTree().initializeTree(algoInfo);
    getVersionTreePanel().revalidate();
  }

  /**
   * Obtm o n de verso a partir de um n selecionado na rvore de verses.
   * 
   * @param treeNode n selecionado na rvore
   * @return Se o n selecionado for a raiz da rvore, retorna null. Se o n
   *         selecionado for um n de verso ou um sub-n de verso, ento
   *         retorna o n da verso.
   */
  public VersionTreeNode getVersionNode(TreeNode treeNode) {
    if (treeNode == null) {
      return null;
    }
    if (treeNode instanceof VersionTreeRootNode) {
      return null;
    }
    if (treeNode instanceof VersionTreeNode) {
      return (VersionTreeNode) treeNode;
    }
    else {
      TreeNode parent = treeNode.getParent();
      return getVersionNode(parent);
    }
  }

  /**
   * Verifica se a verso selecinada mudou em relao  verso corrente.
   * 
   * @return retorna true se a verso selecionado for diferente do que estava
   *         correntemente selecionada, caso contrrio, retorna false
   */
  private boolean changedSelectedVersion() {
    return currentVersionRow != getSelectedRow();
  }

  /**
   * Seleciona o n raiz da rvore de dados, inicializando suas informaes.
   */
  private void selectRootNode() {
    versionTree.setSelectionRow(0);
  }

  /**
   * Seleciona um determinado n na rvore, de acordo com o nmero do n
   * especificado. Inicializa as informaes do n selecionado.
   * 
   * @param nodeRow nmero do n a ser selecionado
   * 
   */
  private void selectNode(int nodeRow) {
    versionTree.selectNode(nodeRow);
  }

  /**
   * Seleciona o n corrente, inicializando os valores do item selecionado.
   * 
   */
  private void selectCurrentNode() {
    if (currentVersionRow == -1) {
      selectRootNode();
    }
    selectNode(currentVersionRow);
  }

  /**
   * Estabelece o n corrente para ser o n selecionado, sem fazer nenhum tipo
   * de inicializao dos seus valores, mantendo o estado corrente do item.
   */
  public void setCurrentNode() {
    infoAlreadyUpdated = true;
    selectCurrentNode();
    infoAlreadyUpdated = false;
  }

  /**
   * Realiza uma ao quando um algoritmo  alterado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo alterado
   */
  public void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
    infoAlreadyUpdated = true;
    versionTree.replaceAlgoNode(algoInfo);
    infoAlreadyUpdated = false;
    selectCurrentNode();
  }
}
