package csbase.client.applications.projectsmanager.dialogs;

import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;

/**
 * Dilogo de cpia de esquema de compartilhamento entre projetos.
 * 
 * @author Tecgraf
 */
public class ReplicateSharedDialog extends ProjectsManagerDialog {

  /**
   * Projeto selecionado.
   */
  final private ProjectsManagerData project;

  /**
   * Tabela com a lista de projetos aos quais as permisses podem ser aplicadas.
   */
  private SortableTable projectsTable;

  /**
   * Boto de confirmao da tela.
   */
  private JButton confirmButton;
  /**
   * Boto de cancelamento da tela.
   */
  private JButton cancelButton;

  /**
   * Projetos selecionados.
   */
  private Set<ProjectsManagerData> selectedProjects;

  /**
   * Construtor.
   * 
   * @param projectsManager A aplicao.
   * @param project Projeto selecionado.
   */
  public ReplicateSharedDialog(final ProjectsManager projectsManager,
    final ProjectsManagerData project) {
    super(projectsManager);
    this.project = project;
    final String title =
      getString("ReplicateSharedDialog.title") + " " + project.getProjectName();
    setTitle(title);
    buildPanel();
  }

  /**
   * Monta o painel principal.
   */
  private void buildPanel() {
    setLayout(new GridBagLayout());

    final JPanel projectListPanel = createProjectListPanel();
    final JPanel buttonsPanel = createButtonsPanel();

    add(projectListPanel, new GBC(0, 0).both());
    add(buttonsPanel, new GBC(0, 1).horizontal().bottom(10));
  }

  /**
   * Cria o painel inferior com os botes "Confirmar" e "Cancelar".
   * 
   * @return painel inferior com os botes "Confirmar" e "Cancelar"
   */
  private JPanel createButtonsPanel() {

    final JPanel panel = new JPanel();

    confirmButton =
      new JButton(LNG.get("ReplicateSharedDialog.button.confirm"));
    confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        setSelectedProjects();
        ReplicateSharedDialog.this.dispose();
      }
    });
    cancelButton = new JButton(LNG.get("ReplicateSharedDialog.button.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        selectedProjects = null;
        ReplicateSharedDialog.this.dispose();
      }
    });
    panel.add(confirmButton);
    panel.add(cancelButton);
    return panel;
  }

  /**
   * Cria o painel com a lista de projetos para os quais se pode (opcionalmente)
   * aplicar as permisses.
   * 
   * @return painel com a lista de projetos para os quais se pode
   *         (opcionalmente) aplicar as permisses
   */
  private JPanel createProjectListPanel() {

    final ProjectsManager projectsManager = getProjectsManager();
    /*
     * geramos uma lista contendo apenas os projetos para os quais o usurio tem
     * permisso de escrita (pois, so os nicos que ele pode alterar)
     */
    final List<ProjectsManagerData> filteredProjects =
      projectsManager.getVisibleProjects();
    final List<ProjectsManagerData> projectsRW =
      new ArrayList<ProjectsManagerData>();
    /*
     * se selectedProjects j fosse uma coleo ao invs de um array o cdigo
     * ficaria muito mais simples... =/
     * 
     * primeiro inserimos apenas os projetos dos quais o usurio  dono (OBS.: o
     * admin  dono de qualquer projeto)
     */
    final Object ownerID = projectsManager.getUser().getId();
    for (ProjectsManagerData pmd : filteredProjects) {
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();
      if (pmd.userIsOwner(ownerID) && spaceAllocation.isOpenable()) {
        projectsRW.add(pmd);
      }
    }
    /*
     * agora removemos o projeto selecionado pelo usurio (j que ele j 
     * afetado pela operao, e portanto no precisa constar da lista)
     */
    projectsRW.remove(project);

    final JPanel panel = new JPanel(new GridBagLayout());
    GBC gbc;
    /*
     * tabela
     */
    projectsTable = new SortableTable(new ProjectListTableModel(projectsRW));
    projectsTable.setRowSelectionAllowed(true);
    projectsTable
      .setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    // preenche todo o espao vertical do scroll pane
    projectsTable.setFillsViewportHeight(true);
    projectsTable.setUseDisabledLook(true);
    projectsTable.setEnabled(true);

    gbc = new GBC(0, 0).both().insets(10, 10, 5, 10);
    panel.add(new JScrollPane(projectsTable), gbc);
    /*
     * texto informativo
     */
    gbc = new GBC(0, 2).insets(0, 10, 10, 10).west();
    final JLabel info =
      new JLabel(LNG.get("ReplicateSharedDialog.info.applyToOthers"),
        ApplicationImages.ICON_INFORMATION_16, SwingConstants.LEFT);
    info.setFont(new Font("Sans", Font.PLAIN, 12));
    panel.add(info, gbc);

    return panel;
  }

  /**
   * Copia a configurao do projeto escolhido aos selecionados na tabela.
   */
  private void setSelectedProjects() {
    selectedProjects = new HashSet<ProjectsManagerData>();
    final ProjectListTableModel model =
      (ProjectListTableModel) projectsTable.getModel();
    final int[] selectedIdxs = projectsTable.getSelectedRows();
    for (int i : selectedIdxs) {
      selectedProjects.add(model.getProject(projectsTable
        .convertRowIndexToModel(i)));
    }
  }

  /**
   * Retorna os projetos selecionados.
   * 
   * @return Os projetos selecionados.
   */
  public Set<ProjectsManagerData> getSelectedProjects() {
    return selectedProjects;
  }
}
