package csbase.client.project;

import java.awt.Window;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.RemoteTask;
import csbase.client.externalresources.LocalFile;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.BugException;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileInfo;
import csbase.logic.ProjectFileType;

/**
 * Tarefa responsvel por importar um ou mais arquivos do sistema de arquivos do
 * cliente para o projeto no servidor.
 */
public class ImportTask extends RemoteTask<Integer> {

  /**
   * Chave para mensagem de erro na ordenao da lista de arquivos a serem
   * importados.
   */
  private static final String FILES_SORTING_ERROR =
    "PRJ_PROJECT_FILE_UPLOAD_FILES_SORTING_ERROR";

  /** Ttulo do dilogo de importao */
  private static final String TITLE = "PRJ_PROJECT_FILE_IMPORT_TITLE";

  /** Mensagem em caso de erro de I/O */
  private static final String IO_ERROR = "PRJ_PROJECT_FILE_IMPORT_IO_ERROR";

  /** Mensagem para passo da importao em execuo */
  private static final String IMPORT_STEP = "PRJ_PROJECT_FILE_IMPORT_STEP";

  /** Tamanho do buffer para I/O. */
  private static final int BUFFER_SIZE = 1024 * 1024;

  /** Arquivos a serem importados */
  private LocalFile[] sourceFiles;

  /** Diretrio de destino para os arquivos importados */
  private ClientProjectFile targetDir;

  /** O projeto */
  private CommonClientProject project;

  /** A janela da aplicao [para dialogs de erro] */
  private Window window;

  /**
   * Timestamp de incio da transferncia (em milissegundos). Esse timestamp 
   * posterior  criao de arquivos no projeto.
   */
  public long transferStart;

  /** Volume total ocupado pelos arquivos a serem importados. */
  private long totalSize;

  /** Arquivo corrente na transferncia */
  private ClientProjectFile currentTransferFile;

  /**
   * Cria a tarefa.
   * 
   * @param project projeto para onde sero importados os arquivos
   * @param window janela-me para exibio de dilogos de erro
   * 
   * @param sourceFiles arquivos a serem importados.
   * @param targetDir diretrio de destino
   */
  public ImportTask(CommonClientProject project, Window window,
    LocalFile[] sourceFiles, ClientProjectFile targetDir) {
    this.project = project;
    this.window = window;
    this.sourceFiles = sourceFiles;
    this.targetDir = targetDir;
  }

  /**
   * Especifica como  feito o tratamento de erros. Trata somente
   * <code>IOException</code>, outros erros so repassados para o mtodo pai na
   * hierarquia.
   * 
   * @param error .
   */
  @Override
  protected void handleError(Exception error) {
    if (error instanceof FileLockedException) {
      StandardDialogs.showErrorDialog(
        window,
        LNG.get(TITLE),
        String.format(LNG.get("csbase.file.locked.error"),
          currentTransferFile.getName()));
    }
    else if (error instanceof IOException) {
      showError(LNG.get(IO_ERROR), error);
    }
    else {
      super.handleError(error);
    }
  }

  /**
   * Executa a tarefa. Transfere o contedo de todos os arquivos de origem para
   * arquivos com o mesmo nome no servidor.
   * 
   * O resultado da task :
   * <ul>
   * <li>-1 : nenhum arquivo foi importado (todas as importaes falharam)
   * <li>0 : todas as importaes foram feitas com sucesso
   * <li>n : nmero de arquivos que no foram importados, por algum motivo
   * (provavelmente por coliso de nomes com arquivos no destino)
   * </ul>
   * 
   * @throws Exception em caso de erro.
   */
  @Override
  protected void performTask() throws Exception {
    /*
     * precisamos primeiro garantir que todos os arquivos que sero importados
     * existem na rvore do projeto
     */
    createAllFiles();
    /*
     * obtemos os arquivos na rvore do projeto que correspondem a cada arquivo
     * que foi selecionado pelo usurio (array sourceFiles)
     */
    final ClientProjectFile[] targetFiles = getFilesToImport();

    transferStart = System.currentTimeMillis();
    int errors = 0;
    for (int i = 0; i < targetFiles.length; i++) {
      LocalFile sourceFile = sourceFiles[i];
      ClientProjectFile targetFile = targetFiles[i];
      if (targetFile == null) {
        /*
         * incrementamos o contador de erros para exibir uma nica mensagem aps
         * o loop
         */
        errors++;
        continue;
      }
      currentTransferFile = targetFile;
      transferFile(sourceFile, targetFile);
    }
    if (errors > 0) {
      if (errors == targetFiles.length) {
        /*
         * todas as importaes falharam
         */
        setResult(Integer.valueOf(-1));
      }
      else {
        /*
         * algumas importaes falharam
         */
        setResult(Integer.valueOf(errors));
      }
    }
    else {
      /*
       * todos os arquivos foram importados com sucesso
       */
      setResult(Integer.valueOf(0));
    }
  }

  /**
   * Percorre a lista com as informaes dos arquivos a serem importados,
   * criando aqueles que ainda no existem.
   * 
   * @throws IOException se houver erro na recuperao das informaes dos
   *         arquivos de origem (locais)
   * @throws RemoteException se houver erro na criao dos arquivos de destino
   *         no projeto
   */
  private void createAllFiles() throws IOException, RemoteException {
    /*
     * precisamos obter a lista de todos os arquivos/diretrios que sero
     * criados como resultado da importao. Se entre os arquivos a serem
     * importados existem diretrios, estes sero percorridos em profundidade
     * para se obter as listas para as respectivas sub-rvores.
     */
    List<ProjectFileInfo> filesInfos = getAllFileInfos();
    List<ProjectFileInfo> filesToCreate = new ArrayList<ProjectFileInfo>();
    for (ProjectFileInfo aFileInfo : filesInfos) {
      if (null == targetDir.getChild(aFileInfo)) {
        filesToCreate.add(aFileInfo);
      }
    }
    /*
     * criamos os arquivos que ainda no existem
     */
    project.createAndWaitForFiles(targetDir, filesToCreate);
  }

  /**
   * Obtm da rvore do projeto os arquivos de destino que correspondem aos
   * arquivos que sero importados.
   * 
   * @return array com os arquivos na rvore do projeto que correspondem aos
   *         arquivos que sero importados
   * 
   * @throws IOException - se houver erro ao recuperar os nomes dos arquivos de
   *         origem
   * @throws RemoteException - se houver erro obtendo as referncias na rvore
   *         do projeto
   */
  private ClientProjectFile[] getFilesToImport() throws RemoteException,
    IOException {
    ClientProjectFile[] files = new ClientProjectFile[sourceFiles.length];
    for (int inx = 0; inx < files.length; inx++) {
      files[inx] = targetDir.getChild(sourceFiles[inx].getName());
    }
    return files;
  }

  /**
   * Transfere um arquivo ou o contedo de um diretrio do cliente para o
   * servidor.
   * 
   * @param sourceFile arquivo/diretrio de origem.
   * @param targetFile arquivo/diretrio de destino.
   * @throws IOException em caso de erro de I/O.
   */
  private void transferFile(LocalFile sourceFile, ClientProjectFile targetFile)
    throws IOException {
    if (targetFile.isDirectory()) {
      // Como no  possvel a importao de diretrios se o cliente no
      // for assinado, pode ser assumido que se um diretrio  destino,
      // a origem tem de ser um StandaloneLocalFile.
      StandaloneLocalFile[] sourceChildren =
        ((StandaloneLocalFile) sourceFile).listFiles();
      Arrays.sort(sourceChildren, new Comparator<LocalFile>() {
        @Override
        public int compare(LocalFile f1, LocalFile f2) {
          try {
            return f1.getName().compareTo(f2.getName());
          }
          catch (IOException ex) {
            showError(FILES_SORTING_ERROR, ex);
            return 0;
          }
        }
      });
      ClientProjectFile[] targetChildren = targetFile.getChildren();
      Arrays.sort(targetChildren);
      for (int j = 0; j < targetChildren.length; j++) {
        StandaloneLocalFile sourceChild = sourceChildren[j];
        ClientProjectFile targetChild = targetChildren[j];
        if (!sourceChild.getName().equals(targetChild.getName())) {
          throw new BugException(String.format("Importao: %s >>> %s",
            sourceChild.getName(), targetChild.getName()));
        }
        transferFile(sourceChild, targetChild);
      }
    }
    else {
      transferFileContents(sourceFile, targetFile);
    }
  }

  /**
   * Transfere efetivamente o contedo de um arquivo de origem no cliente para
   * um arquivo de destino no servidor.
   * 
   * @param sourceFile arquivo de origem.
   * @param targetFile arquivo de destino.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private void transferFileContents(LocalFile sourceFile,
    ClientProjectFile targetFile) throws IOException {
    final String fileName = sourceFile.getName();
    String msg =
      MessageFormat.format(LNG.get(IMPORT_STEP), new Object[] { fileName });
    setStepText(msg);
    BufferedInputStream in =
      new BufferedInputStream(sourceFile.getInputStream());
    targetFile.upload(in, BUFFER_SIZE, null);
    in.close();
  }

  /**
   * <p>
   * Retorna uma lista contendo os dados de todos os arquivos a serem
   * importados, na forma de {@link ProjectFileInfo}s. Os caminhos so relativos
   * aos arquivos selecionados na interface pelo usurio.
   * <p>
   * <p>
   * Exemplo: supondo que o usurio tenha selecionado o arquivo "arq2" e o
   * diretrio "dir1", o qual contm o arquivo "arq1" e o diretrio "subdir",
   * que por sua vez contm o arquivo "arq3", os caminhos dos
   * {@link ProjectFileInfo}s retornados pelo mtodo seriam os seguintes:
   * </p>
   * {"arq2"}<br>
   * {"dir1"}<br>
   * {"dir1", "arq1"}<br>
   * {"dir1", "subdir", "arq3"}
   * 
   * @return lista contendo os dados de todos os arquivos a serem importados.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private List<ProjectFileInfo> getAllFileInfos() throws IOException {
    List<ProjectFileInfo> fileInfos = new ArrayList<ProjectFileInfo>();
    for (LocalFile file : sourceFiles) {
      List<String> path = new ArrayList<String>();
      path.add(file.getName());
      addAllFileInfos(file, path, fileInfos);
    }
    return fileInfos;
  }

  /**
   * Adiciona o caminho do arquivo especificado, assim como dos seus filhos
   * (caso este seja um diretrio),  lista de caminhos.
   * 
   * @param file arquivo ou diretrio a ser importado.
   * @param path caminho do arquivo ou diretrio, relativo aos arquivos
   *        selecionados pelo usurio.
   * @param fileInfoList lista de objetos contendo dados de arquivos.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private void addAllFileInfos(LocalFile file, List<String> path,
    List<ProjectFileInfo> fileInfoList) throws IOException {

    ProjectFileInfo fileInfo =
      new ProjectFileInfo(path.toArray(new String[path.size()]));
    fileInfo.setType(ProjectFileType.UNKNOWN);
    fileInfo.setSize(file.getLength());
    totalSize += fileInfo.getSize();
    fileInfoList.add(fileInfo);
    if ((file instanceof StandaloneLocalFile)) {
      StandaloneLocalFile STFile = (StandaloneLocalFile) file;
      StandaloneLocalFile[] children = STFile.listFiles();
      if (children == null) {
        return;
      }
      fileInfo.setType(ProjectFileType.DIRECTORY_TYPE);
      if (children.length > 0) {
        for (LocalFile child : children) {
          List<String> childPath = new ArrayList<String>();
          childPath.addAll(path);
          childPath.add(child.getName());
          addAllFileInfos(child, childPath, fileInfoList);
        }
      }
    }
  }

  /**
   * Mtodo de aviso de erros de sistema.
   * 
   * @param msg mensagem a ser exibida.
   * @param ex exceo detectada.
   */
  private void showError(String msg, Exception ex) {
    StandardErrorDialogs.showErrorDialog(window, LNG.get(TITLE), msg, ex);
  }

  /**
   * Retorna o timestamp de quando se iniciou a tranferncia.
   * 
   * @return o timestamp de quando se iniciou a tranferncia.
   */
  public long getTransferStart() {
    return transferStart;
  }

  /**
   * Retorna o volume total transferido.
   * 
   * @return o volume total transferido.
   */
  public long getTotalSize() {
    return totalSize;
  }
}
