/*
 * $Id: ProjectTreeMouseAdapter.java 157822 2014-10-30 13:25:28Z pietroguedes $
 */
package csbase.client.project;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Vector;

import javax.swing.JTree;
import javax.swing.tree.TreePath;

import csbase.client.project.action.CommonFileOpenAction;
import csbase.logic.ClientProjectFile;

/**
 * A classe <code>ProjectTreeMouseAdapter</code> modela os eventos de uma rvore
 * de diretrios de um projeto.
 * 
 * @author Taciana
 */
public abstract class ProjectTreeMouseAdapter extends MouseAdapter {

  /**
   * rvore sobre a qual este adapter atua.
   */
  private ProjectTree projectTree;

  /**
   * {@inheritDoc}
   */
  @Override
  public void mouseReleased(MouseEvent me) {
    ClientProjectFile file = getFileFromEventPath(me);
    if (file != null && me.isPopupTrigger()) {
      int mex = me.getX();
      int mey = me.getY();
      showPopupMenu(file, mex, mey);
    }
  }

  /**
   * Processa o evento de mouseClicked e caso tenha ocorrido um double-click
   * executa a ao correspondente. O double-click tem que ser capturado neste
   * evento, seno no Windows o efeito  que a qualquer janela aberta em
   * decorrncia de um double-click aparece atrs da janela em que o evento
   * aconteceu.
   */
  @Override
  public void mouseClicked(MouseEvent me) {
    ClientProjectFile file = getFileFromEventPath(me);
    if (file != null && (me.getClickCount() == 2)) {
      doubleClickAction(file);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void mousePressed(MouseEvent me) {
    int mex = me.getX();
    int mey = me.getY();
    ClientProjectFile file = getFileFromEventPath(me);
    if (file != null && me.isPopupTrigger()) {
      showPopupMenu(file, mex, mey);
    }
  }

  /**
   * Trata o duplo clique do mouse.
   * 
   * @param file arquivo selecionado.
   */
  abstract public void doubleClickAction(ClientProjectFile file);

  /**
   * Disparao ao de abertura de arquivo.
   * 
   * @param file arquivo de projeto.
   */
  final protected void fireOpenAction(ClientProjectFile file) {
    if (file.isDirectory()) {
      return;
    }
    CommonFileOpenAction openAction = new CommonFileOpenAction(projectTree);
    openAction.openFile(file);
  }

  /**
   * Verifica se  um caminho vlido.
   * 
   * @param me evento do mouse
   * @return true se clicou em um n da rvore
   */
  private ClientProjectFile getFileFromEventPath(MouseEvent me) {
    int mex = me.getX();
    int mey = me.getY();
    TreePath path = getTree().getPathForLocation(mex, mey);
    if (path == null) {
      return null;
    }
    ProjectTreeNode lastPathComponent =
      (ProjectTreeNode) path.getLastPathComponent();
    ClientProjectFile file = lastPathComponent.getClientProjectFile();
    return file;
  }

  /**
   * Exibe o menu popup
   * 
   * @param file arquivo selecionado.
   * @param mex posio do evento original (coord. x).
   * @param mey posio do evento original (coord. y).
   */
  private void showPopupMenu(ClientProjectFile file, int mex, int mey) {
    JTree tree = getTree();
    if (tree.getSelectionCount() > 1) {
      if (projectTree.getPopupMultSelection() != null) {
        notifyPopupMultSelectionWasTriggered();
        projectTree.getPopupMultSelection().show(tree, mex, mey);
      }
    }
    else {
      TreePath path = getTree().getPathForLocation(mex, mey);
      tree.setSelectionPath(path);
      if (file.isDirectory()) {
        if ((file.getParent() == null) && (projectTree.getPopupRoot() != null)) {
          notifyPopupRootWasTriggered();
          projectTree.getPopupRoot().show(tree, mex, mey);
        }
        else {
          if (projectTree.getPopupDir() != null) {
            notifyPopupDirWasTriggered(file);
            projectTree.getPopupDir().show(tree, mex, mey);
          }
        }
        return;
      }
      if (projectTree.getPopupFile() != null) {
        notifyPopupFileWasTriggered(file);
        projectTree.getPopupFile().show(tree, mex, mey);
      }
    }
  }

  /**
   * Notifica os listeners sobre a exibio do menu de popup de seleo
   * mltipla.
   */
  private void notifyPopupMultSelectionWasTriggered() {
    Vector<ProjectTreeListener> listeners = projectTree.getListeners();
    for (ProjectTreeListener listener : listeners) {
      listener.popupMultSelectionTriggered();
    }
  }

  /**
   * Notifica os listeners sobre a exibio do menu de popup da raiz do projeto.
   */
  private void notifyPopupRootWasTriggered() {
    Vector<ProjectTreeListener> listeners = projectTree.getListeners();
    for (ProjectTreeListener listener : listeners) {
      listener.popupRootTriggered();
    }
  }

  /**
   * Notifica os listeners sobre a exibio do menu de popup de diretrio.
   * 
   * @param dir O diretrio sobre o qual o menu de popup foi chamado.
   */
  private void notifyPopupDirWasTriggered(ClientProjectFile dir) {
    Vector<ProjectTreeListener> listeners = projectTree.getListeners();
    for (ProjectTreeListener listener : listeners) {
      listener.popupDirTriggered(dir);
    }
  }

  /**
   * Notifica os listeners sobre a exibio do menu de popup de arquivo.
   * 
   * @param file O arquivo sobre o qual o menu de popup foi chamado.
   */
  private void notifyPopupFileWasTriggered(ClientProjectFile file) {
    Vector<ProjectTreeListener> listeners = projectTree.getListeners();
    for (ProjectTreeListener listener : listeners) {
      listener.popupFileTriggered(file);
    }
  }

  /**
   * Construtor
   * 
   * @param projectTree rvore de projeto.
   */
  public ProjectTreeMouseAdapter(ProjectTree projectTree) {
    this.projectTree = projectTree;
  }

  /**
   * Consulta  rvore.
   * 
   * @return rvore Swing
   */
  private JTree getTree() {
    return projectTree.getTree();
  }
}
