/*
 * $Id: SingleFileExportAction.java 149543 2014-02-11 13:02:45Z oikawa $
 */
package csbase.client.project.action;

import java.awt.event.ActionEvent;
import java.io.File;
import java.security.AccessControlException;

import javax.swing.JFileChooser;

import tecgraf.javautils.core.io.FileUtils;
import csbase.client.project.ProjectFileContainer;
import csbase.logic.ClientProjectFile;

/**
 * Ao para exportao de um nico <b>arquivo</b>, a partir do projeto no
 * servidor para o disco local do usurio.
 * <p>
 * Esta ao tenta primeiro acessar diretamente o sistema de arquivos local,
 * utilizando o <code>JFileChooser</code>. Caso no consiga (no caso dos jars
 * no estarem assinados), utiliza a API JNLP.
 * 
 * @author Tecgraf/PUC-Rio.
 */
public class SingleFileExportAction extends CommonFileExportAction {

  /**
   * Construtor.
   * 
   * @param container objeto contendo arquivos de projeto.
   */
  public SingleFileExportAction(ProjectFileContainer container) {
    super(container);
  }

  /**
   * Mtodo que dispara a ao. Dispara uma <code>ExportTask</code> encarregada
   * de fazer a transferncia do arquivo selecionado para o disco local do
   * usurio.
   * <p>
   * Caso as permisses de acesso no possibilitem o acesso direto ao sistema de
   * arquivos local, a transferncia  feita via JNLP.
   * 
   * @param e informaes sobre o evento ocorrido.
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    ClientProjectFile selectedFile = getSelectedFile();
    if (selectedFile == null || selectedFile.isDirectory()) {
      return;
    }
    long fileSize = getFileSize(selectedFile);
    if (fileSize == -1L) {
      // houve algum erro na comunicao com o servidor, aborta a operao
      return;
    }
    if (fileSize == 0L && !confirmEmptyFileExport(selectedFile)) {
      return;
    }
    try {
      /*
       * Primeiro vamos tentar usar o JFileChooser nativo (assumindo que as
       * permisses de acesso possibilitam isto). Se no der certo, o catch mais
       * abaixo usar o file chooser do JNLP.
       */
      File userTarget = getUserTarget(selectedFile);
      if (userTarget == null) {
        return;
      }
      if (userTarget.exists()) {
        if (!confirmOverwrite(userTarget)) {
          return;
        }
        FileUtils.delete(userTarget);
      }
      transferFile(selectedFile, userTarget);
    }
    catch (AccessControlException ex1) {
      /*
       * Acesso ao sistema de arquivos negado (rodando em "sandbox"): interao
       * com sistema de arquivos deve ser feita via JNLP. No  possvel
       * exportar diretrios nem mais de um arquivo.
       */
      transferFileUsingJNLP(selectedFile);
    }
  }

  /**
   * Obtm o arquivo de destino local.
   * 
   * @param clientFile arquivo a ser exportado
   * 
   * @return arquivo local selecionado pelo usurio ou <code>null</code> caso o
   *         usurio tenha cancelado a seleo.
   */
  private File getUserTarget(ClientProjectFile clientFile) {
    JFileChooser chooser = getFileChooser();
    chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    chooser.setMultiSelectionEnabled(false);
    chooser.setSelectedFile(new File(chooser.getCurrentDirectory()
      + File.separator + clientFile.getName()));
    int result = chooser.showSaveDialog(getWindow());
    if (result == JFileChooser.APPROVE_OPTION) {
      return chooser.getSelectedFile();
    }
    return null;
  }
}
