/*
 * AlgorithmManagementProxy.java
 * 
 * @author Ana $Revision: 167962 $ - $Date: 2009-07-10 14:25:16 -0300 (Fri, 10
 * Jul 2009) $
 */
package csbase.client.remote.srvproxies;

import java.awt.Window;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.JFrame;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.Client;
import csbase.client.ClientServerManager;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.AlgorithmManagementListener;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseRuntimeException;
import csbase.exception.PermissionException;
import csbase.exception.algorithms.CategoriesFileNotSavedException;
import csbase.logic.AdminPermission;
import csbase.logic.AlgoEvent;
import csbase.logic.AlgorithmAdminPermission;
import csbase.logic.AlgorithmExecutionPermission;
import csbase.logic.CategoryAlgorithmsExecutionPermission;
import csbase.logic.FileInfo;
import csbase.logic.IPathFactory;
import csbase.logic.MonitoredServerListener;
import csbase.logic.ServerURI;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmOutline;
import csbase.logic.algorithms.AlgorithmProperty;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.HistoryRecord;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * Classe que modela um proxy no cliente para o para acesso s operaes
 * disponibilizadas pelo servio de algoritmos (AlgoService).
 *
 * @author Ana
 */
public class AlgorithmManagementProxy {
  /** Ttulo para janelas de tarefas. */
  private static String taskTitle = LNG.get("algomanager.title");

  /** Cache do conjunto de categorias disponveis no servidor */
  private static CategorySet categorySet;

  /** Algoritmos carregados do servidor */
  private static SortedSet<AlgorithmInfo> allAlgorithmInfo;

  /** Mapeamento dos algoritmos e as categorias associadas */
  private static HashMap<AlgorithmInfo, SortedSet<Category>> algorithmToCategoryMap;

  /**
   * Listeners interessados em mudanas no conjunto de algoritmos /ou categorias
   */
  private static List<AlgorithmManagementListener> managementList;

  /** Indica se todos os algoritmos esto na cache */
  private static boolean hasAllAlgorithms;

  /** Indica se todas as categorias esto na cache */
  private static boolean hasAllCategories;

  /**
   * Representa a operao que ser realizada sobre um ou mais algoritmos.
   *
   */
  public enum AlgorithmOperation {
    /** Gerenciamento do algoritmo */
    ADMIN_ALGORITHM,
    /** Execuo do algoritmo */
    EXECUTE_ALGORITHM,
    /**
     * Qualquer operao que no exija permisso, isto , vai atuar sobre todos
     * os algoritmos
     */
    ANY;
  }

  static {
    managementList = new Vector<AlgorithmManagementListener>();
    hasAllAlgorithms = false;
    hasAllCategories = false;
    categorySet = new CategorySet();
    allAlgorithmInfo = new TreeSet<AlgorithmInfo>();
    algorithmToCategoryMap = new HashMap<AlgorithmInfo, SortedSet<Category>>();
    addCategoryEventsObserver();
    addCategorySetEventsObserver();
    addAlgorithmEventsObserver();
    addServerMonitorListener();
  }

  /**
   * Adiciona um listener para o monitorar o servidor, e preparar o cache para
   * ser recarregado quando a conexo for reestabelecida.
   */
  private static void addServerMonitorListener() {
    ClientServerManager.getInstance().addCommonListener(
      new MonitoredServerListener() {
        @Override
        public void notifyConnectionLost(ServerURI serverURI) {
        }

        @Override
        public void notifyConnectionReestablished(ServerURI serverURI) {
          prepareCacheReload();
          notifyAlgorithmsReloaded();
        }

        @Override
        public void notifyLoggedIn(ServerURI serverURI) {
        }

        @Override
        public void notifyLoggedOut(ServerURI serverURI) {
        }
      });
  }

  /**
   * Obtm o nome do servidor remoto.
   *
   * @return o nome do servidor remoto
   */
  private static String getServerName() {
    final String systemName = Client.getInstance().getSystemName();
    String serverName = null;
    final RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.server.getSystemName());
      }
    };

    // Esse dilogo no deve aparecer rapidamente pois dar uma impresso
    // ruim ao usurio no start-up do desktop. Ele somente aparecer caso
    // haja alguma demora real na conexo com o servidor.
    task.setProgressDialogDelay(10);
    task.execute(DesktopFrame.getInstance().getDesktopFrame(), "", systemName
      + "...");
    if (!task.getStatus()) {
      serverName = "ERROR";
    }
    serverName = task.getResult();
    if (serverName == null) {
      serverName = "[" + systemName + "]";
    }
    return serverName;
  }

  /**
   * <p>
   * Copia arquivos de uma pasta para a outra com a possibilidade de dar
   * permisso de execuo s cpias.
   * </p>
   * <p>
   * Para definir o caminho do arquivo de destino, ser construido um FileInfo
   * usando o targetDir como path se este for diferente de <tt>null</tt> e
   * mantendo o nome do arquivo original. Este novo FileInfo ser passado como
   * parmetro para o {@link IPathFactory#getPath(FileInfo)
   * targetPathFactory.getPath(FileInfo)} que retornar o caminho fsico do novo
   * arquivo.
   * </p>
   * <p>
   * Ao final da execuo os algoritmos sero recarregados.
   * </p>
   *
   * @param sourceVersion Informaes da verso de origem.
   * @param files Arquivos a serem transferidos.
   * @param sourcePathFactory Fbrica para criar o caminho dos arquivos a serem
   *        transferidos.
   * @param targetVersion Informaes da verso de destino.
   * @param targetDir diretrio de destino.
   * @param targetPathFactory Fbrica para criar o caminho dos arquivos de
   *        destino. Este caminho  relativo ao valor do parmetro targetDir.
   * @param setExecutables <tt>true</tt> se as cpias devero ter permisso de
   *        execuo ou no.
   * @param owner janela que requisitou esta operao.
   */
  public static void copyFiles(final AlgorithmVersionInfo sourceVersion,
    final List<FileInfo> files, final IPathFactory sourcePathFactory,
    final AlgorithmVersionInfo targetVersion, final FileInfo targetDir,
    final IPathFactory targetPathFactory, final boolean setExecutables,
    final Window owner) {

    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        algoService.copyFiles(sourceVersion, files, sourcePathFactory,
          targetVersion, targetDir, targetPathFactory, setExecutables);
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
  }

  /**
   * Cria um novo algoritmo.
   *
   * @param name nome do novo algoritmo
   * @param id identificador do novo algoritmo
   * @param fieldValues propriedades do algoritmo
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return informaes do novo algoritmo
   */
  public static AlgorithmInfo createAlgorithm(final String name,
    final String id, final Hashtable<String, String> fieldValues, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.createAlgorithm(name,
          id, fieldValues));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Cria uma nova verso de algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param major numero da verso
   * @param minor numero da reviso
   * @param patch numero da correo
   * @param properties propriedades da verso
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return informaes atualizadas do algoritmo
   */
  public static AlgorithmInfo createVersion(final Object algoId,
    final int major, final int minor, final int patch,
    final Hashtable<String, String> properties, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.createVersion(algoId,
          major, minor, patch, properties));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Cria uma categoria no servidor.
   *
   * @param parentCategoryId identificador da categoria pai
   * @param name nome da categoria a ser criada
   * @param owner janela pai
   *
   * @return a categoria criada
   *
   */
  public static Category createCategory(final String parentCategoryId,
    final String name, Window owner) {

    RemoteTask<Category> task = new RemoteTask<Category>() {
      @Override
      protected void performTask() throws Exception {
        if (ClientRemoteLocator.algorithmService != null) {
          Category category =
            ClientRemoteLocator.algorithmService.createCategory(
              parentCategoryId, name);
          setResult(category);
        }
      }

      @Override
      protected void handleServerError(CSBaseRuntimeException error) {
        if (error instanceof CategoriesFileNotSavedException) {
          StandardErrorDialogs
            .showErrorDialog(
              DesktopFrame.getInstance().getView(),
              "Criando Categoria de Algoritmos",
              "Erro na persistncia do arquivo de categorias. Suas ltimas alteraes sobre categorias esto salvas somente em memria. ");
        }
        else {
          super.handleServerError(error);
        }
      }

    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Cria uma categoria no servidor.
   *
   * @param id identificador da categoria a ser removida
   * @param owner janela pai
   *
   * @return retorna true se a categoria foi removida com sucesso, caso
   *         contrrio, retorna false
   *
   */
  public static boolean removeCategory(final String id, Window owner) {

    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        if (ClientRemoteLocator.algorithmService != null) {
          ClientRemoteLocator.algorithmService.removeCategory(id);
        }
      }

      @Override
      protected void handleServerError(CSBaseRuntimeException error) {
        if (error instanceof CategoriesFileNotSavedException) {
          StandardErrorDialogs
            .showErrorDialog(
              DesktopFrame.getInstance().getView(),
              "Removendo Categoria de Algoritmos",
              "Erro na persistncia do arquivo de categorias. Suas ltimas alteraes sobre categorias esto salvas somente em memria. ");
        }
        else {
          super.handleServerError(error);
        }
      }
    };
    return task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
  }

  /**
   * Associa um conjunto de algoritmos a um conjunto de categorias.
   *
   * Nesse caso, para cada categoria da lista de categorias sero adicionados
   * todos os algoritmos da lista de algoritmos. Essa informao ser persistida
   * em xml.
   *
   * @param categoryIds lista de identificadores das categorias
   * @param algorithmIds lista de identificadores dos algoritmos
   * @param owner janela do cliente
   * @param algoOperation operao que ser realizada sobre os algoritmos das
   *        categorias
   *
   * @return retorna o conjunto de categorias modificadas
   *
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public static CategorySet bindAlgorithmsToCategories(
    final List<Object> algorithmIds, final List<String> categoryIds,
    Window owner, AlgorithmOperation algoOperation) {

    RemoteTask<CategorySet> task = new RemoteTask<CategorySet>() {
      @Override
      protected void performTask() throws Exception {
        if (ClientRemoteLocator.algorithmService != null) {
          CategorySet modifiedCategories =
            ClientRemoteLocator.algorithmService.bindAlgorithmsToCategories(
              algorithmIds, categoryIds);
          setResult(modifiedCategories);
        }
      }

      @Override
      protected void handleServerError(CSBaseRuntimeException error) {
        if (error instanceof CategoriesFileNotSavedException) {
          StandardErrorDialogs
            .showErrorDialog(
              DesktopFrame.getInstance().getView(),
              "Associando algoritmos a categorias",
              "Erro na persistncia do arquivo de categorias. Suas ltimas alteraes sobre categorias esto salvas somente em memria. ");
        }
        else {
          super.handleServerError(error);
        }
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    CategorySet categorySet = task.getResult();

    if (!isAdminLoggedUser()) {
      return getRestrictedCategories(categorySet.getCategories(), algoOperation);
    }
    return categorySet;
  }

  /**
   * Desassocia um conjunto de algoritmos de um conjunto de categorias.
   *
   * Nesse caso, para cada categoria da lista de categorias sero removidos
   * todos os algoritmos da lista de algoritmos. Essa informao ser persistida
   * em xml.
   *
   * @param categoryIds lista de identificadores das categorias
   * @param algorithmIds lista de identificadores dos algoritmos
   * @param owner janela do cliente
   * @param algoOperation operao que ser realizada sobre os algoritmos das
   *        categorias
   *
   * @return retorna o conjunto de categorias modificadas
   *
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public static CategorySet unbindAlgorithmsFromCategories(
    final List<Object> algorithmIds, final List<String> categoryIds,
    Window owner, AlgorithmOperation algoOperation) {

    RemoteTask<CategorySet> task = new RemoteTask<CategorySet>() {
      @Override
      protected void performTask() throws Exception {
        if (ClientRemoteLocator.algorithmService != null) {
          CategorySet modifiedCategories =
            ClientRemoteLocator.algorithmService
              .unbindAlgorithmsFromCategories(algorithmIds, categoryIds);
          setResult(modifiedCategories);
        }
      }

      @Override
      protected void handleServerError(CSBaseRuntimeException error) {
        if (error instanceof CategoriesFileNotSavedException) {
          StandardErrorDialogs
            .showErrorDialog(
              DesktopFrame.getInstance().getView(),
              "Desassociando algoritmos de categorias",
              "Erro na persistncia do arquivo de categorias. Suas ltimas alteraes sobre categorias esto salvas somente em memria. ");
        }
        else {
          super.handleServerError(error);
        }
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    CategorySet categorySet = task.getResult();

    if (!isAdminLoggedUser()) {
      return getRestrictedCategories(categorySet.getCategories(), algoOperation);
    }
    return categorySet;
  }

  /**
   * Atribui um conjunto de algoritmos de um conjunto de categorias,
   * substituindo o conjunto anterior.
   *
   * @param categoryIds lista de identificadores das categorias
   * @param algorithmIds lista de identificadores dos algoritmos
   * @param owner janela do cliente
   * @param algoOperation operao que ser realizada sobre os algoritmos das
   *        categorias
   *
   * @return retorna o conjunto de categorias modificadas
   *
   * @throws CategoriesFileNotSavedException erro ao tentar gravar o xml de
   *         categorias
   */
  public static CategorySet setAlgorithmsToCategories(
    final List<Object> algorithmIds, final List<String> categoryIds,
    Window owner, AlgorithmOperation algoOperation) {

    RemoteTask<CategorySet> task = new RemoteTask<CategorySet>() {
      @Override
      protected void performTask() throws Exception {
        if (ClientRemoteLocator.algorithmService != null) {
          CategorySet modifiedCategories =
            ClientRemoteLocator.algorithmService.setAlgorithmsToCategories(
              algorithmIds, categoryIds);
          setResult(modifiedCategories);
        }
      }

      @Override
      protected void handleServerError(CSBaseRuntimeException error) {
        if (error instanceof CategoriesFileNotSavedException) {
          StandardErrorDialogs
            .showErrorDialog(
              DesktopFrame.getInstance().getView(),
              "Atribuindo algoritmos a categorias",
              "Erro na persistncia do arquivo de categorias. Suas ltimas alteraes sobre categorias esto salvas somente em memria. ");
        }
        else {
          super.handleServerError(error);
        }
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    CategorySet categorySet = task.getResult();

    if (!isAdminLoggedUser()) {
      return getRestrictedCategories(categorySet.getCategories(), algoOperation);
    }
    return categorySet;
  }

  /**
   * Duplica uma verso de um algoritmo.
   *
   * @param algoId identificador do algoritmo cuja verso ser duplicada.
   * @param versionId identificador da verso a ser duplicada.
   * @param major nmero da verso.
   * @param minor nmero da reviso.
   * @param patch nmero da correo.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   */
  public static void duplicateVersion(final Object algoId,
    final Object versionId, final int major, final int minor, final int patch,
    Window owner) {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        algoService.duplicateVersion(algoId, versionId, major, minor, patch);
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
  }

  /**
   * Obtm todas as categorias com os algoritmos restritos que o usurio tem
   * permisso para realizar a operao especificada.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos das
   *        categorias
   *
   * @return conjunto de categorias contendo os algoritmos restritos que o
   *         usurio tem permisso para realizar a operao
   */
  public static CategorySet getAllCategories(Window owner,
    AlgorithmOperation algoOperation) {
    if (algoOperation == null) {
      throw new IllegalArgumentException(
        getParameterNullMessage("algoOperation"));
    }
    CategorySet categorySet = null;
    if (!hasAllCategories) {
      categorySet = loadCategoriesFromServer(owner);
      updateCategoriesCache(categorySet);
    }
    else {
      categorySet = AlgorithmManagementProxy.categorySet;
    }

    if (!isAdminLoggedUser()) {
      return getRestrictedCategories(categorySet.getCategories(), algoOperation);
    }
    return categorySet;
  }

  /**
   * Obtm uma categoria que possui o identificador especificado, contendo os
   * algoritmos restritos que o usurio tem permisso para realizar a operao
   * especificada.
   *
   * @param categoryId identificador da categoria
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos das
   *        categorias
   *
   * @return a categoria encontrada contendo os algoritmos restritos s
   *         permisses do usurio, caso contrrio, retorna null
   */
  public Category getCategory(final String categoryId, Window owner,
    AlgorithmOperation algoOperation) {
    if (algoOperation == null) {
      throw new IllegalArgumentException(
        getParameterNullMessage("algoOperation"));
    }
    RemoteTask<Category> task = new RemoteTask<Category>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.getCategory(categoryId));
      }
    };
    task.execute(owner, LNG.get("algomanager.title.consulting_data"),
      LNG.get("algomanager.msg.consulting_data"));
    Category category = task.getResult();

    if (!isAdminLoggedUser()) {
      Category restrictedCategory = new Category(category);
      updateRestrictedAlgorithms(restrictedCategory, algoOperation);
      updateRestrictedCategories(restrictedCategory, restrictedCategory
        .getCategories().iterator(), algoOperation);
      return restrictedCategory;

    }
    return category;
  }

  /**
   * Obtm os resumos dos algoritmos restritos aos algoritmos que o usurio tem
   * permisso para realizar a operao especificada.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   *
   * @return o array de outlines dos algoritmos restritos
   */
  public static AlgorithmOutline[] getAllAlgorithmOutlines(Window owner,
    AlgorithmOperation algoOperation) {
    if (algoOperation == null) {
      throw new IllegalArgumentException(
        getParameterNullMessage("algoOperation"));
    }
    RemoteTask<AlgorithmOutline[]> task = new RemoteTask<AlgorithmOutline[]>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmOutline[] allAlgoOutlines =
          ClientRemoteLocator.algorithmService.getAllOutlines();
        setResult(allAlgoOutlines);
      }
    };
    task.execute(owner, LNG.get("algomanager.title.consulting_data"),
      LNG.get("algomanager.msg.consulting_data"));
    AlgorithmOutline[] algoOutlines = task.getResult();

    Vector<AlgorithmOutline> outlinesList = new Vector<AlgorithmOutline>();
    Collections.addAll(outlinesList, algoOutlines);
    updateRestrictedAlgorithmOutlines(outlinesList.iterator(), owner,
      algoOperation);
    return outlinesList.toArray(new AlgorithmOutline[outlinesList.size()]);
  }

  /**
   * Obtm os identifcadores dos algoritmos restritos aos algoritmos que o
   * usurio tem permisso para realizar a operao especificada.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   *
   * @return o array de ids dos algoritmos restritos
   */
  public static Object[] getAllAlgorithmIds(Window owner,
    AlgorithmOperation algoOperation) {
    if (algoOperation == null) {
      throw new IllegalArgumentException(
        getParameterNullMessage("algoOperation"));
    }
    RemoteTask<Object[]> task = new RemoteTask<Object[]>() {
      @Override
      public void performTask() throws Exception {
        Object[] allAlgoIds = ClientRemoteLocator.algorithmService.getAllIds();
        setResult(allAlgoIds);
      }
    };
    task.execute(owner, LNG.get("algomanager.title.consulting_data"),
      LNG.get("algomanager.msg.consulting_data"));
    Object[] algoIds = task.getResult();

    Vector<Object> idsList = new Vector<Object>();
    Collections.addAll(idsList, algoIds);
    updateRestrictedAlgorithmIds(idsList.iterator(), owner, algoOperation);
    return idsList.toArray(new Object[idsList.size()]);
  }

  /**
   * Atualiza a lista dos resumos dos algoritmos de acordo com o conjunto
   * restrito de algoritmos para o usurio de acordo com a operao
   * especificada.
   *
   * @param outlineIterator iterador de resumos de algoritmos a ser restringido
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   */
  private static void updateRestrictedAlgorithmOutlines(
    Iterator<AlgorithmOutline> outlineIterator, Window owner,
    AlgorithmOperation algoOperation) {
    AlgorithmInfo[] restrictedAlgoInfos =
      getAllAlgorithmInfos(owner, algoOperation);
    Vector<Object> restrictedAlgoIds = getAlgorithmIdsList(restrictedAlgoInfos);
    while (outlineIterator.hasNext()) {
      AlgorithmOutline algoOutline = outlineIterator.next();
      if (!restrictedAlgoIds.contains(algoOutline.getId())) {
        outlineIterator.remove();
      }
    }
  }

  /**
   * Atualiza a lista dos identificadores dos algoritmos de acordo com o
   * conjunto restrito de algoritmos para o usurio de acordo com a operao
   * especificada.
   *
   * @param idsIterator iterador de identificadores de algoritmos a ser
   *        restringido
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   */
  private static void updateRestrictedAlgorithmIds(
    Iterator<Object> idsIterator, Window owner, AlgorithmOperation algoOperation) {
    AlgorithmInfo[] restrictedAlgoInfos =
      getAllAlgorithmInfos(owner, algoOperation);
    Vector<Object> restrictedAlgoIds = getAlgorithmIdsList(restrictedAlgoInfos);
    while (idsIterator.hasNext()) {
      Object algoId = idsIterator.next();
      if (!restrictedAlgoIds.contains(algoId)) {
        idsIterator.remove();
      }
    }
  }

  /**
   * Obtm a lista de identificadores de algoritmos, a partir de um conjunto de
   * informaes de algoritmos.
   *
   * @param algoInfos array contendo as informaes dos algoritmos
   * @return uma lista de identificadores de algoritmos
   */
  private static Vector<Object> getAlgorithmIdsList(AlgorithmInfo[] algoInfos) {
    Vector<Object> algoIds = new Vector<Object>();
    for (AlgorithmInfo algoInfo : algoInfos) {
      algoIds.add(algoInfo.getId());
    }
    return algoIds;
  }

  // #TODO A remover quando todos sistemas CSBase forem assinados {
  /**
   * Obtem a URL para upload do configurador.
   *
   * @param algoId identificador do algoritmo que ir receber o configurador.
   * @param versionId identificador da verso que ir receber o configurador.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return O endereo para upload.
   */
  public static String getConfiguratorURL(final Object algoId,
    final Object versionId, Window owner) {
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.retrieveConfigUploadURL(
          algoId, versionId));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    String result = task.getResult();
    if (result == null) {
      showError(LNG.get("HTTP_URL_ERROR"), null);
    }
    return result;
  }

  /**
   * Obtem a URL para upload de um executvel.
   *
   * @param algoId identificador do algoritmo que ir receber o executvel.
   * @param versionId identificador da verso que ir receber o executvel.
   * @param platform plataforma que ir receber o executvel.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return O endereo para upload.
   */
  public static String getExecutableURL(final Object algoId,
    final Object versionId, final String platform, Window owner) {
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.retrieveExecUploadURL(
          algoId, versionId, platform));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    String result = task.getResult();
    if (result == null) {
      showError(LNG.get("HTTP_URL_ERROR"), null);
    }
    return result;
  }

  /**
   * Obtem as informaes de um algoritmo.
   *
   * @param name nome do algoritmo.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   *
   * @return as informaes do algoritmo
   */
  public static AlgorithmInfo getAlgorithmInfoByName(final String name,
    Window owner, final AlgorithmOperation algoOperation) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmInfo algoInfo = AlgorithmInfo.getAlgorithmInfo(name);
        setResult(algoInfo);
        if (!isAlgorithmOwnerUser(algoInfo)) {
          // Verifica o tipo de operao que ser realizada sobre os algoritmos,
          // para checar o permissionamento adequado
          if (isAdminOperation(algoOperation)) {
            checkAlgorithmAdminPermission(algoInfo);
          }
          else if (isExecuteOperation(algoOperation)) {
            CategorySet allCategories =
              ClientRemoteLocator.algorithmService.getAllCategories();
            checkAlgorithmExecutePermission(algoInfo, allCategories);
          }
        }
      }
    };
    task.execute(owner, LNG.get("algorithm.title.getting"),
      LNG.get("algorithm.msg.getting"));
    return task.getResult();
  }

  /**
   * Obtem as informaes de um algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   *
   * @return as informaes do algoritmo
   */
  public static AlgorithmInfo getAlgorithmInfoById(final Object algoId,
    Window owner, final AlgorithmOperation algoOperation) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmInfo algoInfo = AlgorithmInfo.getAlgorithmInfo(algoId);
        setResult(algoInfo);
        if (!isAlgorithmOwnerUser(algoInfo)) {
          // Verifica o tipo de operao que ser realizada sobre os algoritmos,
          // para checar o permissionamento adequado
          if (isAdminOperation(algoOperation)) {
            checkAlgorithmAdminPermission(algoInfo);
          }
          else if (isExecuteOperation(algoOperation)) {
            CategorySet allCategories =
              ClientRemoteLocator.algorithmService.getAllCategories();
            checkAlgorithmExecutePermission(algoInfo, allCategories);
          }
        }
      }
    };
    task.execute(owner, LNG.get("algorithm.title.getting"),
      LNG.get("algorithm.msg.getting"));
    return task.getResult();
  }

  /**
   * Inclui uma plataforma para uma verso de algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * @param platform plataforma a incluir
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return informaes atualizadas do algoritmo
   */
  public static AlgorithmInfo includePlatform(final Object algoId,
    final Object versionId, final String platform, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.includePlatform(algoId,
          versionId, platform));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Aplica a recarga dos algoritmos no servidor.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos
   * @return retorna true se a recarga foi feita com sucesso, caso contrrio,
   *         retorna false
   */
  public static boolean reloadAlgorithms(Window owner) {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.reloadAlgorithms());
      }
    };
    if ((task.execute(owner, LNG.get("algorithm.title.reloading"),
      LNG.get("algorithm.msg.reloading")))
      && (task.getResult())) {
      prepareCacheReload();
      return true;
    }
    return false;
  }

  /**
   * Prepara a cache de algoritmos e categorias para ser iniciada, a fim de
   * recarregar seus dados a partir do servidor.
   */
  private static void prepareCacheReload() {
    hasAllAlgorithms = false;
    hasAllCategories = false;
  }

  /**
   * Remove um algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return <code>true</code> se a remoo teve sucesso, <code>false</code>
   *         caso contrrio
   */
  public static boolean removeAlgorithm(final Object algoId, Window owner) {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.removeAlgorithm(algoId));
      }
    };
    return (task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait")) && task
      .getResult());
  }

  /**
   * Remove arquivos executveis de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param platform Plataforma que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @param window Janela pai para efeito de hierarquia de dilogos.
   *
   * @return <code>true</code> se a remoo teve sucesso, <code>false</code>
   *         caso contrrio
   */
  public static boolean removeExecutableFiles(
    final AlgorithmVersionInfo version, final String platform,
    final FileInfo[] files, final Window window) {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        algoService.removeExecutableFiles(version, platform, files);
      }
    };
    return task.execute(window, taskTitle, LNG.get("algomanager.msg.wait"));
  }

  /**
   * Remove arquivos de configurao de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @param window Janela pai para efeito de hierarquia de dilogos.
   *
   * @return <code>true</code> se a remoo teve sucesso, <code>false</code>
   *         caso contrrio
   */
  public static boolean removeConfigurationFiles(
    final AlgorithmVersionInfo version, final FileInfo[] files,
    final Window window) {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        algoService.removeConfigurationFiles(version, files);
      }
    };
    return task.execute(window, taskTitle, LNG.get("algomanager.msg.wait"));
  }

  /**
   * Remove arquivos de documentao de uma verso de algoritmo.
   *
   * @param version Verso que detm os arquivos.
   * @param files Representao dos arquivos a serem removidos.
   * @param window Janela pai para efeito de hierarquia de dilogos.
   *
   * @return <code>true</code> se a remoo teve sucesso, <code>false</code>
   *         caso contrrio
   */
  public static boolean removeDocumentationFiles(
    final AlgorithmVersionInfo version, final FileInfo[] files,
    final Window window) {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        algoService.removeDocumentationFiles(version, files);
      }
    };
    return task.execute(window, taskTitle, LNG.get("algomanager.msg.wait"));
  }

  /**
   * Remove uma plataforma de uma verso de algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * @param platform plataforma a remover
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return informaes atualizadas do algoritmo
   */
  public static AlgorithmInfo removePlatform(final Object algoId,
    final Object versionId, final String platform, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.removePlatform(algoId,
          versionId, platform));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Remove uma verso de algoritmo.
   *
   * @param algoId identificador do algoritmo
   * @param versionId identificador da verso
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return informaes atualizadas do algoritmo
   */
  public static AlgorithmInfo removeVersion(final Object algoId,
    final Object versionId, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.removeVersion(algoId,
          versionId));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Obtm todas as informaes de algoritmos que o usurio tem permisso para
   * realizar a operao especificada.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos
   * @param algoOperation operao que ser realizada sobre os algoritmos
   *
   * @return lista de informaes de algoritmos que o usurio tem permisso para
   *         realizar a operao
   */
  public static AlgorithmInfo[] getAllAlgorithmInfos(Window owner,
    final AlgorithmOperation algoOperation) {
    if (algoOperation == null) {
      throw new IllegalArgumentException(
        getParameterNullMessage("algoOperation"));
    }
    AlgorithmInfo[] algoInfos = null;

    if (!hasAllAlgorithms) {
      algoInfos = loadAlgorithmsFromServer(owner);
      updateAlgorithmsCache(algoInfos);
    }
    else {
      algoInfos =
        allAlgorithmInfo.toArray(new AlgorithmInfo[allAlgorithmInfo.size()]);
    }
    AlgorithmInfo[] restrictedAlgorithms = algoInfos;
    if (!isAdminLoggedUser()) {
      // Verifica o tipo de operao que ser realizada sobre os algoritmos,
      // para checar o permissionamento adequado
      if (isAdminOperation(algoOperation)) {
        restrictedAlgorithms = getRestrictedAdminAlgorithms(algoInfos);
      }
      else if (isExecuteOperation(algoOperation)) {
        CategorySet allCategories = getAllCategories(owner, algoOperation);
        restrictedAlgorithms =
          getRestrictedExecuteAlgorithms(algoInfos, allCategories);
      }
    }
    return restrictedAlgorithms;
  }

  /**
   * Realiza a carga dos algoritmos a partir do servidor.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos
   * @return um array contendo o conjunto de algoritmos lidos do servidor
   */
  private static AlgorithmInfo[] loadAlgorithmsFromServer(Window owner) {
    RemoteTask<AlgorithmInfo[]> task = new RemoteTask<AlgorithmInfo[]>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.getAllInfo());
      }
    };
    // task.setProgressDialogDelay(10);
    task.execute(owner, LNG.get("algomanager.title.consulting_data"),
      LNG.get("algomanager.msg.consulting_data"));
    return task.getResult();
  }

  /**
   * Realiza a carga das categorias a partir do servidor.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos
   * @return o conjunto de categorias lidos do servidor
   */
  private static CategorySet loadCategoriesFromServer(Window owner) {
    RemoteTask<CategorySet> task = new RemoteTask<CategorySet>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.getAllCategories());
      }
    };
    task.execute(owner, LNG.get("algomanager.title.consulting_data"),
      LNG.get("algomanager.msg.consulting_data"));
    return task.getResult();
  }

  /**
   * Obtm os algoritmos que o usurio tem permisso para executar.
   *
   * Os usurios sempre tem permisso de gerenciar ou executar os algoritmos
   * criados por ele (cuja propriedade "criador" est definida).
   *
   * @param algoInfos informaes de todos os algoritmos registrados no servidor
   * @param allCategories todas as categorias do servidor
   *
   * @return um array contendo o conjunto restrito de algoritmos que o usurio
   *         tem permisso para executar
   * @throws PermissionException erro de permisso para execuo dos algoritmos
   */
  private static AlgorithmInfo[] getRestrictedExecuteAlgorithms(
    AlgorithmInfo[] algoInfos, CategorySet allCategories) {
    Vector<AlgorithmInfo> restrictedAlgorithms = new Vector<AlgorithmInfo>();
    boolean hasPermisssion = false;
    for (AlgorithmInfo algorithmInfo : algoInfos) {
      if (isAlgorithmOwnerUser(algorithmInfo)) {
        hasPermisssion = true;
      }
      else {
        try {
          checkAlgorithmExecutePermission(algorithmInfo, allCategories);
          hasPermisssion = true;
        }
        catch (PermissionException e) {
          hasPermisssion = false;
        }
        catch (Exception e) {
          hasPermisssion = false;
        }
      }
      if (hasPermisssion) {
        restrictedAlgorithms.add(algorithmInfo);
      }
    }
    return restrictedAlgorithms.toArray(new AlgorithmInfo[restrictedAlgorithms
      .size()]);
  }

  /**
   * Obtm os algoritmos que o usurio tem permisso para gerenciar.
   *
   * Os usurios sempre tem permisso de gerenciar ou executar os algoritmos
   * criados por ele (cuja propriedade "criador" est definida).
   *
   * @param algoInfos informaes de todos os algoritmos registrados no servidor
   *
   * @return um array contendo o conjunto restrito de algoritmos que o usurio
   *         tem permisso para gerenciar
   */
  private static AlgorithmInfo[] getRestrictedAdminAlgorithms(
    AlgorithmInfo[] algoInfos) {
    Vector<AlgorithmInfo> restrictedAlgorithms = new Vector<AlgorithmInfo>();
    for (AlgorithmInfo algorithmInfo : algoInfos) {
      if (isAlgorithmOwnerUser(algorithmInfo)
        || hasAlgorithmAdminPermission(algorithmInfo)) {
        restrictedAlgorithms.add(algorithmInfo);
      }
    }
    return restrictedAlgorithms.toArray(new AlgorithmInfo[restrictedAlgorithms
      .size()]);
  }

  /**
   * Obtm uma url para permitir o download de um arquivo qualquer a partir da
   * rvore de algoritmos para o disco rgido do usurio.
   *
   * @param algoId identificador do algoritmo ao qual pertence o arquivo.
   * @param filePath caminho para o arquivo.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return url para download de um arquivo qualquer.
   */
  public static String retrieveDownloadURL(final Object algoId,
    final String[] filePath, Window owner) {
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        setResult(algoService.retrieveDownloadURL(algoId, filePath));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  // #TODO A remover quando todos sistemas CSBase forem assinados {
  /**
   * Obtm uma url para permitir o upload de um arquivo html a partir do disco
   * rgido do usurio para a rvore de algoritmos.
   *
   * @param algoId identificador do algoritmo que receber o arquivo.
   * @param versionId identificador da verso que receber o arquivo.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return url para upload de um arquivo html.
   */
  public static String retrieveDocUploadURL(final Object algoId,
    final Object versionId, Window owner) {
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        setResult(algoService.retrieveDocUploadURL(algoId, versionId));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  // #TODO } A remover quando todos sistemas CSBase forem assinados

  /**
   * Obtm o histrico do caminho especificado.
   *
   * @param path caminho cujo histrico deseja-se obter.
   *
   * @return <code>List</code> contendo instncias de <code>HistoryRecord</code>
   *         , representando o histrico do caminho especificado.
   */
  public static List<HistoryRecord> retrieveHistory(final String[] path) {
    RemoteTask<List<HistoryRecord>> task =
      new RemoteTask<List<HistoryRecord>>() {
        @Override
        public void performTask() throws Exception {
          AlgorithmServiceInterface algoService =
            ClientRemoteLocator.algorithmService;
          setResult(algoService.retrieveHistory(path));
        }
      };
    JFrame owner = DesktopFrame.getInstance().getDesktopFrame();
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Mtodo de aviso de erro
   *
   * @param msg mensagem a ser exibida.
   * @param e exceo causadora do erro (ou null)
   */
  private static void showError(String msg, Exception e) {
    final DesktopFrame mainFrame = DesktopFrame.getInstance();
    if (mainFrame == null) {
      return;
    }
    JFrame jFrame = mainFrame.getDesktopFrame();
    if (jFrame == null) {
      return;
    }
    String title =
      LNG.get("algomanager.title.error") + " - " + LNG.get("algomanager.title");
    if (e != null) {
      StandardErrorDialogs.showErrorDialog(jFrame, title, msg, e);
    }
    else {
      StandardErrorDialogs.showErrorDialog(jFrame, title, msg);
    }
  }

  /**
   * Altera as propriedades estendidas de um algorimto.
   *
   * @param name nome do algoritmo
   * @param newValues novos valores para os atributos
   * @param owner janela pai para efeito da hierarquia de dialogos
   *
   * @return AlgorithmInfo alterado
   */
  public static AlgorithmInfo changeAlgorithmProperties(final String name,
    final Hashtable<String, String> newValues, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService
          .changeAlgorithmProperties(name, newValues));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Altera as propriedades estendidas de uma versao de algoritmo
   *
   * @param name nome do algoritmo
   * @param versionId id da versao a ser alterada
   * @param newValues novos valores para os atributos
   * @param owner janela pai para efeito da hierarquia de dialogos
   *
   * @return AlgorithmVersionInfo alterado
   */
  public static AlgorithmInfo changeVersionProperties(final String name,
    final AlgorithmVersionId versionId,
    final Hashtable<String, String> newValues, Window owner) {
    RemoteTask<AlgorithmInfo> task = new RemoteTask<AlgorithmInfo>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.algorithmService.changeVersionProperties(
          name, versionId, newValues));
      }
    };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Retorna a lista de propriedades da verso do algoritmo. Cada property
   * mapeia a chave da propriedade ao label (String de Apresentao).
   *
   * @param owner .
   *
   * @return lista de propriedades da verso do algoritmo (na ordem correta)
   */
  public static List<AlgorithmProperty> getAlgorithmVersionProperties(
    Window owner) {
    RemoteTask<List<AlgorithmProperty>> task =
      new RemoteTask<List<AlgorithmProperty>>() {
        @Override
        public void performTask() throws Exception {
          AlgorithmServiceInterface algoService =
            ClientRemoteLocator.algorithmService;
          setResult(algoService.getVersionProperties());
        }
      };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  /**
   * Retorna a lista de propriedades do algoritmo. Cada property mapeia a chave
   * da propriedade ao label (String de Apresentao).
   *
   * @param owner .
   *
   * @return lista de propriedades do algoritmo (na ordem correta)
   */
  public static List<AlgorithmProperty> getAlgorithmProperties(Window owner) {
    RemoteTask<List<AlgorithmProperty>> task =
      new RemoteTask<List<AlgorithmProperty>>() {
        @Override
        public void performTask() throws Exception {
          AlgorithmServiceInterface algoService =
            ClientRemoteLocator.algorithmService;
          setResult(algoService.getAlgorithmProperties());
        }
      };
    task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"));
    return task.getResult();
  }

  // #TODO A remover quando todos sistemas CSBase forem assinados {
  /**
   * Obtem a URL para upload de um pacote de verso.
   *
   * @param algoId identificador do algoritmo que ir receber a verso.
   * @param owner janela pai para efeito de hierarquia de dilogos.
   *
   * @return O endereo para upload ou {@code null} se houver um erro.
   */
  public static String getVersionURL(final Object algoId, Window owner) {
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        AlgorithmServiceInterface algoService =
          ClientRemoteLocator.algorithmService;
        if (algoService != null) {
          setResult(algoService.retrieveVersionUploadURL(algoId));
        }
      }
    };
    if (!task.execute(owner, taskTitle, LNG.get("algomanager.msg.wait"))) {
      return null;
    }
    Object result = task.getResult();
    if (result == null) {
      showError(LNG.get("HTTP_URL_ERROR"), null);
    }
    return (String) result;
  }

  /**
   * Obtm um o conjunto de categorias restrito aos algoritmos que o usurio tem
   * permisso de executar.
   *
   * @param categories conjunto de categorias a ser manipulado
   * @param algoOperation operao a ser realizada sobre os algoritmos da
   *        categoria
   *
   * @return o conjunto de categorias restrito aos algoritmos que o usurio tem
   *         permisso de executar
   */
  private static CategorySet getRestrictedCategories(
    SortedSet<Category> categories, AlgorithmOperation algoOperation) {
    CategorySet restrictedCategorySet = new CategorySet();
    if (categories != null) {
      for (Category category : categories) {
        Category firstLevelCategory = category;
        if (category.getParentCategory() != null) {
          firstLevelCategory = category.getRootCategory();
        }
        Category restrictedCategory = new Category(firstLevelCategory);
        updateRestrictedAlgorithms(restrictedCategory, algoOperation);
        updateRestrictedCategories(restrictedCategory, restrictedCategory
          .getCategories().iterator(), algoOperation);

        Category modifiedCategory = restrictedCategory;
        if (category != firstLevelCategory) {
          modifiedCategory = restrictedCategory.getCategory(category.getId());
        }
        restrictedCategorySet.addCategory(modifiedCategory);
      }
    }
    return restrictedCategorySet;
  }

  /**
   * Verifica se o usurio que solicitante do servio  o mesmo que criou o
   * algoritmo especificado.
   *
   * @param algoInfo informaes do algoritmo
   * @return retorna true, se o usurio que solicitou o servio  o mesmo que
   *         criou o algoritmo especificado, caso contrrio, retorna false
   */
  private static boolean isAlgorithmOwnerUser(AlgorithmInfo algoInfo) {
    String algorithmOwner = algoInfo.getOwner();
    User user = User.getLoggedUser();
    if (isAdminLoggedUser()
      || (algorithmOwner != null && algorithmOwner.equals(user.getLogin()))) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se o usurio logado  o usurio admin.
   *
   * @return retorna true se o usurio logado for o admin, caso contrrio,
   *         retorna false
   */
  private static boolean isAdminLoggedUser() {
    User user = User.getLoggedUser();
    return user.isAdmin();
  }

  /**
   * Atualiza o conjunto de algoritmos de uma categoria, para o conjunto
   * restrito de algoritmos que o usurio tem permisso para executar.
   *
   * @param category categoria que vai ter seu conjunto de algoritmos restrito
   * @param algoOperation operao a ser realizada sobre os algoritmos da
   *        categoria
   *
   */
  private static void updateRestrictedAlgorithms(Category category,
    AlgorithmOperation algoOperation) {
    Set<AlgorithmInfo> algorithms = category.getAlgorithms();
    Iterator<AlgorithmInfo> algorithmIterator = algorithms.iterator();
    while (algorithmIterator.hasNext()) {
      AlgorithmInfo algoInfo = algorithmIterator.next();
      if (isAlgorithmOwnerUser(algoInfo)) {
        continue;
      }
      if (isAdminOperation(algoOperation)) {
        if (!hasAlgorithmAdminPermission(algoInfo)) {
          algorithmIterator.remove();
        }
      }
      else if (isExecuteOperation(algoOperation)
        && !hasAlgorithmExecutePermission(algoInfo)
        && !hasAlgorithmExecutePermissionByCategory(algoInfo,
          category.getFullName())) {
        algorithmIterator.remove();
      }
    }
  }

  /**
   * Atualiza recursivamente o subconjunto de categorias de uma categoria, para
   * que cada categoria contenha o conjunto restrito de algoritmos que o usurio
   * tem permisso para executar.
   *
   * @param restrictedCategory categoria que contm um conjunto restrito de
   *        algoritmos
   * @param catIterator subcategorias da categoria que tambm devem ser
   *        atualizadas para conter um conjunto restrito de algoritmos
   * @param algoOperation operao a ser realizada sobre os algoritmos da
   *        categoria
   *
   */
  private static void updateRestrictedCategories(Category restrictedCategory,
    Iterator<Category> catIterator, AlgorithmOperation algoOperation) {
    SortedSet<Category> newCategories = new TreeSet<Category>();
    while (catIterator.hasNext()) {
      Category category = catIterator.next();
      Category newCategory = new Category(category);
      newCategory.setParentCategory(restrictedCategory);
      catIterator.remove();
      newCategories.add(newCategory);
      updateRestrictedAlgorithms(newCategory, algoOperation);
      updateRestrictedCategories(newCategory, newCategory.getCategories()
        .iterator(), algoOperation);
    }
    restrictedCategory.setCategories(newCategories);
  }

  /**
   * Verifica se  o administrador ou, caso contrrio, se o usurio tem
   * permisso para gerenciar o algoritmo especificado no servidor corrente.
   *
   * @param algoInfo informaes do algoritmo.
   * @throws Exception erro ocorrido durante a busca pela permisso
   */
  private static void checkAlgorithmAdminPermission(AlgorithmInfo algoInfo)
    throws Exception {
    if (algoInfo == null) {
      throw new IllegalArgumentException(getParameterNullMessage("algoInfo"));
    }

    if (isAdminLoggedUser()) {
      return;
    }
    String algoAttribute =
      AlgorithmAdminPermission.ALGORITHM_UPDATE + algoInfo.getName();

    User user = User.getLoggedUser();
    AlgorithmAdminPermission ap =
      (AlgorithmAdminPermission) user.getMatchAttributesPermission(
        AlgorithmAdminPermission.class, algoAttribute);
    if (ap == null) {
      throw new PermissionException(
        getAlgoAdminPermissionErrorMessage(algoInfo.getName()));
    }
    String serverName = getServerName();
    String name = AdminPermission.LOCAL + serverName;
    if (ap.getMatchAttribute(name) == null) {
      throw new PermissionException(
        getAlgoAdminServerPermissionErrorMessage(serverName));
    }
  }

  /**
   * Verifica se o usurio tem permisso para execuo do algoritmo (se for um
   * algoritmo simples), ou verifica se o usurio possui permisso para executar
   * todos os algoritmos que fazem parte do configurador (no caso de ser um
   * fluxo).
   *
   * Se o usurio for administrador, ele tem permisso sempre.
   *
   * @param algoInfo informaes do algoritmo ou do fluxo
   * @param categorySet conjunto de categorias de algoritmos
   * @throws Exception exceo
   *
   * @throws PermissionException erro de permisso para execuo do algoritmo
   * @throws Exception erro ocorrido durante a busca pela permisso
   */
  private static void checkAlgorithmExecutePermission(AlgorithmInfo algoInfo,
    CategorySet categorySet) throws Exception {
    if (algoInfo == null) {
      throw new IllegalArgumentException(getParameterNullMessage("algoInfo"));
    }

    if (isAdminLoggedUser()) {
      return;
    }

    String systemId = null; // Sistema corrente
    String algorithmName = algoInfo.getName();

    List<String> categoriesFullNames =
      categorySet.getAlgorithmCategoriesFullNames(algoInfo);
    if (AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
      User.getLoggedUser(), systemId, algorithmName)) {
      return;
    }
    // Verifica se tem alguma permisso para as categorias em que o
    // algoritmo faz parte
    if (CategoryAlgorithmsExecutionPermission
      .checkSystemAndCategoriesExecPermission(User.getLoggedUser(), systemId,
        categoriesFullNames)) {
      return;
    }
    throw new PermissionException(
      getAlgoExecutePermissionErrorMessage(algorithmName));
  }

  /**
   * Verifica se o usurio tem permisso para gerenciar o algoritmo
   * especificado.
   *
   * @param algoInfo informaes do algoritmo procurado
   *
   * @return retorna true, se o usurio tem permisso para gerenciar o
   *         algoritmo, caso contrrio, retorna false
   *
   */
  private static boolean hasAlgorithmAdminPermission(AlgorithmInfo algoInfo) {
    if (algoInfo == null) {
      throw new IllegalArgumentException(getParameterNullMessage("algoInfo"));
    }

    boolean hasPermission = true;

    if (!isAdminLoggedUser()) {
      try {
        checkAlgorithmAdminPermission(algoInfo);
      }
      catch (Exception e) {
        hasPermission = false;
      }
    }
    return hasPermission;
  }

  /**
   * Verifica se o usurio tem uma permisso direta para executar o algoritmo
   * especificado.
   *
   * @param algoInfo informaes do algoritmo procurado
   *
   * @return retorna true, se o usurio tem permisso para executar o algoritmo,
   *         caso contrrio, retorna false
   *
   */
  private static boolean hasAlgorithmExecutePermission(AlgorithmInfo algoInfo) {
    if (algoInfo == null) {
      throw new IllegalArgumentException(getParameterNullMessage("algoInfo"));
    }
    boolean hasPermission = true;
    try {
      if (isAdminLoggedUser() || isAlgorithmOwnerUser(algoInfo)) {
        return true;
      }
      String systemId = null; // Sistema corrente

      if (AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(
        User.getLoggedUser(), systemId, algoInfo.getName())) {
        hasPermission = true;
      }
      else {
        hasPermission = false;
      }
    }
    catch (Exception e) {
      hasPermission = false;
    }
    return hasPermission;
  }

  /**
   * Verifica se o usurio tem permisso para executar um determinado algoritmo
   * a partir da categoria especificada.
   *
   * @param algoInfo informaes do algoritmo a ser executado
   * @param categoryFullName nome completo da categoria que contm o algoritmo a
   *        ser executado
   *
   * @return retorna true, se o usurio tem permisso para executar o algoritmo,
   *         caso contrrio, retorna false
   *
   */
  private static boolean hasAlgorithmExecutePermissionByCategory(
    AlgorithmInfo algoInfo, String categoryFullName) {
    boolean hasPermission = true;
    try {
      if (isAdminLoggedUser() || isAlgorithmOwnerUser(algoInfo)) {
        return true;
      }

      String systemId = null; // Sistema corrente
      if (CategoryAlgorithmsExecutionPermission
        .checkSystemAndCategoryExecPermission(User.getLoggedUser(), systemId,
          categoryFullName)) {
        hasPermission = true;
      }
      else {
        hasPermission = false;
      }
    }
    catch (Exception e) {
      hasPermission = false;
    }
    return hasPermission;
  }

  /**
   * Obtm mensagem de erro para o caso do parmetro de um mtodo ser nulo.
   *
   * @param parameter parmetro nulo
   *
   * @return a mensagem de erro correspondente
   */
  private static String getParameterNullMessage(String parameter) {
    String msg = LNG.get("AlgorithmManagementProxy.error.parameter.null");
    Object[] args = new Object[] { parameter };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * executar o algoritmo com o nome especificado.
   *
   * @param algoName parmetro invlido
   *
   * @return a mensagem de erro correspondente
   */
  private static String getAlgoExecutePermissionErrorMessage(String algoName) {
    String msg =
      LNG.get("AlgorithmManagementProxy.error.algo.execute.no_permission");
    Object[] args = new Object[] { algoName };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * administrar o algoritmo com o nome especificado.
   *
   * @param algoName parmetro invlido
   *
   * @return a mensagem de erro correspondente
   */
  private static String getAlgoAdminPermissionErrorMessage(String algoName) {
    String msg =
      LNG.get("AlgorithmManagementProxy.error.algo.admin.no_permission");
    Object[] args = new Object[] { algoName };
    return MessageFormat.format(msg, args);
  }

  /**
   * Obtm mensagem de erro para o caso do usurio no ter permisso para
   * administrar algoritmos no servidor com o nome especificado.
   *
   * @param serverName parmetro invlido
   *
   * @return a mensagem de erro correspondente
   */
  private static String getAlgoAdminServerPermissionErrorMessage(
    String serverName) {
    String msg =
      LNG.get("AlgorithmManagementProxy.error.algo.admin.server.no_permission");
    Object[] args = new Object[] { serverName };
    return MessageFormat.format(msg, args);
  }

  /**
   * Mapeia os algoritmos s categorias associadas.
   */
  private static void mapAlgorithmsToCategories() {
    algorithmToCategoryMap.clear();

    for (Category category : categorySet.getAllCategories()) {
      Iterator<AlgorithmInfo> algorithmIterator = allAlgorithmInfo.iterator();
      while (algorithmIterator.hasNext()) {
        AlgorithmInfo algo = algorithmIterator.next();
        if (category.containsAlgorithm(algo)) {
          if (!algorithmToCategoryMap.containsKey(algo)) {
            algorithmToCategoryMap.put(algo, new TreeSet<Category>());
          }
          algorithmToCategoryMap.get(algo).add(category);
        }
      }
    }
  }

  /**
   * Mapeia os algoritmos da categoria especificada. Isso geralmente deve ser
   * feito quando uma nova categoria  adicionada no sistema.
   *
   * @param category categoria cujos algoritmos devem ser mapeados
   */
  private static void mapCategoryAlgorithms(Category category) {
    Set<AlgorithmInfo> algorithms = category.getAlgorithms();
    for (AlgorithmInfo algorithmInfo : algorithms) {
      if (!algorithmToCategoryMap.containsKey(algorithmInfo)) {
        algorithmToCategoryMap.put(algorithmInfo, new TreeSet<Category>());
      }
      algorithmToCategoryMap.get(algorithmInfo).add(category);
    }
    mapSubCategoriesAlgorithms(category.getCategories());
  }

  /**
   * Desmapeia os algoritmos da categoria especificada. Isso geralmente deve ser
   * feito quando uma nova categoria  removida do sistema, ou tem a lista dos
   * seus algoritmos alterada.
   *
   * @param categoryId identificador da categoria cujos algoritmos devem ser
   *        desmapeados
   */
  private static void unmapCategoryAlgorithms(String categoryId) {
    Category category = categorySet.getCategory(categoryId);
    Set<AlgorithmInfo> algorithms = category.getAlgorithms();
    for (AlgorithmInfo algorithmInfo : algorithms) {
      if (algorithmToCategoryMap.containsKey(algorithmInfo)) {
        SortedSet<Category> categoryset =
          algorithmToCategoryMap.get(algorithmInfo);
        categoryset.remove(category);
      }
    }
    unmapSubCategoriesAlgorithms(category.getCategories());
  }

  /**
   * Mapeia os algoritmos das sub-categorias especificadas.
   *
   * @param subCategories sub-categorias cujos algoritmos devem ser mapeados
   */
  private static void mapSubCategoriesAlgorithms(
    SortedSet<Category> subCategories) {
    for (Category subCat : subCategories) {
      mapCategoryAlgorithms(subCat);
    }
  }

  /**
   * Desmapeia os algoritmos das sub-categorias especificadas.
   *
   * @param subCategories sub-categorias cujos algoritmos devem ser desmapeados
   */
  private static void unmapSubCategoriesAlgorithms(
    SortedSet<Category> subCategories) {
    for (Category subCat : subCategories) {
      unmapCategoryAlgorithms(subCat.getId());
    }
  }

  /**
   * Mapeia as categorias em que o algoritmo especificado faz parte. Isso
   * geralmente deve ser feito quando um novo algoritmo  adicionada no sistema,
   * ou quando so modificadas as categorias em que o algoritmo faz parte.
   *
   * @param algoInfo algoritmo cujas categorias devem ser mapeadas
   */
  private static void mapAlgorithmCategories(AlgorithmInfo algoInfo) {
    if (!algorithmToCategoryMap.containsKey(algoInfo)) {
      algorithmToCategoryMap.put(algoInfo, new TreeSet<Category>());
    }
  }

  /**
   * Desmapeia as categorias em que o algoritmo especificado faz parte. Isso
   * geralmente deve ser feito quando um algoritmo  removido do sistema, ou
   * quando so modificadas as categorias em que o algoritmo faz parte.
   *
   * @param algoInfo algoritmo cujas categorias devem ser desmapeadas
   */
  private static void unmapAlgorithmCategories(AlgorithmInfo algoInfo) {
    if (algorithmToCategoryMap.containsKey(algoInfo)) {
      SortedSet<Category> categories = algorithmToCategoryMap.get(algoInfo);
      for (Category category : categories) {
        category.removeAlgorithm(algoInfo);
      }
      algorithmToCategoryMap.remove(algoInfo);
    }
  }

  // /**
  // * Obtm o conjunto de categorias carregadas do servidor.
  // *
  // * @return o conjunto de categorias de algoritmos
  // */
  // public static SortedSet<Category> getCategories() {
  // return categorySet.getCategories();
  // }

  /**
   * Obtm as categorias associadas a um determinado algoritmo.
   *
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param algoOperation operao que ser realizada sobre os algoritmos
   * @param algorithmInfo algoritmo procurado
   *
   * @return as categorias associadas ao algoritmo, ou null, caso no tenha
   *         nenhuma categoria associada
   */
  public static SortedSet<Category> getAlgorithmCategories(Window owner,
    AlgorithmOperation algoOperation, AlgorithmInfo algorithmInfo) {
    SortedSet<Category> categories = algorithmToCategoryMap.get(algorithmInfo);
    if (isExecuteOperation(algoOperation) && !isAdminLoggedUser()) {
      return getRestrictedCategories(categories, algoOperation).getCategories();
    }
    return categories;
  }

  /**
   * Verifica se a operao sobre o algoritmo  de execuo.
   *
   * @param algoOperation operao sobre o algoritmo
   * @return retorna true se a operao sobre o algoritmo for de execuo, caso
   *         contrrio, retorna false
   */
  private static boolean isExecuteOperation(AlgorithmOperation algoOperation) {
    return algoOperation.equals(AlgorithmOperation.EXECUTE_ALGORITHM);
  }

  /**
   * Verifica se a operao sobre o algoritmo  de administrao.
   *
   * @param algoOperation operao sobre o algoritmo
   * @return retorna true se a operao sobre o algoritmo for de administrao,
   *         caso contrrio, retorna false
   */
  private static boolean isAdminOperation(AlgorithmOperation algoOperation) {
    return algoOperation.equals(AlgorithmOperation.ADMIN_ALGORITHM);
  }

  /**
   * Atualiza os algoritmos da aplicao a partir do servidor.
   *
   * @param algoInfos informaes dos algoritmos
   */
  public static void updateAlgorithmsCache(AlgorithmInfo[] algoInfos) {
    allAlgorithmInfo.clear();
    for (AlgorithmInfo algorithmInfo : algoInfos) {
      allAlgorithmInfo.add(algorithmInfo);
    }
    mapAlgorithmsToCategories();
    hasAllAlgorithms = true;
  }

  /**
   * Atualiza os dados da cache de categorias e refaz o mapeamento de algoritmos
   * e categorias.
   *
   * @param categorySet novo conjunto de categorias lidos do servidor
   */
  public static void updateCategoriesCache(CategorySet categorySet) {
    AlgorithmManagementProxy.categorySet = categorySet;
    mapAlgorithmsToCategories();
    hasAllCategories = true;
  }

  /**
   * Remove do cache o algoritmo especificado.
   *
   * @param algoInfo algoritmo removido
   */
  private static void removeAlgorithm(AlgorithmInfo algoInfo) {
    allAlgorithmInfo.remove(algoInfo);
    unmapAlgorithmCategories(algoInfo);
  }

  /**
   * Cria ou substitui no cache o algoritmo especificado.
   *
   * @param algoInfo algoritmo criado
   */
  private static void createAlgorithm(AlgorithmInfo algoInfo) {
    allAlgorithmInfo.add(algoInfo);
    mapAlgorithmCategories(algoInfo);
  }

  /**
   * Atualiza no cache o algoritmo especificada.
   *
   * @param algoInfo algoritmo alterado
   */
  private static void updateAlgorithm(AlgorithmInfo algoInfo) {
    removeAlgorithm(algoInfo);
    createAlgorithm(algoInfo);
  }

  /**
   * Remove do cache a categoria especificada.
   *
   * @param categoryId identificador da categoria a ser removida
   */
  private static void removeCategory(String categoryId) {
    Category category = categorySet.getCategory(categoryId);
    if (category == null) {
      return;
    }
    unmapCategoryAlgorithms(category.getId());
    Category parentCategory = null;
    parentCategory = category.getParentCategory();
    if (parentCategory == null) {
      categorySet.removeCategory(categoryId);
    }
    else {
      // Substituir na categoria pai
      parentCategory.removeCategory(category);
    }
  }

  /**
   * Cria ou substitui no cache a categoria especificada.
   *
   * @param category categoria a ser includa ou substituda
   */
  private static void createCategory(Category category) {
    Category parentCategory = null;
    parentCategory = category.getParentCategory();
    if (parentCategory == null) {
      removeCategory(category.getId());
      categorySet.addCategory(category);
      mapCategoryAlgorithms(category);
    }
    else {
      // Substituir o ramo completo acima da categoria sendo criada, pois o
      // pai de cada uma mudou no servidor
      Category rootCreatedCategory = getRootCategory(parentCategory);
      Category oldCategory =
        categorySet.getRootCategory(rootCreatedCategory.getId());
      if (oldCategory != null) {
        removeCategory(oldCategory.getId());
      }
      categorySet.addCategory(rootCreatedCategory);
      mapCategoryAlgorithms(rootCreatedCategory);
    }
  }

  /**
   * Cria ou substitui no cache a categoria especificada.
   *
   * @param category categoria a ser includa ou substituda
   */
  private static void updateCategory(Category category) {
    createCategory(category);
  }

  /**
   * Atualiza no cache as categorias modificadas.
   *
   * @param modifiedCategorySet conjunto com as categorias modificadas
   */
  private static void updateCategories(CategorySet modifiedCategorySet) {
    for (Category category : modifiedCategorySet.getCategories()) {
      updateCategory(category);
    }
  }

  /**
   * Obtm a categoria correspondente  categoria raiz (de mais alto nvel) na
   * hierarquia da categoria especificada.
   *
   * @param category categoria procurada
   *
   * @return a categoria raiz (de mais alto nvel) na hierarquia da categoria
   *         especificada
   */
  private static Category getRootCategory(Category category) {
    Category parent = category.getParentCategory();
    if (parent == null) {
      return category;
    }
    return getRootCategory(parent);
  }

  /**
   * Adiciona um listener para mudanas ocorridas no repositrio de algoritmos e
   * no conjunto de categorias no servidor.
   *
   * @param listener listener a ser adicionado
   */
  public static void addManagementListener(AlgorithmManagementListener listener) {
    if (!managementList.contains(listener)) {
      managementList.add(listener);
    }
  }

  /**
   * Remove um listener do proxy.
   *
   * @param listener listener
   */
  public static void removeManagementListener(
    AlgorithmManagementListener listener) {
    managementList.remove(listener);
  }

  /**
   * Notifica os listeners da aplicao que uma categoria foi criada.
   *
   * @param category categoria criada ou a categoria raiz que recebeu a nova
   *        categoria
   *
   */
  private static void notifyCategoryCreated(Category category) {
    for (AlgorithmManagementListener listener : managementList) {
      listener.categoryCreated(category);
    }
  }

  /**
   * Notifica os listeners da aplicao que uma categoria foi removida.
   *
   * @param category categoria removida
   */
  private static void notifyCategoryRemoved(Category category) {
    for (AlgorithmManagementListener listener : managementList) {
      listener.categoryRemoved(category);
    }
  }

  /**
   * Notifica os listeners da aplicao que um conjunto de categorias foi
   * alterada.
   *
   * @param modifiedCategorySet conjunto com as categorias modificadas
   */
  private static void notifyCategoryUpdated(CategorySet modifiedCategorySet) {
    for (AlgorithmManagementListener listener : managementList) {
      listener.categoryUpdated(modifiedCategorySet);
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi criado.
   *
   *
   */
  private static void notifyAlgorithmsReloaded() {
    for (AlgorithmManagementListener listener : managementList) {
      listener.algorithmsReloaded();
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi criado.
   *
   * @param algoInfo algoritmo criado
   *
   */
  private static void notifyAlgorithmCreated(AlgorithmInfo algoInfo) {
    for (AlgorithmManagementListener listener : managementList) {
      listener.algorithmCreated(algoInfo);
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi removido.
   *
   * @param algoInfo algoritmo removido
   *
   */
  private static void notifyAlgorithmRemoved(AlgorithmInfo algoInfo) {
    for (AlgorithmManagementListener listener : managementList) {
      listener.algorithmRemoved(algoInfo);
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi alterado.
   *
   * @param algoInfo algoritmo alterado
   *
   */
  private static void notifyAlgorithmUpdated(AlgorithmInfo algoInfo) {
    for (AlgorithmManagementListener listener : managementList) {
      listener.algorithmUpdated(algoInfo);
    }
  }

  /**
   * Atualiza a representao da rvore, sempre que for recebida uma notificao
   * de alterao no servidor.
   *
   * @param event evento ocorrido no servio de algoritmos
   *
   * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
   */
  public static void handleCategoryEvent(final AlgoEvent event) {
    final Category category =
      (event.item instanceof Category) ? (Category) event.item : null;

    switch (event.type) {
      case AlgoEvent.CREATE:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            prepareCacheReload();
            //createCategory(category);
            notifyCategoryCreated(category);
          }
        });
        break;

      case AlgoEvent.DELETE:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            prepareCacheReload();
            //removeCategory(category.getId());
            notifyCategoryRemoved(category);
          }
        });
        break;

      case AlgoEvent.MODIFY:
        if (event.item instanceof CategorySet) {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              // Atualiza no cache as categorias que foram modificadas
              CategorySet modifiedCategorySet = (CategorySet) event.item;
              //updateCategories(modifiedCategorySet);
              prepareCacheReload();
              notifyCategoryUpdated(modifiedCategorySet);
            }
          });
        }
        break;

      default:
        // String msg = getString("AlgorithmsManager.msg.error.unknown_event",
        // new Object[] { new Integer(event.type) });
        // System.err.println(msg);
        break;
    }
  }

  /**
   * Atualiza os algoritmos da aplicao, sempre que for recebida uma
   * notificao de alterao no servidor.
   *
   * @param event evento ocorrido no servio de algoritmos
   *
   * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
   */
  private static void handleAlgorithmEvent(AlgoEvent event) {
    final AlgorithmInfo algoInfo =
      (event.item instanceof AlgorithmInfo) ? (AlgorithmInfo) event.item : null;
    switch (event.type) {
      case AlgoEvent.CREATE:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            prepareCacheReload();
            //createAlgorithm(algoInfo);
            notifyAlgorithmCreated(algoInfo);
          }
        });
        break;

      case AlgoEvent.DELETE:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            prepareCacheReload();
            //removeAlgorithm(algoInfo);
            notifyAlgorithmRemoved(algoInfo);
          }
        });
        break;

      case AlgoEvent.MODIFY:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            //updateAlgorithm(algoInfo);
            prepareCacheReload();
            notifyAlgorithmUpdated(algoInfo);
          }
        });
        break;

      case AlgoEvent.RELOAD:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            prepareCacheReload();
            notifyAlgorithmsReloaded();
          }
        });
        break;

      default:
        // String msg = getString("AlgorithmsManager.msg.error.unknown_event",
        // new Object[] { new Integer(event.type) });
        // System.err.println(msg);
        break;
    }
  }

  /**
   * Adiciona um observador para eventos de categorias de algoritmos que ocorram
   * no servidor.
   */
  private static void addCategoryEventsObserver() {
    Category.addObserver(new Observer() {
      @Override
      public void update(Observable o, Object arg) {
        AlgoEvent event = (AlgoEvent) arg;
        handleCategoryEvent(event);
      }
    });
  }

  /**
   * Adiciona um observador para eventos de categorias de algoritmos que ocorram
   * no servidor.
   */
  private static void addCategorySetEventsObserver() {
    CategorySet.addObserver(new Observer() {
      @Override
      public void update(Observable o, Object arg) {
        AlgoEvent event = (AlgoEvent) arg;
        handleCategoryEvent(event);
      }
    });
  }

  /**
   * Adiciona um observador para eventos de categorias de algoritmos que ocorram
   * no servidor.
   */
  private static void addAlgorithmEventsObserver() {
    AlgorithmInfo.addObserver(new Observer() {
      @Override
      public void update(Observable o, Object arg) {
        AlgoEvent event = (AlgoEvent) arg;
        handleAlgorithmEvent(event);
      }
    });
  }
}
