package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.SortedSet;
import java.util.Vector;

import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import tecgraf.javautils.gui.tree.DefaultNode;
import tecgraf.javautils.gui.tree.FilterableTreePanel;
import csbase.client.applications.Application;
import csbase.client.applications.algorithmsmanager.models.CategoryNodeInterface;
import csbase.client.applications.algorithmsmanager.models.CategoryTreeNode;
import csbase.client.applications.algorithmsmanager.models.CategoryTreeRootNode;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.logic.algorithms.Category;

/**
 * Representa uma viso da rvore de dados e seus respectivos filhos. Para
 * inserir essa viso grfica voc deve obter o painel correspondente.
 * 
 */
public class CategoryTreeView {
  /** Painel de seleo de categorias */
  private CategorySelectionPanel categorySelectionPanel;

  /** rvore de categorias de algoritmos */
  private FilterableTreePanel categoryTreePanel;

  /** Ns da rvore de dados */
  private SortedSet<Category> nodes;

  /** Nome da raiz da rvore de categorias */
  public static String ROOT_NAME;

  /** Backup de ns expandidos da rvore. */
  private Vector<TreePath> expandedTreePathList;

  /** Backupo da poro da rvore visvel dentro do ScrollPane. */
  private Rectangle visibleRectangle;

  /**
   * Indica que as informaes j foram atualizadas. Foi necessrio para tratar
   * os casos de seleo de um item da JList em que o evento de mudana chega,
   * mas  somente para o item aparecer selecionado e no precisa inicializar
   * esse elemento
   */
  private boolean alreadySelected;

  /**
   * Linha do n de categoria correntemente selecionado na rvore
   */
  private int currentCategoryRow;

  /** Listener de seleo adicionado na rvore */
  private TreeSelectionListener treeSelectionListener;

  /**
   * Constri o painel da rvore de dados.
   * 
   * @param categorySelectionPanel painel de seleo de categorias
   * @param nodes ns da rvore de dados
   */
  public CategoryTreeView(CategorySelectionPanel categorySelectionPanel,
    SortedSet<Category> nodes) {
    setNodes(nodes);
    this.categorySelectionPanel = categorySelectionPanel;
    currentCategoryRow = -1;
    ROOT_NAME =
      getApplication().getString("CategoryTreeView.root.label.categories");
  }

  /**
   * Obtm a aplicao.
   * 
   * @return a aplicao
   */
  private Application getApplication() {
    return categorySelectionPanel.getApplication();
  }

  /**
   * Estabelece o conjunto de ns da rvore de dados.
   * 
   * @param nodes ns da rvore de dados
   */
  private void setNodes(SortedSet<Category> nodes) {
    this.nodes = nodes;
  }

  /**
   * Obtm o painel com a rvore de categorias de algoritmos.
   * 
   * @return o painel com a rvore de categorias de algoritmos
   */
  public FilterableTreePanel getTreePanel() {
    if (categoryTreePanel == null) {
      categoryTreePanel = createDataTreePanel();
    }
    return categoryTreePanel;
  }

  /**
   * Constri o painel com a rvore de dados.
   * 
   * @return o painel com a rvore de dados
   */
  private FilterableTreePanel createDataTreePanel() {
    if (treeSelectionListener != null) {
      getCategoryTree().getSelectionModel().removeTreeSelectionListener(
        treeSelectionListener);
    }
    categoryTreePanel =
      new FilterableTreePanel(new CategoryTreeRootNode(nodes));
    createTreeSelectionListener();
    getCategoryTree().getSelectionModel().setSelectionMode(
      TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
    return categoryTreePanel;
  }

  /**
   * Cria o listener de seleo na rvore de categorias.
   */
  private void createTreeSelectionListener() {
    treeSelectionListener = new TreeSelectionListener() {

      @Override
      public void valueChanged(TreeSelectionEvent e) {
        Object node = e.getPath().getLastPathComponent();
        CategoryNodeInterface categoryNode = null;
        if (node instanceof CategoryNodeInterface) {
          categoryNode = (CategoryNodeInterface) node;
        }
        initCategorySelection(categoryNode);
      }
    };
    getCategoryTree().getSelectionModel().addTreeSelectionListener(
      treeSelectionListener);
  }

  /**
   * Obtm a rvore de categorias, a partir do painel com filtro que contm a
   * rvore.
   * 
   * @return a rvore de categorias
   */
  private JTree getCategoryTree() {
    return categoryTreePanel.getTree();
  }

  /**
   * Obtm o primeiro n selecionado na rvore de dados.
   * 
   * @return o primeiro n selecionado na rvore de dados
   */
  public CategoryNodeInterface getFirstSelectedNode() {
    CategoryNodeInterface node = null;
    JTree dataTree = categoryTreePanel.getTree();
    int selectedCount = dataTree.getSelectionCount();
    if (selectedCount == 1) {
      Object selectionPath = dataTree.getSelectionPath().getLastPathComponent();
      node = (CategoryNodeInterface) selectionPath;
    }
    return node;
  }

  /**
   * Obtm o primeiro n selecionado na rvore de dados.
   * 
   * @return o primeiro n selecionado na rvore de dados
   */
  public int getSelectedRow() {
    JTree dataTree = categoryTreePanel.getTree();
    int[] selectionRows = dataTree.getSelectionRows();
    if (selectionRows != null && selectionRows.length > 0) {
      return selectionRows[0];
    }
    return 0;
  }

  /**
   * Obtm a lista dos dados selecionados na rvore de dados.
   * 
   * @return retorna a lista de dados selecionados na rvore de dados. Se a
   *         seleo
   */
  public List<DataInterface> getSelectedDataList() {
    List<DataInterface> selectedDataList = new ArrayList<DataInterface>();
    JTree dataTree = categoryTreePanel.getTree();
    TreePath[] selectionPaths = dataTree.getSelectionPaths();
    if (selectionPaths != null) {
      for (TreePath treePath : selectionPaths) {
        Object treeNode = treePath.getLastPathComponent();
        if (treeNode instanceof CategoryNodeInterface) {
          selectedDataList.add((CategoryNodeInterface) treeNode);
        }
      }
    }
    return selectedDataList;
  }

  /**
   * Obtm os ns referentes ao caminho completo do n selecionado.
   * 
   * @return uma lista com os ns referentes ao caminho completo do n
   *         selecionado
   */
  public List<CategoryNodeInterface> getFullPathSelectedNode() {
    List<CategoryNodeInterface> fullPath =
      new ArrayList<CategoryNodeInterface>();
    JTree dataTree = getCategoryTree();
    int selectedCount = dataTree.getSelectionCount();
    if (selectedCount == 1) {
      Object[] paths = dataTree.getSelectionPath().getPath();
      CategoryTreeRootNode rootNode = (CategoryTreeRootNode) paths[0];
      fullPath.add(rootNode);
      for (Object path : paths) {
        if (path instanceof CategoryTreeNode) {
          CategoryTreeNode node = (CategoryTreeNode) path;
          fullPath.add(node);
        }
      }
    }
    return fullPath;
  }

  /**
   * Obtm o n raiz da rvore de dados.
   * 
   * @return retorna o n raiz da rvore de dados, caso contrrio, retorna null
   * 
   */
  public CategoryTreeRootNode getRootNode() {
    TreePath rootPath = getCategoryTree().getPathForRow(0);
    Object treeNode = rootPath.getLastPathComponent();
    if (treeNode instanceof CategoryTreeRootNode) {
      return (CategoryTreeRootNode) treeNode;
    }
    return null;
  }

  /**
   * Seleciona o n raiz da rvore de dados.
   * 
   */
  public void selectRootNode() {
    getCategoryTree().setSelectionRow(0);
  }

  /**
   * Obtm um n a partir de um caminho da rvore de dados.
   * 
   * @param path caminho completo de um n da rvore
   * @return o n correspondente a esse caminho
   */
  protected CategoryNodeInterface getNode(TreePath path) {
    if (path != null) {
      Object lastPath = path.getLastPathComponent();
      if (CategoryNodeInterface.class.isAssignableFrom(lastPath.getClass())) {
        return (CategoryNodeInterface) lastPath;
      }
    }
    return null;
  }

  /**
   * Verifica se o n raiz da rvore de dados est selecionado.
   * 
   * @return retorna true, se o n raiz da rvore de dados estiver selecionado,
   *         caso contrrio, retorna false
   * 
   */
  public boolean isRootNodeSelected() {
    JTree dataTree = getCategoryTree();
    return dataTree.isRowSelected(0);
  }

  /**
   * Obtm o nmero de itens selecionados na rvore de dados.
   * 
   * @return o nmero de itens selecionados na rvore de dados
   */
  public int getSelectionCount() {
    return getCategoryTree().getSelectionCount();
  }

  /**
   * Seleciona a categoria corrente, inicializando os dados do item selecionado.
   */
  public void selectCurrentCategory() {
    selectCurrentNode();
    initCategorySelection(getFirstSelectedNode());
  }

  /**
   * Seleciona o n categoria correspondente  categoria especificada.
   * 
   * @param category categoria a ser selecionada
   */
  public void selectCategory(Category category) {
    TreePath categoryPath = getCategoryPath(category);
    if (categoryPath == null) {
      selectCurrentCategory();
    }
    else {
      selectNode(categoryPath);
      if (!alreadySelected) {
        Object node = categoryPath.getLastPathComponent();
        if (node instanceof CategoryNodeInterface) {
          initCategorySelection((CategoryNodeInterface) node);
        }
      }
    }
  }

  /**
   * Seleciona o n da rvore que corresponde ao caminho especificado.
   * 
   * @param categoryPath caminho do n
   */
  private void selectNode(TreePath categoryPath) {
    getCategoryTree().getSelectionModel().setSelectionPath(categoryPath);
  }

  /**
   * Obtm o caminho do n categoria correspondente  categoria especificada.
   * 
   * @param category categoria procurada
   * @return o caminho do n categoria correspondente  categoria
   */
  private TreePath getCategoryPath(Category category) {
    TreeModel model = getCategoryTree().getModel();
    DefaultNode rootNode = (DefaultNode) model.getRoot();
    Vector<DefaultNode> nodePaths = new Vector<DefaultNode>();
    return getCategoryPath(category, nodePaths, rootNode);
  }

  /**
   * Obtm recursivamente o caminho do n categoria correspondente  categoria
   * especificada, fazendo uma busca recursiva por cada n filho da categoria.
   * 
   * @param category categoria procurada
   * @param nodePaths lista com cada path do caminho completo do n
   *        correspondente  categoria procurada
   * @param node n a ser verificado
   * @return o caminho do n categoria correspondente  categoria
   */
  private TreePath getCategoryPath(Category category,
    Vector<DefaultNode> nodePaths, DefaultNode node) {
    if (node.equals(DefaultMutableTreeNode.EMPTY_ENUMERATION)) {
      return null;
    }
    nodePaths.add(node);

    boolean found = false;
    Category childCategory = null;
    DefaultNode childNode = null;

    List<DefaultNode> children = node.getChildren();
    for (int index = 0; index < children.size(); index++) {
      childNode = children.get(index);
      if (childNode instanceof CategoryTreeNode) {
        childCategory = ((CategoryTreeNode) childNode).getNode();
        if (category.getId().equals(childCategory.getId())) {
          found = true;
          break;
        }
        TreePath childrenPath = getCategoryPath(category, nodePaths, childNode);
        if (childrenPath != null) {
          return childrenPath;
        }
      }
      nodePaths.remove(childNode);
    }
    if (!found) {
      return null;
    }
    nodePaths.add(childNode);

    return new TreePath(nodePaths.toArray(new Object[0]));
  }

  /**
   * Guarda uma cpia do estado atual de expanso dos ns da rvore.
   */
  public void backupExpandedNodes() {
    expandedTreePathList = new Vector<TreePath>();
    TreePath rootPath = new TreePath(getCategoryTree().getModel().getRoot());
    Enumeration<TreePath> expandedNodes =
      getCategoryTree().getExpandedDescendants(rootPath);

    // No guardar diretamente a enumerao, porque se ela  alterada
    // internamente no
    // componente com filtro que tem a JTree, no  possvel recuperar todos os
    // seus elementos corretamente

    if (expandedNodes != null) {
      while (expandedNodes.hasMoreElements()) {
        TreePath expPath = expandedNodes.nextElement();
        expandedTreePathList.add(expPath);
      }
    }
    visibleRectangle = getCategoryTree().getVisibleRect();
  }

  /**
   * Mtodo utilitrio para exibir na tela os caminhos de um determinado n da
   * rvore.
   * 
   * @param treeNode n da rvore, cujo caminho ser exibido
   */
  protected void printTreePath(TreePath treeNode) {
    if (treeNode == null) {
      return;
    }
    Object[] path = treeNode.getPath();
    System.out.println("CategoryTreeView.printTreePath()");
    System.out.println("path count: " + treeNode.getPathCount() + "  - "
      + path.length);
    for (Object pathItem : path) {
      if (pathItem instanceof CategoryTreeNode) {
        CategoryTreeNode treenode = (CategoryTreeNode) pathItem;
        System.out.println("pathItem: " + treenode.getNode());
      }
      else {
        System.out.println("pathItem - raiz: "
          + ((CategoryTreeRootNode) pathItem).getName());
      }
    }
  }

  /**
   * Restaura o estado de expanso dos ns da rvore, modificado aps alguma
   * atualizao.
   */
  public void restoreExpandedNodes() {
    if (expandedTreePathList == null) {
      return;
    }
    for (TreePath oldPath : expandedTreePathList) {
      TreePath newPath = getNewPath(oldPath);
      if (getCategoryTree().isCollapsed(newPath)) {
        getCategoryTree().expandPath(newPath);
      }
    }
    getCategoryTree().scrollRectToVisible(visibleRectangle);
  }

  /**
   * Obtm o novo path do n a ser expandido a partir do path antigo.
   * 
   * @param oldPath path antigo
   * @return novo path do n que corresponde ao path antigo
   */
  private TreePath getNewPath(TreePath oldPath) {
    Object[] oldPaths = oldPath.getPath();
    TreeModel model = getCategoryTree().getModel();
    if (oldPaths.length == 1) {
      return new TreePath(model.getRoot());
    }

    DefaultNode node = (DefaultNode) model.getRoot();
    Object[] newPaths = new Object[oldPaths.length];
    newPaths[0] = node;

    // Percorre os ns abaixo do n raiz
    for (int i = 1; i < oldPaths.length; i++) {
      DefaultNode oldNode = (DefaultNode) oldPaths[i];
      List<DefaultNode> children = node.getChildren();
      if (children.equals(DefaultMutableTreeNode.EMPTY_ENUMERATION)) {
        return null;
      }

      boolean found = false;
      Category oldCategory = null;
      Category childCategory = null;
      DefaultNode childNode = null;

      for (int index = 0; index < children.size(); index++) {
        childNode = children.get(index);
        if (oldNode instanceof CategoryTreeNode
          && childNode instanceof CategoryTreeNode) {
          oldCategory = ((CategoryTreeNode) oldNode).getNode();
          childCategory = ((CategoryTreeNode) childNode).getNode();
          if (oldCategory.getId().equals(childCategory.getId())) {
            found = true;
            break;
          }
        }
        if (childNode.toString().equals(oldPaths[i].toString())) {
          found = true;
          break;
        }
      }
      if (!found) {
        return null;
      }
      node = childNode;
      newPaths[i] = node;
    }
    return new TreePath(newPaths);
  }

  /**
   * Inicializa a seleo de uma nova categoria na rvore de categorias. O
   * painel de edio precisa ser atualizado na gerncia de categorias.
   * 
   * @param node n categoria selecionado na rvore
   */
  private void initCategorySelection(CategoryNodeInterface node) {
    if (node == null || alreadySelected) {
      categorySelectionPanel.verifyAndChangeButtonsState();
      return;
    }
    if (categorySelectionPanel.isCreationEditionPanel()
      && !categorySelectionPanel.confirmSelectionChanged()) {
      return;
    }
    else {
      if (changedSelectedCategory()) {
        if (!categorySelectionPanel.confirmSelectionChanged()) {
          setCurrentNode();
          return;
        }
        currentCategoryRow = getSelectedRow();
      }
      categorySelectionPanel.initCategoryNodeEdition(node);
    }

    categorySelectionPanel.verifyAndChangeButtonsState();
  }

  /**
   * Verifica se a verso selecinada mudou em relao  verso corrente.
   * 
   * @return retorna true se a verso selecionado for diferente do que estava
   *         correntemente selecionada, caso contrrio, retorna false
   */
  private boolean changedSelectedCategory() {
    return currentCategoryRow != getSelectedRow();
  }

  /**
   * Seleciona o n corrente, inicializando os valores do item selecionado.
   * 
   */
  private void selectCurrentNode() {
    if (currentCategoryRow == -1) {
      selectRootNode();
    }
    selectNode(currentCategoryRow);
  }

  /**
   * Seleciona uma determinada linha da rvore de dados.
   * 
   * @param row linha a ser selecionada
   * 
   */
  public void selectNode(int row) {
    getCategoryTree().setSelectionInterval(row, row);
  }

  /**
   * Estabelece o n corrente para ser o n selecionado, sem fazer nenhum tipo
   * de inicializao dos seus valores, mantendo o estado corrente do item.
   */
  public void setCurrentNode() {
    alreadySelected = true;
    selectCurrentNode();
    alreadySelected = false;
  }

  /**
   * Atualiza a rvore de categorias, a partir de um novo conjunto de
   * categorias, mantendo o estado (ns expandidos) atual da rvore.
   * 
   * @param nodes novo conjunto de categorias
   */
  public void updateCategoryTree(SortedSet<Category> nodes) {
    setNodes(nodes);
    backupExpandedNodes();
    categoryTreePanel = createDataTreePanel();
    restoreExpandedNodes();
  }

}
