/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree;

import java.awt.Window;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

import csbase.client.applications.algorithmsmanager.PropertiesListener;
import csbase.logic.algorithms.AlgorithmProperty;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * A classe <code>AlgoPropertiesPanel</code> representa o painel de edio das
 * propriedades dos algoritmos e das verses.
 */
public class PropertiesPanel {

  /** Componentes de entrada de dados das propriedades */
  private JTextField[] propertiesInputField;

  /** Configurao das propriedades do algoritmo ou verso */
  private List<AlgorithmProperty> algoProperties;

  /**
   * Valores das propriedades do algoritmo indexada pela chave
   * <code>AlgorithmProperty.key</code>
   */
  private Map<String, String> propertiesValues;

  /** Ttulo do painel */
  private String title;

  /** Janela em que o painel est colado */
  private Window owner;

  /** Painel com as propriedades */
  private JPanel mainPanel;

  /** Indica de o painel  editavel */
  private boolean editable;

  /** Listeners de mudanas nas propriedades do algoritmo */
  private List<PropertiesListener> propertiesListener;

  /**
   * Contrutor.
   *
   * @param algoProperties Configurao das propriedades do algoritmo ou verso.
   * @param propertiesValues Valores das propriedades do algoritmo indexada pela
   *        chave <code>AlgorithmProperty.key</code>
   * @param title Ttulo do painel
   * @param owner Janela em que o painel est colado
   * @param editable Indica de o painel  editavel
   */
  public PropertiesPanel(List<AlgorithmProperty> algoProperties,
    Map<String, String> propertiesValues, String title, Window owner,
    boolean editable) {
    this.algoProperties = algoProperties;
    this.propertiesValues = propertiesValues;
    this.title = title;
    this.owner = owner;
    this.editable = editable;
    this.propertiesInputField = new JTextField[algoProperties.size()];
    this.propertiesListener = new Vector<PropertiesListener>();
    this.mainPanel = buildPanel();
  }

  /**
   * Retorna o painel de propriedades.
   *
   * @return o painel de propriedades
   */
  public JPanel getPropertiesPanel() {
    return mainPanel;
  }

  /**
   * Constri o painel de entrada de dados das propriedades.
   *
   * @return o painel de entrada de dados das propriedades
   */
  private JPanel buildPanel() {
    int size = algoProperties.size();
    JComponent[][] components = new JComponent[size][2];
    for (int i = 0; i < size; i++) {
      final String propKey = algoProperties.get(i).getKey();
      final String propValue = (propertiesValues.get(propKey) == null) ? ""
        : propertiesValues.get(propKey);
      JTextField inputField = new JTextField(propValue, 30);
      inputField.setEditable(editable);
      inputField.addCaretListener(new CaretListener() {

        @Override
        public void caretUpdate(CaretEvent e) {
          notifyPropertyListener(wasModified());
        }
      });
      propertiesInputField[i] = inputField;
      components[i][0] = new JLabel(algoProperties.get(i).getLabel());
      components[i][1] = inputField;
    }
    JPanel panel = GUIUtils.createBasicGridPanel(components);
    panel.setBorder(BorderFactory.createTitledBorder(title));
    return panel;
  }

  /**
   * Indica se houve alterao nas propriedades
   *
   * @return true se houver alterao das propriedades, ou false caso nenhuma
   *         propriedade tenha sido alterada.
   */
  public boolean wasModified() {
    for (int i = 0; i < algoProperties.size(); i++) {
      AlgorithmProperty property = algoProperties.get(i);
      String newPropValue = propertiesInputField[i].getText().trim();
      String propKey = property.getKey();
      String propValue = propertiesValues.get(propKey);
      // Se a o valor existente para a propriedade for nulo e o usurio no
      // tiver preenchido, nada mudou
      if (newPropValue.length() == 0 && propValue == null) {
        continue;
      }
      if (!newPropValue.equals(propValue)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se as propriedades esto nos seus em um formato vlido. Caso no
   * estejam, exibe um dilogo com o erro e retorna false
   *
   * @return true se as propriedades esto no seu no formato vlido, ou false
   *         caso alguma no esteja em um formato vlido.
   */
  public boolean validateProperties() {
    boolean result = true;
    List<String> invalidLabels = new ArrayList<String>();
    for (int i = 0; i < algoProperties.size(); i++) {
      AlgorithmProperty algoProp = algoProperties.get(i);
      String textFieldValue = propertiesInputField[i].getText().trim();
      if (!algoProp.isValidValue(textFieldValue)) {
        result = false;
        invalidLabels.add(algoProp.getLabel());
      }
    }
    if (!result) {
      String errorTitle = LNG.get("algomanager.title.error") + " - " + title;
      StringBuilder msg = new StringBuilder();
      msg.append(LNG.get("algomanager.error.validation.properties.1"));
      msg.append("\n");
      for (String label : invalidLabels) {
        msg.append("\n  - " + label);
      }
      msg.append("\n\n");
      msg.append(LNG.get("algomanager.error.validation.properties.2"));
      StandardDialogs.showErrorDialog(owner, errorTitle, msg.toString());
    }
    return result;
  }

  /**
   * Retorna os novos valores das propriedades.
   *
   * @return os novos valores das propriedades.
   */
  public Hashtable<String, String> getProperties() {
    Hashtable<String, String> propsUpdated = new Hashtable<String, String>();
    for (int i = 0; i < algoProperties.size(); i++) {
      AlgorithmProperty property = algoProperties.get(i);
      String propKey = property.getKey();
      String newPropValue = propertiesInputField[i].getText().trim();
      if (newPropValue.length() > 0) {
        propsUpdated.put(propKey, newPropValue);
      }
    }
    return propsUpdated;
  }

  /**
   * Inicializa os campos do painel a partir de novos valores das propriedades
   * do algoritmo.
   *
   * @param propertyValues valores das propriedades do algoritmo
   */
  public void initializeFields(Hashtable<String, String> propertyValues) {
    if (propertyValues == null) {
      return;
    }
    propertiesValues = propertyValues;
    for (int i = 0; i < propertiesInputField.length; i++) {
      JTextField field = propertiesInputField[i];
      String propKey = algoProperties.get(i).getKey();
      String propValue = (propertiesValues.get(propKey) == null) ? ""
        : propertiesValues.get(propKey);
      field.setText(propValue);
    }
  }

  /**
   * Habilita ou desabilita a edio das propriedades do algoritmo.
   *
   * @param state se true, habilita a edio, caso contrrio, desabilita a
   *        edio das propriedades do algoritmo
   */
  public void enableEdition(boolean state) {
    for (int i = 0; i < propertiesInputField.length; i++) {
      JTextField field = propertiesInputField[i];
      field.setEditable(state);
    }
  }

  /**
   * Adiciona um listener de mudanas nas propriedades do algoritmo.
   *
   * @param propertyListener listener de mudanas nas propriedades do algoritmo
   */
  public void addPropertiesListener(PropertiesListener propertyListener) {
    propertiesListener.add(propertyListener);
  }

  /**
   * Remove um listener de mudanas nas propriedades do algoritmo.
   *
   * @param propertyListener listener de mudanas nas propriedades do algoritmo
   */
  public void removePropertiesListener(PropertiesListener propertyListener) {
    propertiesListener.remove(propertyListener);
  }

  /**
   * Notifica os listeners de que ocorreram mudanas nas propriedades do
   * algoritmo.
   *
   * @param wasModified se true, alguma propriedade foi alterada, caso
   *        contrrio, todos os valores das propriedades esto iguais os valores
   *        originais
   */
  private void notifyPropertyListener(boolean wasModified) {
    for (PropertiesListener propertyListener : propertiesListener) {
      propertyListener.propertiesChanged(wasModified);
    }
  }
}
