package csbase.client.applications.flowapplication.graph.utils;

import java.awt.geom.Point2D;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphFileDescriptor;
import csbase.client.applications.flowapplication.graph.GraphLink;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.graph.ParametersTransferable;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;

/**
 * Classe com mtodos utilitrios para auxiliar as aes de copiar/colar
 * parmetros e ligaes entre os ns dos grafos.
 * 
 * @author isabella
 */
public class CopyAndPasteOperation {

  /**
   * N de origem (o valor dos parmetros sero copiados dele). Pode ser nulo;
   * nesse caso s teremos o valor dos parmetros em
   * {@link CopyAndPasteOperation#parameters}.
   */
  private final GraphNode fromNode;

  /**
   * N de destino (o valor dos parmetros sero colados nele).
   */
  private final GraphNode toNode;

  /**
   * Os parmetros a serem copiados.
   */
  private final ParametersTransferable parameters;

  /**
   * Indica se as ligaes do algoritmo devem ser copiadas do n origem, caso
   * esse exista.
   */
  private boolean copyLinks;

  /**
   * Um 'backup' dos valores dos parmetros do n de destino.
   */
  private final Map<Object, Object> memento;

  /**
   * Construtor.
   * 
   * @param fromNode N de origem. (No aceita <code>null</code>)
   * @param toNode N de destino. (No aceita <code>null</code>)
   * @param copyLinks Indica se as ligaes do algoritmo devem ser copiadas do
   *        n origem.
   */
  public CopyAndPasteOperation(final GraphNode fromNode,
    final GraphNode toNode, final boolean copyLinks) {
    this(new ParametersTransferable(fromNode), fromNode, toNode, copyLinks);
  }

  /**
   * Construtor
   * 
   * @param parameters Os parmetros a serem copiados. (No aceita
   *        <code>null</code>)
   * @param toNode N de destino. (No aceita <code>null</code>)
   */
  public CopyAndPasteOperation(final ParametersTransferable parameters,
    final GraphNode toNode) {
    this(parameters, null, toNode, false);
  }

  /**
   * Construtor privado.
   * 
   * O parmetro parameters e o parmetro fromNode no podem ser nulos
   * simultaneamente.
   * 
   * @param parameters Os parmetros a serem copiados.
   * @param fromNode N de origem.
   * @param toNode N de destino. (No aceita <code>null</code>)
   * @param copyLinks Indica se as ligaes do algoritmo devem ser copiadas do
   *        n origem, caso esse exista.
   */
  protected CopyAndPasteOperation(final ParametersTransferable parameters,
    final GraphNode fromNode, final GraphNode toNode, final boolean copyLinks) {
    if (toNode == null) {
      throw new IllegalArgumentException("O parmetro toNode est nulo!");
    }
    if (fromNode == null) {
      this.copyLinks = false;

      if (parameters == null) {
        throw new IllegalArgumentException(
          "Os parmetros fromNode e parameters esto nulos!");
      }
    }
    else {
      this.copyLinks = copyLinks;
    }
    this.fromNode = fromNode;
    this.parameters = parameters;
    this.toNode = toNode;
    this.memento = getMemento();
  }

  /**
   * Retorna o estado da operao, inclusive indicando se esta pode prosseguir.
   * Se necessrio, pede a confirmao do usurio.
   * 
   * @return o estado da operao.
   */
  public OperationStatus execute() {
    OperationStatus status = paste();
    boolean confirm = showConfirmationDialogIfNeeded(status);
    if (!confirm) {
      rollbackPaste();
    }
    status.setConfirmed(confirm);
    return status;
  }

  /**
   * Obtm uma cpia de segurana dos valores dos parmetros de um n.
   * 
   * @return O mapa com o nome do parmetros do n e seus respectivos valores.
   */
  protected Map<Object, Object> getMemento() {
    final AlgorithmConfiguratorView algorithmConfiguratorView =
      toNode.getAlgorithmConfiguratorView();
    final AlgorithmConfigurator configurator =
      algorithmConfiguratorView.getConfigurator();
    return configurator.exportValues();
  }

  /**
   * "Cola" o valor da lista de parmetros em um determinado n. Caso todos os
   * parmetros existam com as mesmas propriedades no n, a operao pode ser
   * feita automaticamente. Caso alguns parmetros no existam, ou tenham
   * propriedades diferentes no n de destino, o usurio ter que confirmar a
   * operao.
   * 
   * @return o estado da operao.
   */
  protected OperationStatus paste() {
    OperationStatus status = new OperationStatus();
    if (!toNode.isWellFormed()) {
      final String errorMessage =
        MessageFormat.format(LNG.get(CopyAndPasteOperation.class.getName()
          + ".error_wrong_labels"), toNode.getAlgorithmName());
      status.addError(errorMessage);
    }

    final Set<String> parameterNames = parameters.getParameterNames();
    for (final String parameterName : parameterNames) {
      // Confere se parmetro existe no destino
      if (parameterExists(parameterName)) {
        // Confere se o tipo do parmetro  o mesmo no destino
        final String parameterType = parameters.getParameterType(parameterName);
        if (hasSameParameterType(parameterName, parameterType)) {
          // Confere se o label do parmetro  o mesmo no destino
          final String parameterLabel =
            parameters.getParameterLabel(parameterName);
          if (!hasSameParameterLabel(parameterName, parameterLabel)) {
            String newParameterLabel = "";
            try {
              newParameterLabel = toNode.getParameterLabel(parameterName);
            }
            catch (final ParameterNotFoundException e) {
              // O parmetro existe no n aqui, a exception no deve acontecer.
            }
            // S um warning  necessrio, o campo pode ser considerado o mesmo
            final String warningMessage =
              MessageFormat.format(
                LNG.get(CopyAndPasteOperation.class.getName()
                  + ".error_wrong_labels"), parameterName, parameterLabel,
                newParameterLabel);
            status.addWarning(warningMessage);
          }
          String parameterValue = null;
          try {
            parameterValue = parameters.getParameterValue(parameterName);
            toNode.setParameterValue(parameterName, parameterValue);
          }
          catch (final ParseException e) {
            final String errorMessage =
              MessageFormat.format(
                LNG.get(CopyAndPasteOperation.class.getName()
                  + ".error_pasting_value"), parameterName, parameterValue);
            status.addError(errorMessage);
          }
          catch (ParameterNotFoundException e) {
            /*
             * O parmetro j deve existir, pois j testamos isso.
             */
          }
        }
        else {
          final String errorMessage =
            MessageFormat.format(
              LNG.get(CopyAndPasteOperation.class.getName()
                + ".error_wrong_types"), parameterName);
          status.addError(errorMessage);
        }
      }
      else {
        final String warningMessage =
          MessageFormat.format(
            LNG.get(CopyAndPasteOperation.class.getName()
              + ".warning_no_parameter"), parameterName);
        status.addWarning(warningMessage);
      }
    }

    if (copyLinks && !copyLinks()) {
      final String errorMessage =
        LNG.get(CopyAndPasteOperation.class.getName() + ".error_links");
      status.addError(errorMessage);
    }

    return status;
  }

  /**
   * Compara o label de um parmetro especfico do n, com um determinado label.
   * Caso sejam diferentes,  adicionado um registro na lista de avisos.
   * 
   * @param parameterName Nome do parmetro que ser comparado
   * @param parameterLabel Label que ser comparado ao do parmetro do n
   * @return Verdadeiro se os labels so iguais ou Falso caso contrrio
   */
  protected boolean hasSameParameterLabel(final String parameterName,
    final String parameterLabel) {
    String nodeParameterLabel;
    try {
      nodeParameterLabel = toNode.getParameterLabel(parameterName);
    }
    catch (final ParameterNotFoundException e) {
      return false;
    }
    if (!nodeParameterLabel.equals(parameterLabel)) {
      return false;
    }
    return true;
  }

  /**
   * Compara o tipo de um parmetro especfico do n, com um determinado tipo.
   * Caso sejam diferentes,  adicionado um registro na lista de erros.
   * 
   * @param parameterName Nome do parmetro que ser comparado
   * @param parameterType Tipo que ser comparado ao do parmetro do n
   *        diferentes
   * @return Verdadeiro se os tipos so iguais ou Falso caso contrrio
   */
  protected boolean hasSameParameterType(final String parameterName,
    final String parameterType) {
    String nodeParameterType;
    try {
      nodeParameterType = toNode.getParameterType(parameterName);
    }
    catch (final ParameterNotFoundException e) {
      return false;
    }
    if (!nodeParameterType.equals(parameterType)) {
      return false;
    }
    return true;
  }

  /**
   * Determina se um determinado parmetro existe no n. Caso negativo, 
   * adicionado um registro na lista de avisos.
   * 
   * @param parameterName Nome do parmetro que ser testado
   * @return Verdadeiro se o parmetro existir no n ou Falso caso contrrio
   */
  protected boolean parameterExists(final String parameterName) {
    if (!toNode.getParameterNames().contains(parameterName)) {
      return false;
    }
    return true;
  }

  /**
   * Mostra uma janela com avisos e erros que so apresentados ao usurio e pede
   * uma confirmao para continuar a operao que estava sendo feita.
   * 
   * @param status o estado da operao.
   * 
   * @return verdadeiro se o usurio confirmar ou falso, caso contrrio.
   */
  protected boolean showConfirmationDialogIfNeeded(OperationStatus status) {
    final List<String> errors = status.getErrors();
    final List<String> warnings = status.getWarnings();
    if (!warnings.isEmpty() || !errors.isEmpty()) {
      final String title =
        (LNG.get(CopyAndPasteOperation.class.getName() + ".title"));
      final StringBuilder content = new StringBuilder();
      content.append("<html>");
      content.append("<b>");
      content
        .append(LNG.get(CopyAndPasteOperation.class.getName() + ".header"));
      content.append("</b>");
      if (!errors.isEmpty()) {
        content.append("<br><br>");
        content.append("<i>");
        content.append(LNG.get(CopyAndPasteOperation.class.getName()
          + ".errors"));
        content.append("</i>");

        content.append("<ul>");
        for (final String error : errors) {
          content.append("<li>" + error + "</li>");
        }
        content.append("</ul>");
      }
      if (!warnings.isEmpty()) {
        if (errors.isEmpty()) {
          content.append("<br><br>");
        }
        content.append("<i>");
        content.append(LNG.get(CopyAndPasteOperation.class.getName()
          + ".warnings"));
        content.append("</i>");

        content.append("<ul>");
        for (final String warning : warnings) {
          content.append("<li>" + warning + "</li>");
        }
        content.append("</ul>");
      }
      content.append("</html>");
      final int result =
        StandardDialogs.showYesNoDialog(toNode.getGraph().getParentWindow(),
          title, content);
      if (result == StandardDialogs.NO_OPTION) {
        return false;
      }
    }
    return true;
  }

  /**
   * Restaura o estado dos parmetros de um n, aps o cancelamento de uma
   * operao de "colar".
   */
  protected void rollbackPaste() {
    final AlgorithmConfigurator configurator =
      toNode.getAlgorithmConfiguratorView().getConfigurator();
    try {
      configurator.importValues(memento);
    }
    catch (final ParseException e) {
      StandardErrorDialogs.showErrorDialog(toNode.getGraph().getParentWindow(),
        e);
    }
  }

  /**
   * Copia todas as ligaes de um n e "cola" no outro. Os links originais so
   * removidos do grafo se a cpia for bem sucedida.
   * 
   * @return verdadeiro se todas as ligaes forem copiadas com sucesso ou
   *         falso, caso pelo menos uma no consiga ser copiada.
   */
  public boolean copyLinks() {
    final boolean result = copyFromLinks();
    return result && copyToLinks();
  }

  /**
   * Copia todas as ligaes que <b>saem</b> de um n e "cola" no outro. Os
   * links originais so removidos do grafo se a cpia for bem sucedida.
   * 
   * @return verdadeiro se todas as ligaes forem copiadas com sucesso ou
   *         falso, caso pelo menos uma no consiga ser copiada.
   */
  protected boolean copyFromLinks() {
    if (fromNode != null) {
      boolean result = true;
      final Collection<GraphLink> fromLinks = fromNode.getLinkFromCollection();
      for (final GraphLink fromLink : fromLinks) {
        final GraphNode endNode = fromLink.getEndNode();
        if (canCopyLink(fromLink, toNode, endNode)) {
          result = result && copyLink(fromLink, toNode, endNode);
        }
      }
      return result;
    }
    return false;
  }

  /**
   * Copia todas as ligaes que <b>chegam</b> em um n e "cola" no outro. Os
   * links originais so removidos do grafo se a cpia for bem sucedida.
   * 
   * @return verdadeiro se todas as ligaes forem copiadas com sucesso ou
   *         falso, caso pelo menos uma no consiga ser copiada.
   */
  protected boolean copyToLinks() {
    if (fromNode != null) {
      boolean result = true;
      final Collection<GraphLink> toLinks = fromNode.getLinkToCollection();
      for (final GraphLink toLink : toLinks) {
        final GraphNode startNode = toLink.getStartNode();
        if (canCopyLink(toLink, startNode, toNode)) {
          result = result && copyLink(toLink, startNode, toNode);
        }
      }
      return result;
    }
    return false;
  }

  /**
   * Copia o link especificado e o liga nos ns. O link original  removido do
   * grafo se a cpia for bem sucedida.
   * 
   * @param link Ligao a ser copiada
   * @param startNode N que representa a sada da ligao
   * @param endNode N que representa a entrada da ligao
   * @return verdadeiro se a ligao for copiada com sucesso ou falso, caso
   *         contrrio.
   */
  protected boolean copyLink(final GraphLink link, final GraphNode startNode,
    final GraphNode endNode) {
    try {
      final String outputParamName =
        link.getOutputFileDescriptor().getParameterName();
      final String inputParamName =
        link.getInputFileDescriptor().getParameterName();
      final GraphFileDescriptor inputFileDescriptor =
        endNode.getInputFileDescriptor(inputParamName);
      final GraphFileDescriptor outputFileDescriptor =
        startNode.getOutputFileDescriptor(outputParamName);
      if (inputFileDescriptor != null && outputFileDescriptor != null) {
        final List<Point2D> pointList = link.getIntermediatePointList();
        link.deattach();
        final Graph graph = startNode.getGraph();
        graph.createGraphLink(link.getId(), inputFileDescriptor,
          outputFileDescriptor, pointList);
        return true;
      }
    }
    catch (final Exception e) {
      // No h como tratar.
    }
    return false;
  }

  /**
   * Indica se uma ligao de um n poderia ser copiada automaticamente para
   * outro.
   * 
   * @param link Ligao a ser copiada.
   * @param startNode N origem.
   * @param endNode N destino.
   * @return verdadeiro se a ligao puder ser copiada automaticamente ou falso,
   *         caso contrrio.
   */
  protected boolean canCopyLink(final GraphLink link,
    final GraphNode startNode, final GraphNode endNode) {
    if (startNode == null || endNode == null || link == null) {
      return false;
    }
    final String outputParamName =
      link.getOutputFileDescriptor().getParameterName();
    final String inputParamName =
      link.getInputFileDescriptor().getParameterName();
    final GraphFileDescriptor inputFileDescriptor =
      endNode.getInputFileDescriptor(inputParamName);
    final GraphFileDescriptor outputFileDescriptor =
      startNode.getOutputFileDescriptor(outputParamName);
    if (inputFileDescriptor != null && outputFileDescriptor != null) {
      return true;
    }
    return false;
  }

}
