/*
 * Detalhes da ltima alterao:
 * 
 * $Id$
 */
package csbase.client.project.dialogs;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.rmi.RemoteException;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import net.miginfocom.swing.MigLayout;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.openbus.OpenBusAccessPoint;
import csbase.exception.CSBaseException;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;

/**
 * Representa um dilogo de agendamento de atualizaes de um arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class UpdateFileDialog extends DesktopComponentDialog {
  /**
   * Prefixo das chaves de traduo.
   */
  private static final String LNG_PREFIX = "UpdateFileDialog.";
  /**
   * O arquivo objeto da atualizao.
   */
  private ClientProjectFile file;
  /**
   * Campo texto utilizado para se obter o intervalo da atualizao.
   */
  private JTextField intervalField;
  /**
   * Boto responsvel por parar as atualizaes.
   */
  private JButton stopButton;
  /**
   * Responsvel por indicar se o usurio deseja receber notificaes sobre a
   * atualizao do arquivo.
   */
  private JCheckBox notificationBox;

  /**
   * Cria o dilogo.
   * 
   * @param owner A janela dona do dilogo.
   * @param file O arquivo cuja atualizao ser agendada.
   */
  public UpdateFileDialog(JFrame owner, ClientProjectFile file) {
    super(owner, MessageFormat.format(LNG.get(LNG_PREFIX + "title"),
      new Object[] { file.getName() }));
    this.file = file;
    this.createGui();
  }

  /**
   * Cria a interface grfica.
   */
  private void createGui() {
    this.setModal(true);
    this.setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
    this.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });

    JPanel mainPanel =
      new JPanel(new MigLayout("ins 10 0 0 0", "[][grow]", "[]10[]20[][]"));

    mainPanel.add(new JLabel(LNG.get(LNG_PREFIX + "interval")), "gapleft 10");
    long interval = this.file.getUpdateInterval();
    String intervalText = "";
    if (interval > 0) {
      intervalText = String.valueOf(interval);
    }
    this.intervalField = new JTextField(intervalText, 5);
    mainPanel.add(this.intervalField, "grow x, split 2");
    mainPanel.add(new JLabel(LNG.get(LNG_PREFIX + "seconds")),
      "gapright 10, wrap");

    this.notificationBox = new JCheckBox(LNG.get(LNG_PREFIX + "notification"));
    mainPanel.add(this.notificationBox, "skip 1, gapright 10, wrap");

    mainPanel.add(this.makeButtonPanel(), "gapleft 10, grow x, spanx");

    String text = "";
    if (interval > 0) {
      text =
        MessageFormat.format(LNG.get(LNG_PREFIX + "update.status.user"),
          this.file.getUpdateUserLogin());
    }
    else {
      text = LNG.get(LNG_PREFIX + "update.status.none");
    }

    JLabel statusBar = new JLabel(text);
    statusBar.setBorder(BorderFactory.createLoweredBevelBorder());
    mainPanel.add(statusBar, "grow x, spanx");

    this.getContentPane().add(mainPanel);
    this.setResizable(false);
    this.pack();
    this.center(this.getOwner());
  }

  /**
   * Cria o painel de botes.
   * 
   * @return O painel de botes.
   */
  private JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    String startButtonTitle;
    if (this.file.getUpdateInterval() == 0) {
      startButtonTitle = LNG.get(LNG_PREFIX + "start");
    }
    else {
      startButtonTitle = LNG.get(LNG_PREFIX + "update");
    }
    JButton startButton = (JButton) panel.add(new JButton(startButtonTitle));
    startButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (handleStartAction()) {
          close();
        }
      }
    });

    this.stopButton =
      (JButton) panel.add(new JButton(LNG.get(LNG_PREFIX + "stop")));
    this.stopButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (handleStopAction()) {
          close();
        }
      }
    });

    JButton cancelButton =
      (JButton) panel.add(new JButton(LNG.get(LNG_PREFIX + "close")));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        close();
      }
    });

    GUIUtils.matchPreferredSizes(new JComponent[] { startButton,
        this.stopButton, cancelButton });
    this.getRootPane().setDefaultButton(startButton);
    if (this.file.getUpdateInterval() == 0) {
      this.stopButton.setEnabled(false);
    }
    else {
      this.stopButton.setEnabled(true);
    }
    return panel;
  }

  /**
   * Executa a ao do boto iniciar.
   * 
   * @return {@code true} caso a aao tenha sido executada, ou {@code false},
   *         caso contrrio.
   */
  private boolean handleStartAction() {
    String text = this.intervalField.getText();
    int interval;
    try {
      interval = Integer.parseInt(text);
    }
    catch (NumberFormatException e) {
      interval = -1;
    }
    if (interval <= 0) {
      StandardDialogs.showErrorDialog(this, MessageFormat.format(LNG
        .get(LNG_PREFIX + "invalid.interval.title"), new Object[] { this
        .getTitle() }), LNG.get(LNG_PREFIX + "invalid.interval.message"));
      return false;
    }
    final int updateInterval = interval;
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws RemoteException, CSBaseException {
        OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
        oap.init();
        byte[] chain = oap.makeChain();

        ClientRemoteLocator.projectService.startUpdate(file.getProjectId(),
          file.getPath(), updateInterval, notificationBox.isSelected(), chain);
        file.setUpdateInfo(User.getLoggedUser().getLogin(), updateInterval);
      }
    };
    if (task.execute(this.getOwner(), LNG.get(LNG_PREFIX + "task.title"),
      MessageFormat.format(LNG.get(LNG_PREFIX + "task.message"),
        new Object[] { this.file.getName() }))) {
      this.stopButton.setEnabled(true);
      StandardDialogs.showInfoDialog(this, MessageFormat.format(LNG
        .get(LNG_PREFIX + "update.start.title"), this.file.getName()), LNG
        .get(LNG_PREFIX + "update.start.success.message"));
      return true;
    }
    StandardDialogs.showInfoDialog(this, MessageFormat.format(
      LNG.get(LNG_PREFIX + "update.start.title"), this.file.getName()), LNG
      .get(LNG_PREFIX + "update.start.error.message"));
    return false;
  }

  /**
   * Executa a ao do boto parar.
   * 
   * @return {@code true} caso a aao tenha sido executada, ou {@code false},
   *         caso contrrio.
   */
  private boolean handleStopAction() {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws RemoteException {
        ClientRemoteLocator.projectService.stopUpdate(file.getProjectId(),
          file.getPath());
        file.setUpdateInfo(null, 0);
      }
    };
    if (task.execute(this.getOwner(), LNG.get(LNG_PREFIX + "task.title"),
      MessageFormat.format(LNG.get(LNG_PREFIX + "task.message"),
        new Object[] { this.file.getName() }))) {
      StandardDialogs.showInfoDialog(this, MessageFormat.format(LNG
        .get(LNG_PREFIX + "update.stop.title"), this.file.getName()), LNG
        .get(LNG_PREFIX + "update.stop.success.message"));
      this.stopButton.setEnabled(false);
      return true;
    }
    StandardDialogs
      .showInfoDialog(this, MessageFormat.format(
        LNG.get(LNG_PREFIX + "update.stop.title"), this.file.getName()), LNG
        .get(LNG_PREFIX + "update.stop.error.message"));
    return false;
  }
}
