package csbase.client.algorithms.view.flow;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.util.Collection;

import javax.swing.JScrollPane;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.view.AbstractAlgorithmConfiguratorPanel;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.filters.AskForParameterValuesActionFilter;
import csbase.client.applications.flowapplication.filters.HighlightElementFilter;
import csbase.client.applications.flowapplication.filters.HintElementFilter;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.zoom.ZoomControl;
import csbase.client.applications.flowapplication.zoom.ZoomModel;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * Painel representando a viso dos parmetros de um configurador de fluxo.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class FlowAlgorithmConfiguratorPanel extends
  AbstractAlgorithmConfiguratorPanel<FlowAlgorithmConfigurator> {

  /**
   * Viso do fluxo.
   */
  private Graph graph;

  /**
   * A rea de trabalho onde o fluxo  desenhado.
   */
  private Workspace workspace;

  /**
   * Construtor
   * 
   * @param owner A janela que criou esta (No aceita {@code null}).
   * @param configurator o configurador de algortimo que ser representado por
   *        este painel.
   * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
   *        {@link Mode#REPORT}.
   * 
   * @throws ClientException em caso de falha.
   */
  public FlowAlgorithmConfiguratorPanel(final Window owner,
    final FlowAlgorithmConfigurator configurator, final Mode mode)
    throws ClientException {

    super(owner, configurator, mode);
  }

  /**
   * Obtm o grafo.
   * 
   * @return O grafo.
   */
  public Graph getGraph() {
    return graph;
  }

  /**
   * <p>
   * Obtm a rea de trabalho.
   * </p>
   * 
   * <p>
   * A rea de trabalho  a regio onde so desenhados os elementos do grafo que
   * representa o fluxo de algoritmos. Ela  responsvel pelo controle de zoom e
   * o controle das barras de rolagem.
   * </p>
   * 
   * @return a rea de trabalho.
   */
  public Workspace getWorkspace() {
    return workspace;
  }

  /**
   * <p>
   * Obtm o configurador de algortimo representado por este painel.
   * </p>
   * <p>
   * O Graph representa a viso de um fluxo. Um Graph  construido a partir de
   * um modelo (Flow) que  obtido atravs de um configurador
   * (FlowAlgorithmConfigurator). Uma vez construido, o Graph se livra do modelo
   * e a partir dai passa a agir de forma independente a ele e como consequncia
   * ao configurador que criou o modelo. Sendo assim, se existe um Graph, o
   * configurador deve ser criado atravs deste, pois o configurador inicial
   * poderia estar desatualizado. Este mtodo j trata isso.
   * </p>
   * 
   * @return o configurador de algortimo representado por este painel.
   */
  @Override
  public FlowAlgorithmConfigurator getConfigurator() {
    FlowAlgorithmConfigurator algorithmConfigurator = super.getConfigurator();
    if (getGraph() == null) {
      return algorithmConfigurator;
    }

    AlgorithmVersionInfo version = algorithmConfigurator.getAlgorithmVersion();
    Flow flow = getGraph().toFlow();
    if (version == null) {
      String filePath = algorithmConfigurator.getFilePath();
      if (filePath == null) {
        return new FlowAlgorithmConfigurator(flow);
      }
      return new FlowAlgorithmConfigurator(filePath, flow);
    }
    return new FlowAlgorithmConfigurator(version, flow);
  }

  /**
   * @see csbase.client.algorithms.view.AbstractAlgorithmConfiguratorPanel#resetValues()
   */
  @Override
  public void resetValues() {
    super.resetValues();
    Collection<GraphNode> nodes = graph.getNodeCollection();
    for (GraphNode node : nodes) {
      node.getAlgorithmConfiguratorView().getConfigurator().resetValues();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected AbstractParametersPanel createParametersReportPanel()
    throws ClientException {
    return new FlowParametersReportPanel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected AbstractParametersPanel createParametersConfigurationPanel()
    throws ClientException {
    return new FlowParametersConfigurationPanel();
  }

  /**
   * Painel interno que detm a rea de trabalho onde o fluxo  desenhado.
   */
  private abstract class AbstractFlowParametersPanel extends
    AbstractParametersPanel {

    /**
     * Construtor
     * 
     * @throws ClientException em caso de falha.
     */
    protected AbstractFlowParametersPanel() throws ClientException {
      final Window owner = FlowAlgorithmConfiguratorPanel.this.getOwner();
      RemoteTask<Graph> task = new RemoteTask<Graph>() {
        @Override
        protected void performTask() throws Exception {
          FlowAlgorithmConfigurator configurator = getConfigurator();
          Flow flow = configurator.getFlow();
          Graph taskGraph = new Graph(owner, flow);
          setResult(taskGraph);
        }
      };

      final String prefix =
        "csbase.client.algorithms.view.flow.FlowAlgorithmConfiguratorPanel.";
      final String taskTitle = LNG.get(prefix + "reading_configurator_title");
      final String taskMessage = LNG.get(prefix + "reading_configurator");
      boolean successful = task.execute(owner, taskTitle, taskMessage);

      if (successful) {
        graph = task.getResult();
        workspace = new Workspace(graph, false, true);
        createFilters(workspace);
        initialize(workspace);
      }
      else {
        throw new ClientException(task.getError());
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean highlightValidationResult(ViewValidationResult result) {
      return getGraph().highlightValidationResult(result);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ViewValidationResult validate(ValidationMode mode) {
      return getGraph().validate(mode);
    }

    /**
     * Inicializa os componentes da viso.
     * 
     * @param workspaceArg A rea de trabalho onde o fluxo  desenhado.
     */
    protected abstract void initialize(final Workspace workspaceArg);

    /**
     * Cria os filtros.
     * 
     * @param workspaceArg A rea de trabalho onde o fluxo  desenhado.
     */
    private void createFilters(final Workspace workspaceArg) {
      // Faz highlight no n que estiver com o mouse em cima.
      final HighlightElementFilter highlightElementFilter =
        new HighlightElementFilter(workspaceArg);
      highlightElementFilter.attach();

      // Mostra a dica do n que estiver com o mouse em cima.
      final HintElementFilter hintElementFilter =
        new HintElementFilter(workspaceArg);
      hintElementFilter.attach();
    }
  }

  /**
   * Painel de configurao do fluxo.
   */
  private final class FlowParametersConfigurationPanel extends
    AbstractFlowParametersPanel {

    /**
     * Construtor
     * 
     * @throws ClientException em caso de falha.
     */
    protected FlowParametersConfigurationPanel() throws ClientException {
      super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void initialize(final Workspace workspaceArg) {
      setLayout(new GridBagLayout());
      final JScrollPane scrollPane = new JScrollPane(workspaceArg);
      add(scrollPane, new GBC(0, 0).both());

      final AskForParameterValuesActionFilter filter =
        new AskForParameterValuesActionFilter(workspaceArg);
      filter.attach();
    }
  }

  /**
   * Painel de visualizao do fluxo.
   */
  private final class FlowParametersReportPanel extends
    AbstractFlowParametersPanel {

    /**
     * Controle de zoom.
     */
    private ZoomControl zoomControl;

    /**
     * Construtor
     * 
     * @throws ClientException em caso de falha.
     */
    protected FlowParametersReportPanel() throws ClientException {
      super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void initialize(final Workspace workspaceArg) {
      setLayout(new BorderLayout());
      final JScrollPane scrollPane = new JScrollPane(workspaceArg);
      add(scrollPane, BorderLayout.CENTER);

      final ZoomModel zoomModel = workspaceArg.getZoomModel();
      zoomControl = new ZoomControl(zoomModel);
      add(zoomControl, BorderLayout.NORTH);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Dimension getMinimumSize() {
      return zoomControl.getPreferredSize();
    }
  }
}
