package csbase.client.applicationmanager.resourcehelpers;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.util.List;

import csbase.client.Client;
import csbase.client.applications.Application;
import csbase.logic.applicationservice.ApplicationRegistry;
import tecgraf.javautils.version.VersionNumber;
import tecgraf.javautils.version.VersionNumberTransition;
import tecgraf.javautils.version.VersionUtils;
import tecgraf.javautils.xml.conversion.XMLConversionInterface;
import tecgraf.javautils.xml.conversion.XMLConverter;
import tecgraf.javautils.xml.conversion.exception.XMLConversionException;
import tecgraf.javautils.xml.exception.XMLException;

/**
 * Utilitrio para tratamento de DTDs.
 * 
 * @author Tecgraf/PUC-Rio
 * @param <T> classe da aplicao.
 */
final public class ApplicationDTDResourceHelper<T extends Application> extends
  ApplicationResourceHelper<T> {

  /**
   * Cria um conversor de XMLs com base em verses inciais e finais.
   * 
   * @param tag
   * @param fromVersion
   * @param toVersion
   * @return uma converso.
   * @throws Exception em caso de erro na construo do conversor.
   */
  final private XMLConverter createXMLConverter(final String tag,
    final VersionNumber fromVersion, final VersionNumber toVersion)
    throws Exception {
    final ApplicationResourceVersionFinder<T> finder =
      new ApplicationResourceVersionFinder<T>(this, tag);
    final List<VersionNumber> versionsList =
      VersionUtils.getExistantVersionsList(finder, fromVersion, toVersion);
    if (versionsList.size() <= 0) {
      return null;
    }
    final List<VersionNumberTransition> transitionList =
      VersionUtils.getTransitionPairsList(versionsList);
    final XMLConverter cnv = createXMLConverter(tag, transitionList);
    return cnv;
  }

  /**
   * Cria um conversor de XMLs com base em uma lista de transies.
   * 
   * @param tag
   * @param transitionList lista de transies.
   * @return uma converso.
   * @throws Exception em caso de erro na construo do conversor.
   */
  private XMLConverter createXMLConverter(final String tag,
    List<VersionNumberTransition> transitionList) throws Exception {
    final XMLConverter converter = new XMLConverter();
    final Client client = Client.getInstance();
    converter.setEncoding(client.getSystemDefaultCharsetName());
    for (VersionNumberTransition transition : transitionList) {
      final VersionNumber fromVersion = transition.getFromVersion();
      final VersionNumber toVersion = transition.getToVersion();
      XMLConversionInterface conversion =
        createConversion(tag, fromVersion, toVersion);
      if (conversion != null) {
        converter.addConversion(conversion);
      }
    }
    return converter;
  }

  /**
   * @param tag
   * @param fromVersion
   * @param toVersion
   * @return conversor entre verses.
   * @throws Exception em caso de erro na construo do conversor.
   */
  private XMLConversionInterface createConversion(String tag,
    VersionNumber fromVersion, VersionNumber toVersion) throws Exception {
    final ApplicationRegistry reg = getApplicationRegistry();
    final String appClassName = reg.getClassName();
    final int idx = appClassName.lastIndexOf(".");
    final String path = appClassName.substring(0, idx);
    final String toId = toVersion.toStringAsTag();
    final String fromId = fromVersion.toStringAsTag();
    final String dirName = "dtdconversions";
    final String cnvClassPrefix = "ConversionFor";
    final String sufix = tag + "_" + fromId + "_To_" + toId;
    final String cnvClassText = dirName + "." + cnvClassPrefix + "_" + sufix;
    final String cnvClassName = path + "." + cnvClassText;
    Class<?> cnvClass = Class.forName(cnvClassName);
    @SuppressWarnings("unchecked")
    Class<? extends AbstractApplicationXMLConversion<T>> clazz =
      (Class<? extends AbstractApplicationXMLConversion<T>>) cnvClass;

    final T application = getApplication();
    final Class<VersionNumber> versionClass = VersionNumber.class;
    final Class<? extends Application> appClass = application.getClass();
    final Constructor<? extends AbstractApplicationXMLConversion<T>> constructor =
      clazz.getConstructor(appClass, versionClass, versionClass, String.class);
    final XMLConversionInterface conversion =
      constructor.newInstance(application, fromVersion, toVersion, tag);

    return conversion;
  }

  /**
   * @param converter conversor
   * @param tag tag
   * @param inputStream stream de entrada
   * @return stream stream convertido
   * @throws XMLConversionException em caso de erro durante a converso
   * @throws IOException em caso de erro ao salvar o arquivo temporrio
   */
  final public InputStream buildConvertedInputStream(
    final XMLConverter converter, final String tag,
    final InputStream inputStream) throws XMLConversionException, IOException {
    if (inputStream == null) {
      final String err = "Null input stream!";
      throw new IOException(err);
    }

    final InputStreamReader reader = new InputStreamReader(inputStream);
    final String urlPrefix = getResourceURLPrefix();
    converter.setDTDprefix(urlPrefix);
    final String prefix = this.getClass().getSimpleName();
    final File tmpFile = File.createTempFile(prefix, null);

    converter.convert(reader, true);
    converter.saveTo(tmpFile);
    final FileInputStream stream = new FileInputStream(tmpFile);
    return stream;
  }

  /**
   * Extrai nome do arquivo a ser buscado com base em um texto (systemId).
   * 
   * @param systemId system id.
   * @return nome do arquivo.
   */
  final public String extractFileNameFromSystemId(final String systemId) {
    if (systemId == null) {
      return null;
    }

    final String fileName;
    if (systemId.startsWith("http://") || systemId.startsWith("file://")) {
      final int idx = systemId.lastIndexOf("/");
      fileName = systemId.substring(idx);
    }
    else {
      fileName = systemId;
    }
    return fileName;
  }

  /**
   * Construo de conversor entre verses com base em uma tag de resource,
   * buscando da verso "zero" at a verso correne definida por propriedade da
   * aplicao ({@link #getResourceVersionFromProperty(String)}).
   * 
   * @param tag tag do resource a ser buscado.
   * @return conversor
   * @throws Exception em caso de erro na construo do conversor.
   */
  final public XMLConverter buildXMLConverter(final String tag)
    throws Exception {
    final VersionNumber fromVersion = VersionNumber.ZERO_VERSION;
    final VersionNumber toVersion = getResourceVersionFromProperty(tag);

    final ApplicationResourceVersionFinder<T> finder =
      new ApplicationResourceVersionFinder<T>(this, tag);
    final List<VersionNumber> versionsList =
      VersionUtils.getExistantVersionsList(finder, fromVersion, toVersion);
    if (versionsList.size() <= 0) {
      return null;
    }
    final XMLConverter cnv = createXMLConverter(tag, fromVersion, toVersion);
    return cnv;
  }

  /**
   * Monta uma string a ser usada como DTD.
   * 
   * @param version verso
   * @param tag tag
   * @return texto para DTD
   * @throws XMLException em caso de erro na parametrizao
   */
  final public String generateReadDTD(VersionNumber version, String tag)
    throws XMLException {
    if (version == null) {
      final String err = "No version defined!";
      throw new XMLException(err);
    }
    if (!version.isValid()) {
      final String err = "Invalid version defined!";
      throw new XMLException(err);
    }

    final String dtdPrefix = getResourceURLPrefix();
    final String fileName = getVersionedFileName(tag, version);
    final String dtd = dtdPrefix + "/" + fileName;
    return dtd;
  }

  /**
   * Monta uma string a ser usada como DTD cuja escrita est vinculada a ltima
   * verso do DTD.
   * 
   * @param tag tag
   * @return texto para DTD
   * @throws XMLException em caso de erro.
   */
  final public String generateWriteDTD(String tag) throws XMLException {
    final VersionNumber version = getResourceVersionFromProperty(tag);
    final String ext = getFileExtension();
    final String vText = version.toString();
    final String dtd = tag + "-" + vText + "." + ext;
    return dtd;
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  public ApplicationDTDResourceHelper(final T application) {
    super(application, ApplicationResourceType.DTD);
  }
}
