package csbase.client.applications.algorithmsmanager.versiontree;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

/**
 * Essa classe representa o modelo da rvore com as verses do algoritmo.
 * 
 */
public class VersionTreeModel implements TreeModel {
  /** N raiz da rvore */
  private VersionTreeRootNode root;

  /** Listeners de eventos do modelo da rvore. */
  private List<TreeModelListener> listeners;

  /**
   * Cria o modelo.
   * 
   * @param root n raiz.
   */
  public VersionTreeModel(VersionTreeRootNode root) {
    listeners = new ArrayList<>();
    this.root = root;
  }

  /**
   * @see javax.swing.tree.TreeModel#getRoot()
   */
  public Object getRoot() {
    return root;
  }

  /**
   * @see javax.swing.tree.TreeModel#getChildCount(java.lang.Object)
   */
  public int getChildCount(Object parent) {
    TreeNode node = (TreeNode) parent;
    return node.getChildCount();
  }

  /**
   * @see javax.swing.tree.TreeModel#isLeaf(java.lang.Object)
   */
  public boolean isLeaf(Object obj) {
    TreeNode node = (TreeNode) obj;
    return node.isLeaf();
  }

  /**
   * @see javax.swing.tree.TreeModel#addTreeModelListener(javax.swing.event.TreeModelListener)
   */
  public void addTreeModelListener(TreeModelListener l) {
    listeners.add(l);
  }

  /**
   * @see javax.swing.tree.TreeModel#removeTreeModelListener(javax.swing.event.TreeModelListener)
   */
  public void removeTreeModelListener(TreeModelListener l) {
    listeners.remove(l);
  }

  /**
   * @see javax.swing.tree.TreeModel#getChild(java.lang.Object, int)
   */
  public Object getChild(Object parent, int index) {
    TreeNode node = (TreeNode) parent;
    return node.getChildAt(index);
  }

  /**
   * @see javax.swing.tree.TreeModel#getIndexOfChild(java.lang.Object,
   *      java.lang.Object)
   */
  public int getIndexOfChild(Object parent, Object child) {
    TreeNode parentNode = (TreeNode) parent;
    TreeNode childNode = (TreeNode) child;
    return parentNode.getIndex(childNode);
  }

  /**
   * @see javax.swing.tree.TreeModel#valueForPathChanged(javax.swing.tree.TreePath,
   *      java.lang.Object)
   */
  public void valueForPathChanged(TreePath path, Object newValue) {
  }

  /**
   * Notifica os listeners de que ns foram removidos da rvore.
   * 
   * @param parentNode n pai de todos os ns removidos.
   * @param removedNodesIndexes ndices dos ns removidos.
   * @param removedNodes ns removidos.
   */
  private void nodesWereRemoved(DefaultMutableTreeNode parentNode,
    int[] removedNodesIndexes, Object[] removedNodes) {
    for (Iterator<TreeModelListener> iter = listeners.iterator(); iter
      .hasNext();) {
      TreeModelListener listener = (TreeModelListener) iter.next();
      TreePath path = new TreePath(parentNode.getPath());
      TreeModelEvent event = new TreeModelEvent(this, path,
        removedNodesIndexes, removedNodes);
      listener.treeNodesRemoved(event);
    }
  }

  /**
   * Notifica os listeners de que ns foram inseridos na rvore.
   * 
   * @param parentNode n pai dos ns inseridos
   * @param insertedNodesIndexes ndices dos ns inseridos
   * @param insertedNodes ns inseridos na rvore
   */
  private void nodesWereInserted(DefaultMutableTreeNode parentNode,
    int[] insertedNodesIndexes, Object[] insertedNodes) {
    for (Iterator<TreeModelListener> iter = listeners.iterator(); iter
      .hasNext();) {
      TreeModelListener listener = (TreeModelListener) iter.next();
      TreePath path = new TreePath(parentNode.getPath());
      TreeModelEvent event = new TreeModelEvent(this, path,
        insertedNodesIndexes, insertedNodes);
      listener.treeNodesInserted(event);
    }
  }

  /**
   * Notifica os listeners que a estrutura da rvore mudou.
   * 
   * @param parentNode n a partir do qual a estrutura foi alterada.
   */
  private void nodeStructureChanged(DefaultMutableTreeNode parentNode) {
    for (Iterator<TreeModelListener> iter = listeners.iterator(); iter
      .hasNext();) {
      TreeModelListener listener = (TreeModelListener) iter.next();
      TreePath path = new TreePath(parentNode.getPath());
      TreeModelEvent event = new TreeModelEvent(this, path);
      listener.treeStructureChanged(event);
    }
  }

  /**
   * Adiciona um algoritmo ao modelo.
   * 
   * @param newVersionode n a ser adicionado.
   */
  public void addVersion(DefaultMutableTreeNode newVersionode) {
    root.add(newVersionode);
    int newNodeIndex = root.getIndex(newVersionode);
    nodesWereInserted(root, new int[] { newNodeIndex },
      new Object[] { newVersionode });
  }

  /**
   * Remove um algoritmo do modelo.
   * 
   * @param removedNodeIndex ndice do algoritmo a ser removido.
   */
  public void removeVersion(int removedNodeIndex) {
    Object removedNode = root.getChildAt(removedNodeIndex);
    root.remove(removedNodeIndex);
    nodesWereRemoved(root, new int[] { removedNodeIndex },
      new Object[] { removedNode });
  }

  /**
   * Modifica o n raiz.
   * 
   * @param root novo n raiz.
   */
  public void setRoot(VersionTreeRootNode root) {
    this.root = root;
    nodeStructureChanged(root);
  }

  /**
   * Indica se determinado algoritmo est presente no modelo.
   * 
   * @param name nome do algoritmo a ser buscado.
   * 
   * @return true se o algoritmo for encontrado no modelo.
   */
  public boolean algorithmExists(String name) {
    Enumeration children = root.children();
    while (children.hasMoreElements()) {
      DefaultMutableTreeNode node = (DefaultMutableTreeNode) children
        .nextElement();
      if (node.toString().equals(name)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Remove todas as verses de algoritmos.
   */
  public void removeAllVersions() {
    int childCount = root.getChildCount();
    Object[] removedNodes = new Object[childCount];
    int[] removedIndexes = new int[childCount];
    for (int i = 0; i < childCount; i++) {
      TreeNode removedNode = root.getChildAt(i);
      root.remove(i);
      removedNodes[i] = removedNode;
      removedIndexes[i] = i;
    }
    nodesWereRemoved(root, removedIndexes, removedNodes);
  }
}
