package csbase.client.applications.jobmonitor.drivers;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import csbase.client.applications.Application;
import csbase.client.applications.jobmonitor.JobMonitor;
import csbase.client.applications.jobmonitor.JobMonitorSystemInfo;
import csbase.client.applications.jobmonitor.columns.DoubleJobInfoColumn;
import csbase.client.applications.jobmonitor.columns
  .DoubleWithMagnitudeJobInfoColumn;
import csbase.client.applications.jobmonitor.columns.StringJobInfoColumn;
import csbase.client.applications.jobmonitor.columns.SystemColumn;
import csbase.client.applications.jobmonitor.rowmodel.JobInfoRow;
import csbase.client.facilities.configurabletable.column.IConfigurableColumn;
import csbase.client.facilities.configurabletable.stringprovider
  .ApplicationStringProvider;
import csbase.client.facilities.configurabletable.stringprovider
  .IStringProvider;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;
import csbase.logic.SGASet;

/**
 * Driver encarregado de converter informaes dos jobs que esto no formato XML
 * para uma tabela configurvel.
 * 
 * O XML DEVE atender as seguintes restries: <br>
 * 1 - Deve ter um elemento root nomeado como 'Data'; <br>
 * 2 - Os nicos elementos filhos de 'Data' processados so os 'Job' <br>
 * 3 - Atributos dos elementos so desconsiderados na construo da tabela;
 * 
 * Obs: Cada elemento filho de 'Job' ser transformado em uma coluna da tabela
 * configurvel.
 * 
 * Ex: <br>
 * {@code
 * 
 *  <Data>
 *    <Job>
 *      firstJobId
 *      <... elementos diversos>
 *        ...
 *      <...>
 *    </Job>
 *    <Job>
 *      <Job_Id>secondJobId</Job_Id>
 *      <... elementos diversos>
 *        ...
 *      <...>
 *    </Job>
 *  </Data>
 *  
 * }
 * 
 * @author Tecgraf
 */
public class PBSDriver extends JobInfoDriver {

  /** Constante que define o identificador da tabela criada por este driver */
  public final static String PBS_DRIVER_TABLE = "PBS_DRIVER_TABLE";

  /** Nome do elemento root do XML que contm as informaes dos jobs. */
  public final static String ROOT_ELEMENT = "Data";

  /** Nome do elemento que descreve as informaes de um job especfico. */
  public final static String JOB_ELEMENT = "Job";

  /** Nome da propriedade que descreve o servidor em que o job est em execuo. */
  public final static String CLUSTER_PROPERTY = "Cluster";

  /** Nome da propriedade que descreve o nome de um job. */
  public final static String JOB_NAME_PROPERTY = "Job_Name";

  /** Lista de ordem de grandezes que os valores numricos podem ter. */
  private List<String> magnitudes;

  /**
   * Enumerao usada para definir os tipos de colunas que esse driver constroi.
   * Essa enumerao s  usada no algoritmo de heurstica que define o tipo de
   * uma coluna a partir dos valores das linhas.
   * 
   * @author Tecgraf
   */
  private enum ColumnType {
    /** Coluna que exibe valor numrico */
    DOUBLE,

    /** Coluna que exibe valor numrico com ordem de grandeza */
    DOUBLE_WITH_MAGNITUDE,

    /** Coluna que exibe um string */
    STRING;

    /** Ordem da grandeza do valor */
    private String magnitude = "";

    /**
     * Atribui a ordem de grandeza do valor.
     * 
     * @param magnitude - ordem de grandeza do valor.
     */
    public void setMagnitude(String magnitude) {
      this.magnitude = magnitude;
    }

    /**
     * Retorna a ordem de grandeza do valor.
     * 
     * @return ordem de grandeza do valor.
     */
    public String getMagnitude() {
      return magnitude;
    }

  }

  /**
   * Construtor padro.
   * 
   * @param application - referncia para a aplicao.
   */
  public PBSDriver(Application application) {
    super(application);

    magnitudes = new ArrayList<String>();
    magnitudes.add("kb");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<JobInfoRow> getRows(List<SGASet> sgas) {
    List<JobInfoRow> rows = new LinkedList<JobInfoRow>();

    for (SGASet sgaSet : sgas) {
      String jobsInfo = sgaSet.getJobsInfo();

      Node root = parseXML(jobsInfo);

      if (root != null) {

        NodeList jobList = root.getChildNodes();

        for (int i = 0; i < jobList.getLength(); i++) {
          Node job = jobList.item(i);

          JobInfoRow newLine = new JobInfoRow();
          newLine.add(CLUSTER_PROPERTY, sgaSet.getName());

          NodeList jobInfo = job.getChildNodes();

          for (int j = 0; j < jobInfo.getLength(); j++) {

            Node jobChild = jobInfo.item(j);

            if (jobChild.getNodeType() == Node.TEXT_NODE) {
              newLine.add(JOB_ID_PROPERTY, jobChild.getNodeValue());
            }
            else {
              addToNewLine("", jobChild, newLine);
            }
          }

          rows.add(newLine);
        }
      }
    }

    return rows;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigurableTable<JobInfoRow> getTable(List<SGASet> sgas) {
    List<JobInfoRow> rows = getRows(sgas);

    List<IConfigurableColumn<JobInfoRow>> columns = getColumns(rows);

    // apenas as 10 primeiras colunas so exibidas.
    for (int i = 0; i < columns.size(); i++) {
      columns.get(i).setVisible(i < 10);
    }

    ConfigurableTable<JobInfoRow> table = null;

    if (columns.size() > 0) {
      addExtraColumns(columns);
      table =
        new ConfigurableTable<JobInfoRow>(PBS_DRIVER_TABLE, columns, null);

      table.updateRows(rows);
    }

    return table;
  }

  /**
   * Adiciona novas colunas a lista de colunas que compem a tabela que exibe
   * informaes de jobs. No necessariamente essas novas colunas exibem
   * atributos do objeto {@link JobInfoRow}, por exemplo, podemos adicionar uma
   * coluna que mostra o resultado de alguma operao entre os valores de outras
   * colunas.
   * 
   * @param columns - lista de colunas.
   */
  private void addExtraColumns(List<IConfigurableColumn<JobInfoRow>> columns) {
    IStringProvider provider = new ApplicationStringProvider(getApplication());

    JobMonitorSystemInfo systemInfo =
      ((JobMonitor) getApplication()).getSystemInfo();

    columns.add(0, new SystemColumn(systemInfo, provider, JOB_NAME_PROPERTY));
  }

  /**
   * Funo que adiciona recursivamente os valores de um elemento como valores
   * de uma linha.
   * 
   * @param prefix - prefixo usado ao adicionar o nome de uma propriedade da
   *        linha.
   * @param elem - elemento XML.
   * @param line - linha da tabela.
   */
  private void addToNewLine(String prefix, Node elem, JobInfoRow line) {
    NodeList children = elem.getChildNodes();

    if (children.getLength() == 0) {
      line.add(prefix + elem.getNodeName(), null);
    }
    else if (children.getLength() == 1
      && children.item(0).getNodeType() == Node.TEXT_NODE) {

      String value = children.item(0).getNodeValue();

      line.add(prefix + elem.getNodeName(), value);
    }
    else {

      for (int i = 0; i < children.getLength(); i++) {
        addToNewLine(prefix + elem.getNodeName() + ".", children.item(i), line);
      }

    }

  }

  /**
   * Cria as colunas que exibem as informaes dos jobs a partir das linhas.
   * 
   * @param rows - linhas da tabela.
   * @return colunas da tabela.
   */
  private List<IConfigurableColumn<JobInfoRow>> getColumns(List<JobInfoRow> rows) {

    Map<String, Set<ColumnType>> columnsTypes = inferColumnsType(rows);

    // criando as colunas 
    List<IConfigurableColumn<JobInfoRow>> columns =
      new ArrayList<IConfigurableColumn<JobInfoRow>>();

    for (Entry<String, Set<ColumnType>> entry : columnsTypes.entrySet()) {

      String columnName = entry.getKey();
      Set<ColumnType> types = entry.getValue();

      if (types.size() == 1) {

        ColumnType type = types.iterator().next();

        switch (type) {
          case DOUBLE:
            columns.add(new DoubleJobInfoColumn(columnName));
            break;

          case DOUBLE_WITH_MAGNITUDE:
            columns.add(new DoubleWithMagnitudeJobInfoColumn(columnName, type
              .getMagnitude()));
            break;

          case STRING:
            columns.add(new StringJobInfoColumn(columnName));
            break;
        }
      }
      else {
        columns.add(new StringJobInfoColumn(columnName));
      }
    }

    return columns;
  }

  /**
   * Mtodo que infere o tipo de uma coluna baseado nos respectivos valores das
   * linhas. A heurstica utilizada : <br/>
   * - Se todos os valores de uma mesma coluna forem do mesmo tipo, ento
   * criamos uma coluna desse tipo. <br/>
   * - Se uma coluna possuir pelo menos dois valores de tipos diferente, ento
   * criamos uma coluna que exibe Strings.
   * 
   * @param rows - lista com as linhas.
   * @return mapeamento do nome da coluna a um conjunto de tipos dos valores
   *         daquela coluna.
   */
  private Map<String, Set<ColumnType>> inferColumnsType(List<JobInfoRow> rows) {
    Map<String, Set<ColumnType>> columnsTypes =
      new LinkedHashMap<String, Set<ColumnType>>();

    for (JobInfoRow row : rows) {
      for (Entry<String, String> entry : row.entrySet()) {
        String columnName = entry.getKey();
        String columnValue = entry.getValue();

        ColumnType type = getColumnType(columnValue);

        if (columnsTypes.get(columnName) == null) {
          columnsTypes.put(columnName, new HashSet<ColumnType>());
        }

        columnsTypes.get(columnName).add(type);
      }
    }
    return columnsTypes;
  }

  /**
   * Dado uma string que representa o valor da coluna, retorna um
   * {@link ColumnType} que melhor se adequa ao valor.
   * 
   * @param columnValue - valor da coluna.
   * @return tipo da coluna.
   */
  private ColumnType getColumnType(String columnValue) {

    ColumnType result = ColumnType.STRING;

    if (isDouble(columnValue)) {
      result = ColumnType.DOUBLE;
    }
    else if (isDoubleWithMagnitude(columnValue)) {
      result = ColumnType.DOUBLE_WITH_MAGNITUDE;
      result.setMagnitude(getMagnitude(columnValue));
    }

    return result;
  }

  /**
   * Verifica se o valor  um double.
   * 
   * @param value - String que representa o valor.
   * @return true se o valor for um double, false caso contrrio.
   */
  private boolean isDouble(String value) {
    try {
      Double.parseDouble(value);
      return true;
    }
    catch (NumberFormatException e) {
      return false;
    }
  }

  /**
   * Verifica se o valor tem ordem de grandeza.
   * 
   * @param columnValue - valor.
   * @return true se o valor possuir alguma ordem de grandeza, false caso
   *         contrrio.
   */
  private boolean isDoubleWithMagnitude(String columnValue) {
    String magnitude = getMagnitude(columnValue);
    return magnitude != null;
  }

  /**
   * Obtm a ordem de grandeza do valor.
   * 
   * @param columnValue - valor da coluna.
   * @return ordem de grandeza do valor.
   */
  private String getMagnitude(String columnValue) {

    for (String m : magnitudes) {
      if (columnValue.endsWith(m)) {

        String value = columnValue.substring(0, columnValue.lastIndexOf(m));

        if (isDouble(value)) {
          return m;
        }
      }
    }

    return null;
  }

  /**
   * Faz o parsing do XML recebido por parmetro.
   * 
   * @param xml - String que representa o XML a ser processado.
   * @return Se o XML estiver sintticamente correto ento retorna a rvore
   *         sinttica equivalente ao XML, null caso contrrio.
   */
  private Node parseXML(String xml) {

    if (xml == null || xml.isEmpty()) {
      return null;
    }

    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    DocumentBuilder parser;

    Element elem = null;
    InputSource input = new InputSource();
    input.setCharacterStream(new StringReader(xml));

    try {
      parser = factory.newDocumentBuilder();
      org.w3c.dom.Document doc = parser.parse(input);
      elem = doc.getDocumentElement();
    }
    catch (Exception e) {
      throw new RuntimeException("XML com erro de sintaxe:" + xml, e);
    }

    validate(elem, xml);

    return elem;
  }

  /**
   * Valida o XML seguindo o padro do PBS. Caso o XML no seja vlido, uma
   * exceo  lanada informando o usurio que os dados recebidos esto
   * invlidos.
   * 
   * @param root - n raiz.
   * @param xml - xml original adicionado nas excees para depurao.
   */
  private void validate(Node root, String xml) {
    if (!ROOT_ELEMENT.equals(root.getNodeName())) {
      throw new RuntimeException("O elemento root deveria ser '" + ROOT_ELEMENT
        + "'; XML original:" + xml);
    }

    NodeList jobList = root.getChildNodes();

    if (jobList == null) {
      throw new RuntimeException(
        "Lista de jobs no pode ser nula; XML original:" + xml);
    }

    if (jobList.getLength() == 0) {
      throw new RuntimeException(
        "Lista de jobs deve ter ao menos um job; XML original:" + xml);
    }

    for (int i = 0; i < jobList.getLength(); i++) {
      Node job = jobList.item(i);

      if (!JOB_ELEMENT.equals(job.getNodeName())) {
        throw new RuntimeException("Elemento root '" + ROOT_ELEMENT
          + "' s pode ser composto por elementos '" + JOB_ELEMENT
          + "'; XML original:" + xml);
      }
    }
  }

}
