package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.RemoveProjectTask;
import csbase.logic.User;

/**
 * Ao que trata da remoo de projetos do sistema.
 * 
 * @author jnlopes
 * 
 */
public class RemoveProjectAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public RemoveProjectAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc} Mtodo que invoca a remote task que efetivamente remove
   * projetos.
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final boolean ret = runTask(projects);
    if (ret) {
      projectsManager.removeProjects(projects);
    }
    else {
      StandardDialogs.showInfoDialog(frame,
        getString("RemoveProjectAction.project.removal.error.title"),
        getString("RemoveProjectAction.project.removal.error.message"));
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(List<ProjectsManagerData> projects) throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    final RemoveProjectTask prt =
      new RemoveProjectTask(projectsManager, projects);
    prt.execute(frame, getString("RemoveProjectAction.title"),
      getString("RemoveProjectAction.message"));
    if (prt.wasCancelled()) {
      final String err = getString("RemoveProjectAction.cancelled.message");
      prt.showError(err);
      return false;
    }
    if (prt.getStatus() != true) {
      final Exception exception = prt.getError();
      throw exception;
    }
    return prt.getResult();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_DELETE_16;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final List<ProjectsManagerData> prjList =
      new ArrayList<ProjectsManagerData>();

    prjList.addAll(projectList);
    final List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();
      if (spaceAllocation == ProjectSpaceAllocation.WAITING_ALLOCATION
        || spaceAllocation == ProjectSpaceAllocation.WAITING_DEALLOCATION) {
        failedProjects.add(pmd);
        prjList.remove(i);
        continue;
      }

      final User loggedUser = User.getLoggedUser();
      final String loggedUserId = loggedUser.getLogin();
      if (!loggedUser.isAdmin() && !pmd.getOwnerId().equals(loggedUserId)) {
        failedProjects.add(pmd);
        prjList.remove(i);
        continue;
      }
      i++;
    }

    final ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("RemoveProjectAction.project.selection.denied.message")
          + getString("RemoveProjectAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("RemoveProjectAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      final String someDeniedMessage =
        getString("RemoveProjectAction.project.selection.some.denied.message")
          + getString("RemoveProjectAction.project.requirements.message");
      final int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("RemoveProjectAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }
    else {
      final int res =
        StatusDialog.showAllOkDialog(projectsManager,
          getString("RemoveProjectAction.confirm.removal.dialog"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }
}
