/*
 * $Id$
 */
package csbase.client.facilities.algorithms.executor;

import java.awt.Window;
import java.util.Map;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.RemoteTask;
import csbase.client.facilities.algorithms.parameters.NodeParametersValues;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.remote.ClientRemoteLocator;

/**
 * Ao de configurar um algoritmo que pode ser submetido para execuo
 * programaticamente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class CommandConfiguration {
  /**
   * Nome do algoritmo
   */
  private String algorithmName;

  /**
   * Identificador da verso do algoritmo
   */
  private AlgorithmVersionId algoVersionId;

  /**
   * Janela pai que originou a ao
   */
  private Window owner;

  /**
   * Configurador de algoritmos
   */
  private AlgorithmConfigurator configurator;

  /**
   * Construtor.
   * 
   * @param algorithmName nome do algoritmo
   * @param algoVersionId identificador da verso do algoritmo
   * @param owner janela pai que originou a ao
   */
  public CommandConfiguration(String algorithmName,
    AlgorithmVersionId algoVersionId, Window owner) {
    this.algorithmName = algorithmName;
    this.algoVersionId = algoVersionId;
    this.owner = owner;
  }

  /**
   * Construtor.
   * 
   * @param configurator configurador do algoritmo
   * @param owner janela pai que originou a ao
   */
  public CommandConfiguration(AlgorithmConfigurator configurator, Window owner) {
    this.configurator = configurator;
    this.algorithmName = configurator.getAlgorithmName();
    this.algoVersionId = configurator.getAlgorithmVersionId();
    this.owner = owner;
  }

  /**
   * Obtm o nome do algoritmo.
   * 
   * @return o nome do algoritmo
   */
  public String getAlgorithmName() {
    return algorithmName;
  }

  /**
   * Obtm o identificador da verso do algoritmo.
   * 
   * @return o identificador da verso do algoritmo
   */
  public AlgorithmVersionId getAlgoVersionId() {
    return algoVersionId;
  }

  /**
   * Obtm a janela pai que originou a ao.
   * 
   * @return a janela pai que originou a ao
   */
  public Window getOwner() {
    return owner;
  }

  /**
   * Configura um determinado algoritmo.
   * 
   * @param paramValues valores dos parmetros a serem preenchidos
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   */
  public boolean config(final NodeParametersValues paramValues) {
    if (getConfigurator() == null) {
      return false;
    }
    try {
      final Map<String, String> allParametersValues =
        paramValues.getAllParametersValues();
      configurator.setParameterValuesByName(allParametersValues);
    }
    catch (ParseException pe) {
      final String msgTag = "ConfigAlgorithmAction.msg.error.invalid.params";
      final String msg = LNG.get(msgTag);
      StandardErrorDialogs.showErrorDialog(owner, msg, pe);
      return false;
    }
    catch (ParameterNotFoundException e) {
      final String msgTag = "ConfigAlgorithmAction.msg.error.invalid.params";
      final String msg = LNG.get(msgTag);
      StandardErrorDialogs.showErrorDialog(owner, msg, e);
      return false;
    }
    return true;
  }

  /**
   * Obtm o configurador para o algoritmo especificado.
   * 
   * @return o configurador para o algoritmo especificado
   */
  public AlgorithmConfigurator getConfigurator() {
    if (configurator == null) {
      // Obtm o configurador do algoritmo
      RemoteTask<AlgorithmConfigurator> configuratorTask =
        new RemoteTask<AlgorithmConfigurator>() {
          @Override
          protected void performTask() throws Exception {
            if (algoVersionId == null) {
              algoVersionId =
                ClientRemoteLocator.algorithmService.getInfo(algorithmName)
                  .getLastVersion().getId();
            }
            setResult(ClientRemoteLocator.algorithmService
              .createAlgorithmConfigurator(algorithmName, algoVersionId));
          }
        };
      boolean success =
        configuratorTask.execute(owner, owner.getName(),
          LNG.get("ConfigAlgorithmAction.info.get.configurator"));
      if (success) {
        configurator = configuratorTask.getResult();
      }
    }
    return configurator;
  }

  /**
   * Obtm os valores dos parmetros a serem exportados pelo configurador.
   * 
   * @return os valores dos parmetros a serem exportados pelo configurador
   */
  public Map<Object, Object> getExportedValues() {
    return configurator.exportValues();
  }

}
