/*
 * PermissionInfoDialog.java
 *
 * $Id: PermissionInfoDialog.java 175496 2016-08-17 13:08:33Z lmachado $
 */
package csbase.client.ias;

import java.awt.Color;
import java.awt.Container;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.lang.reflect.Constructor;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.PermissionProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.gui.CardPanel;
import csbase.logic.Permission;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * A classe <code>PermissionInfoDialog</code>  responsvel por incluir ou
 * modificar as informaes de uma permisso, atravs de uma interface
 * especfica.
 *
 * @author Marcus Leal
 * @version $Revision: 175496 $
 */
public class PermissionInfoDialog extends DesktopComponentDialog {
  /** Classe genrica para permisses simples */
  private static final String SIMPLE_PERMISSION_CLASS =
    "csbase.logic.SimplePermission";

  /** Classe genrica para permisses com atributos */
  private static final String ATTRIBUTES_PERMISSION_CLASS =
    "csbase.logic.AttributesPermission";

  /** Classe genrica para permisses com usurio/senha */
  private static final String USER_PASSWORD_PERMISSION_CLASS =
    "csbase.logic.UserPasswordPermission";

  /** Classe genrica para permisses com atributos dinmicos. */
  private static final String CHOICE_PERMISSION_CLASS =
    "csbase.logic.ChoicePermission";

  /**
   * Sufixo para localizao da descrio da classe de permisso no arquivo de
   * propriedades.
   */
  private static final String DESCRIPTION_SUFFIX = ".desc";

  /** Resultados possveis da validao dos campos. */
  enum FieldsState {
    /**
     * Indica que todos os campos esto validados e prontos para serem
     * persistidos.
     */
    FIELDS_OK,

    /** Indica que existem campos no preenchidos */
    EMPTY_FIELDS,

    /** Indica que a senha no confere com a confirmao da mesma */
    PASSWORDS_DONT_MATCH,

    /** Indica erro nos atributos */
    ATTRIBUTES_ERROR;
  }

  /** Janela pai */
  private Window owner;

  /** Ttulo da janela */
  private String windowLabel;

  /**
   * Objeto a ser usado para persistir os dados de permisso entrados pelo
   * usurio.
   */
  private Permission permission;

  /** Campo nome da permisso */
  private JTextField name;

  /** Campo descrio da permisso */
  private JTextField description;

  /** Boto para navegar pelos atributos de uma ChoicePermission */
  JButton nextButton;

  /** Boto para navegar pelos atributos de uma ChoicePermission */
  JButton previousButton;

  /** Combo de seleo da classe de permisso */
  private JComboBox permissionClass;

  /** Campo de descrio da classe de permisso */
  private JTextArea permissionClassDesc;

  /** Boto de incluso/alterao da permisso */
  JButton includePermissionButton;

  /** String que indica qual a classe genrica atualmente selecionada */
  private String selectedPermissionClass;

  /** Indica se uma nova permisso est sendo criada */
  private boolean isNew;

  /**
   * CardPanel que contm os paineis para insero/alterao dos atributos de
   * uma permisso.
   */
  private CardPanel permissionAtributePanels;

  /**
   * Separador entre os campos comuns a todas as permisses e o painel de
   * insero/alterao dos atributos de uma permisso.
   */
  JSeparator separator;

  /**
   * Cria uma janela para incluso de dados de uma nova permisso.
   *
   * @param owner janela que originou esta PermissionInfoDialog.
   */
  public PermissionInfoDialog(Window owner) {
    this(owner, null);
  }

  /**
   * Cria uma janela para modificao de dados de uma permisso j cadastrada no
   * sistema, ou de incluso de nova permisso, caso o parmetro permission seja
   * nulo.
   *
   * @param owner janela que originou esta PermissionInfoDialog.
   * @param permission a permisso que vai ser modificada
   */
  public PermissionInfoDialog(Window owner, Permission permission) {
    super(owner);
    this.owner = owner;
    if (permission == null) {
      isNew = true;
      windowLabel = LNG.get("IAS_PERMISSION_INCLUSION_TITLE");
    }
    else {
      windowLabel = LNG.get("IAS_PERMISSION_UPDATE_TITLE");
      isNew = false;
      this.permission = permission;
    }
    setTitle(windowLabel);
    display();
  }

  /**
   * Cria e mostra a janela de incluso/modificao da permisso.
   */
  protected void display() {

    // Cria o painel de fundo
    JPanel backPane = new JPanel(new GridBagLayout());
    backPane.add(makeButtonPanel(), new GBC(0, 1).south().insets(0, 5, 5, 5));
    // Cria o painel com os campos e o boto limpar.
    JPanel mainPane = new JPanel(new GridBagLayout());
    mainPane.add(makeMainPanel(), new GBC(0, 0).insets(2, 0, 0, 0).both());
    mainPane.add(makePermissionsButtonPanel(), new GBC(0, 1).south().insets(3));
    backPane.add(mainPane, new GBC(0, 0).insets(5, 5, 0, 5).both());
    // Cria a janela
    setContentPane(backPane);
    setDefaultCloseOperation(DesktopComponentDialog.DISPOSE_ON_CLOSE);
    pack();
    center(owner);
    setVisible(true);
  }

  /**
   * Constri o painel para preenchimento ou alterao dos dados da permisso de
   * usurio. De acordo com a classe de permisso que for selecionada, dois
   * painis opcionais podero ou no ser exibidos: o painel de usurio/senha e
   * o painel de atributos (OBS: neste caso, o termo "painel" no representa um
   * JPanel, mas um simples agrupamento lgico de campos).
   *
   * @return o painel principal da janela
   */
  protected Container makeMainPanel() {
    JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    // Painel principal
    panel.add(makeCommonPanel(),
      new GBC(0, 0).northwest().insets(10, 10, 0, 10).both());

    permissionAtributePanels = new CardPanel(true, false);

    setResizable(true);

    permissionAtributePanels.add(new UserPasswordPanel(this),
      USER_PASSWORD_PERMISSION_CLASS);
    permissionAtributePanels.add(new AttributesPanel(this),
      ATTRIBUTES_PERMISSION_CLASS);
    permissionAtributePanels.add(new ChoicePermissionPanel(this, windowLabel),
      CHOICE_PERMISSION_CLASS);
    permissionAtributePanels.add(new JPanel(), SIMPLE_PERMISSION_CLASS);

    panel.add(permissionAtributePanels,
      new GBC(0, 1).south().insets(5, 0, 0, 0).both().fillxy());

    initFields();
    return panel;
  }

  /**
   * @return Painel com o grupo de componentes comuns a todas as permisses.
   */
  private JPanel makeCommonPanel() {
    JPanel panel = new JPanel(new GridBagLayout());

    // Cria os componentes
    JLabel nameL = new JLabel(LNG.get("IAS_PERMISSION") + ":");
    name = new JTextField(20);
    JLabel descriptionL = new JLabel(LNG.get("IAS_DESCRIPTION") + ":");
    description = new JTextField(20);
    JLabel permissionClassL = new JLabel(LNG.get("IAS_PERMISSION_CLASS") + ":");
    permissionClass = makeClassComboBox();
    permissionClass.setSelectedIndex(-1);
    permissionClassDesc = makePermissionClassDescription(permissionClass);
    // Label permisso
    GBC gbc = new GBC(0, 0).northwest().insets(5, 0, 10, 5);
    panel.add(nameL, gbc);

    // Campo permisso
    gbc = new GBC(1, 0).horizontal().insets(0, 0, 5, 0).filly();
    panel.add(name, gbc);

    // Label descrio
    gbc = new GBC(0, 1).northwest().insets(5, 0, 10, 5);
    panel.add(descriptionL, gbc);

    // Campo descrio
    gbc = new GBC(1, 1).horizontal().insets(0, 0, 5, 0).filly();
    panel.add(description, gbc);

    // Label classe de permisso
    gbc = new GBC(0, 2).northwest().insets(5, 0, 0, 5);
    panel.add(permissionClassL, gbc);

    // Campo classe de permisso
    gbc = new GBC(1, 2).horizontal().insets(0, 0, 5, 0).filly();
    panel.add(permissionClass, gbc);

    // Campo descrio da classe de permisso
    gbc = new GBC(1, 3).weights(1, 2).northwest().insets(5, 0, 0, 0).fillxy()
      .both();
    JScrollPane scrollDesc = new JScrollPane(permissionClassDesc);
    scrollDesc.setBorder(null);
    panel.add(scrollDesc, gbc);

    separator = new JSeparator();
    gbc = new GBC(0, 4).horizontal().width(3).insets(15, 0, 12, 0).fillx();
    panel.add(separator, gbc);
    separator.setVisible(false);

    return panel;
  }

  /**
   * Constri o componente com a lista de classes de permisso disponveis.
   *
   * @return o componente que mostra todas as classes de permisso
   */
  private JComboBox makeClassComboBox() {
    final JComboBox combo = new JComboBox();
    Vector<String> classes = PermissionProxy.getPermissionClasses(this,
      windowLabel, LNG.get("IAS_WAITING_ALL_PERMISSION_CLASSES"));
    Vector<PermissionClassName> classesNames = null;
    try {
      classesNames = makePermissionClassNames(classes);
      Collections.sort(classesNames);
      int classesSize = classesNames.size();
      for (int i = 0; i < classesSize; i++) {
        combo.addItem(classesNames.get(i));
      }
      if (classesSize > 0) {
        combo.setSelectedIndex(0);
      }
      combo.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(ItemEvent evt) {
          if (combo.getSelectedIndex() > -1) {
            Object item = evt.getItem();
            int stateChange = evt.getStateChange();
            classStateChanged(item, stateChange);
          }
        }
      });
      combo.setEnabled(isNew);
    }
    catch (Exception e) {
      combo.setEditable(false);
      StandardErrorDialogs.showErrorDialog(this,
        LNG.get("ERRO") + " - " + windowLabel, e);
    }
    return combo;
  }

  /**
   * Dado um Vector de classes de permisso, retorna um Vector com Wrappers de
   * classes de permisso, com os dados relevantes para exibio.
   *
   * @param listClasses Vector contendo as classes de permisso
   *
   * @return Vector contendo os Wrappers de classes de permisso
   *
   * @throws ClassNotFoundException caso ocorra problema na instanciao do
   *         Wrapper de permisso.
   */
  private Vector<PermissionClassName> makePermissionClassNames(
    Vector<String> listClasses) throws ClassNotFoundException {
    if (listClasses == null) {
      return null;
    }
    Vector<PermissionClassName> listClassNames =
      new Vector<PermissionClassName>();
    for (int i = 0; i < listClasses.size(); i++) {
      listClassNames.add(new PermissionClassName(listClasses.get(i)));
    }
    return listClassNames;
  }

  /**
   * Atualiza a exibio dos campos da tabela de acordo com o evento ocorrido
   * com o combo de classes de permisso. Se uma nova classe tiver sido
   * selecionada, exibe os campos correspondentes; se uma classe j selecionada
   * tiver sido desmarcada, esconde os campos correspondentes (OBS: sempre que o
   * usurio seleciona uma classe de permisso, dois eventos so disparados -
   * evento classe anterior desmarcada e evento nova classe selecionada).
   *
   * @param item Classe de permisso selecionada
   * @param stateChange Indica qual tipo de evento ocorreu (classe selecionada e
   *        classe desmarcada so eventos de interesse)
   * @return True, se o tem selecionado na combo de permisses corresponder a
   *         uma classe de pernisses. False, caso contrrio.
   */
  private boolean classStateChanged(Object item, int stateChange) {
    PermissionClassName permissionClassName = (PermissionClassName) item;
    if (permissionClassName == null) {
      return false;
    }
    // Atualiza a descrio da classe de permisso
    permissionClassDesc
      .setText(getPermissionClassDescription(permissionClassName));
    permissionClassDesc.setCaretPosition(0);
    selectedPermissionClass = permissionClassName.getSuperClassName();
    if (selectedPermissionClass == null) {
      StandardErrorDialogs.showErrorDialog(this,
        LNG.get("PermissionInfoDialog.error.msg.permission_class_not_found"));
      return false;
    }

    // De acordo com o item selecionado, exibe/oculta o "painel" associado
    if (stateChange == ItemEvent.SELECTED) {
      if (permissionAtributePanels.getCurrentCardName()
        .equals(selectedPermissionClass)
        && selectedPermissionClass.equals(CHOICE_PERMISSION_CLASS)) {
        permissionAtributePanels.show(SIMPLE_PERMISSION_CLASS);
      }
      permissionAtributePanels.show(selectedPermissionClass);
      if (selectedPermissionClass != SIMPLE_PERMISSION_CLASS) {
        ((AbstractPermissionPanel) permissionAtributePanels.getCurrentCard())
          .changeClass();
      }

      this.rootPane.revalidate();
    }

    pack();

    return true;
  }

  /**
   * Obtm a descrio da superclasse de permisso (ou classe genrica) a partir
   * do arquivo de propriedades.
   *
   * @param item Wrapper para exibio de uma permisso
   *
   * @return descrio textual da classe genrica
   */
  private String getPermissionClassDescription(PermissionClassName item) {
    String desc = "";
    if (item != null) {
      desc = LNG.get(item.getClassName() + DESCRIPTION_SUFFIX);
    }
    return desc;
  }

  /**
   * Cria um JTextArea contendo a descrio para a classe de permisso
   * atualmente selecionada no combo passado como parmetro.
   *
   * @param combo JComboBox no qual a classe de permisso a ser descrita est
   *        selecionada.
   *
   * @return JTextArea com a descrio da classe de permisso atualmente
   *         selecionada.
   */
  private JTextArea makePermissionClassDescription(JComboBox combo) {
    JTextArea textArea = new JTextArea(5, 30);
    textArea.setLineWrap(true);
    textArea.setWrapStyleWord(true);
    textArea.setEditable(false);
    textArea.setEnabled(false);
    textArea.setBorder(null);
    textArea.setCaretPosition(0);
    textArea.setDisabledTextColor(Color.BLACK);
    textArea.setFont(textArea.getFont().deriveFont(Font.PLAIN));
    textArea.setBackground(javax.swing.UIManager.getDefaults()
      .getColor("TextField.inactiveBackground"));
    return textArea;
  }

  /**
   * Inicializa os campos da janela com os dados da permisso a ser modificada
   * (a no ser que uma nova permisso esteja sendo criada). Atualiza a exibio
   * dos "painis" (grupamento de campos) opcionais.
   */
  protected void initFields() {
    // Listener para verificar alteraes do usurio nos campos da permisso
    KeyListener keyChangeListener = new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        checkDataChange();
      }
    };
    if (!isNew) {
      // Preenche os campos principais
      name.setText(permission.getName());
      description.setText(permission.getDescription());
      PermissionClassName className = null;
      try {
        className = new PermissionClassName(permission.getClass().getName());
      }
      catch (ClassNotFoundException e) {
        StandardErrorDialogs.showErrorDialog(this,
          LNG.get("ERRO") + " - " + windowLabel, e);
      }
      if (className != null) {
        permissionClass.setSelectedItem(className);
      }
      name.addKeyListener(keyChangeListener);
      description.addKeyListener(keyChangeListener);

      // Decide se os painis opcionais devem ou no ser mostrados
      if (!classStateChanged(permissionClass.getSelectedItem(),
        ItemEvent.SELECTED)) {
        StandardErrorDialogs.showErrorDialog(this,
          LNG.get("PermissionInfoDialog.error.msg.permission_class_not_found"));
        return;
      }

      permissionAtributePanels.show(selectedPermissionClass);
      if (!selectedPermissionClass.equals(SIMPLE_PERMISSION_CLASS)) {
        AbstractPermissionPanel permissionPanel =
          (AbstractPermissionPanel) permissionAtributePanels.getCurrentCard();
        permissionPanel.importFrom(permission);
        separator.setVisible(true);
      }
    }
    else {
      permissionClass.setSelectedIndex(0);
    }
  }

  /**
   * Cria um painel contendo o boto limpar.
   *
   * @return o painel criado com o boto de limpar campos
   */
  private JPanel makePermissionsButtonPanel() {
    JPanel panel = new JPanel();
    panel.add(previousButton);
    panel.add(nextButton);
    return panel;
  }

  /**
   * Constri o painel com os botes da janela de incluso/ modificao de
   * grupos de usurio. As aes previstas so para confirmar a operao
   * solicitada ou desistir da operao.
   *
   * @return o painel de botes
   */
  protected JPanel makeButtonPanel() {
    JPanel buttonPanel = new JPanel();
    if (isNew) {
      buttonPanel.add(makeAddPermissionButton());
      buttonPanel.add(makeCancelButton());
    }
    else {
      buttonPanel.add(makeChangeButton());
      buttonPanel.add(makeCloseButton());
    }
    return buttonPanel;
  }

  /**
   * Cria o boto de incluso de permisso.
   *
   * @return o boto de incluso de permisso.
   */
  protected JButton makeAddPermissionButton() {
    includePermissionButton = new JButton(LNG.get("IAS_INCLUDE"));
    includePermissionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        switch (validateFields()) {
          case FIELDS_OK:
            createPermission();
            break;
          case EMPTY_FIELDS:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_FILL_ALL_FIELDS"));
            break;
          case PASSWORDS_DONT_MATCH:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_CONFIRM_PERMISSION_PASSWORD_ERROR"));
            break;
          case ATTRIBUTES_ERROR:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_CONFIRM_PERMISSION_ATTRIBUTES_ERROR"));
            break;
        }
      }
    });
    return includePermissionButton;
  }

  /**
   * Cria o boto de modificao de permisso. Esse boto somente  habilitado
   * quando alguma modificao  feita nos campos.
   *
   * @return o boto de modificao de dados da permisso.
   */
  protected JButton makeChangeButton() {
    includePermissionButton = new JButton(LNG.get("IAS_PERMISSION_UPDATE"));
    includePermissionButton.setEnabled(false);
    includePermissionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        switch (validateFields()) {
          case FIELDS_OK:
            modifyPermission();
            PermissionInfoDialog.this.close();
            break;
          case EMPTY_FIELDS:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_FILL_ALL_FIELDS"));
            break;
          case PASSWORDS_DONT_MATCH:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_CONFIRM_PERMISSION_PASSWORD_ERROR"));
            break;
          case ATTRIBUTES_ERROR:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_CONFIRM_PERMISSION_ATTRIBUTES_ERROR"));
            break;
        }
      }
    });
    return includePermissionButton;
  }

  /**
   * Cria o boto de cancelamento de operao.
   *
   * @return o boto de cancelamento.
   */
  protected JButton makeCancelButton() {
    JButton cancelButton = new JButton(LNG.get("IAS_CLOSE"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        close();
      }
    });
    return cancelButton;
  }

  /**
   * Cria o boto de fechamento da janela.
   *
   * @return o boto de fechamento.
   */
  protected JButton makeCloseButton() {
    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        // Se os campos nao forem alterados ou estiverem em branco, fecha
        // a janela.
        if (!hasChanged() || isEmpty()) {
          close();
          return;
        }
        Object[] options = { LNG.get("IAS_UPDATE"), LNG.get("IAS_CANCEL") };
        int selected = JOptionPane.showOptionDialog(PermissionInfoDialog.this,
          LNG.get("IAS_PERMISSION_UPDATE_CONFIRMATION"), windowLabel,
          JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE, null,
          options, options[0]);
        if (options[selected] == LNG.get("IAS_CANCEL")) {
          close();
          return;
        }
        switch (validateFields()) {
          case FIELDS_OK:
            modifyPermission();
            close();
            break;
          case EMPTY_FIELDS:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_FILL_ALL_FIELDS"));
            break;
          case PASSWORDS_DONT_MATCH:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_CONFIRM_PERMISSION_PASSWORD_ERROR"));
            break;
          case ATTRIBUTES_ERROR:
            StandardErrorDialogs.showErrorDialog(PermissionInfoDialog.this,
              windowLabel, LNG.get("IAS_CONFIRM_PERMISSION_ATTRIBUTES_ERROR"));
            break;
        }

      }
    });
    return closeButton;
  }

  /**
   * Verifica se o campo de nome est preenchido (o campo de descrio no 
   * obrigatrio). Caso a permisso seja do tipo autenticada, os campos de
   * usurio e senha tambm so requeridos. A ausncia de parmetros para
   * permisses que aceitam atributos  vlida e portanto no verificada.
   *
   * @return Resultado da validao dos campos.
   *
   * @see csbase.logic.AttributesPermission
   */
  protected FieldsState validateFields() {
    // Campos principais
    String nameText = name.getText().trim();
    if (nameText.equals("") || selectedPermissionClass == null) {
      return FieldsState.EMPTY_FIELDS;
    }
    if (selectedPermissionClass != SIMPLE_PERMISSION_CLASS) {
      AbstractPermissionPanel permissionPanel =
        (AbstractPermissionPanel) permissionAtributePanels.getCurrentCard();
      return permissionPanel.validateFields();
    }
    return FieldsState.FIELDS_OK;
  }

  /**
   * Limpa todos os campos.
   */
  private void clearFields() {
    if (isNew) {
      permission = null;
    }
    name.setText("");
    description.setText("");
    if (selectedPermissionClass != SIMPLE_PERMISSION_CLASS) {
      ((AbstractPermissionPanel) permissionAtributePanels.getCurrentCard())
        .clearFields();
    }
    name.requestFocus();
  }

  /**
   * Verifica se os dados foram modificados. Em caso positivo, habilita o boto
   * de alterao da permisso.
   */
  void checkDataChange() {
    if (hasChanged()) {
      includePermissionButton.setEnabled(true);
    }
    else {
      includePermissionButton.setEnabled(false);
    }
  }

  /**
   * Verifica se algum dos campos de preenchimento de dados da permisso foi
   * alterado.
   *
   * @return true se algum campo foi editado ou false caso contrario
   */
  private boolean hasChanged() {
    if (!name.getText().trim().equals(permission.getName())) {
      return true;
    }
    if (!description.getText().trim().equals(permission.getDescription())) {
      return true;
    }
    if (selectedPermissionClass == null) {
      return false;
    }
    if (!selectedPermissionClass.equals(SIMPLE_PERMISSION_CLASS)) {
      return ((AbstractPermissionPanel) permissionAtributePanels
        .getCurrentCard()).hasChanged(permission);
    }
    return false;
  }

  /**
   * Verifica se os campos de preenchimento de dados da permisso esto sem
   * preenchimento.
   *
   * @return true se os campos estiverem vazios ou false caso contrario
   */
  private boolean isEmpty() {
    return ((name.getText().trim().equals(""))
      && description.getText().trim().equals(""));
  }

  /**
   * Obtm um objeto permisso com os dados digitados pelo usurio.
   *
   * @return permisso com os dados digitados pelo usurio.
   */
  public Permission getScreenPermission() {
    Permission screenPermission = null;

    // Obtm os campos principais
    String nameText = name.getText().trim();
    String descriptionText = description.getText().trim();

    // Instancia a classe de permisso
    PermissionClassName item =
      (PermissionClassName) permissionClass.getSelectedItem();
    try {
      Class<?> selectedClass = Class.forName(item.getClassName());
      Constructor<?> constructor = selectedClass.getConstructor();
      screenPermission = (Permission) constructor.newInstance();

      screenPermission.setName(nameText);
      screenPermission.setDescription(descriptionText);

      if (!selectedPermissionClass.equals(SIMPLE_PERMISSION_CLASS)) {
        ((AbstractPermissionPanel) permissionAtributePanels.getCurrentCard())
          .exportTo(screenPermission);
      }
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(this,
        LNG.get("ERRO") + " - " + windowLabel, e);
    }
    return screenPermission;
  }

  /**
   * Executa a ao de criar a permisso, usando os dados preenchidos nos campos
   * da janela.
   */
  private void createPermission() {
    permission = getScreenPermission();
    if (permission == null) {
      return;
    }
    // Persiste o objeto de permisso
    permission = PermissionProxy.createPermission(this, windowLabel,
      LNG.get("IAS_WAITING_PERMISSION_CREATION"), permission);
    if (permission != null) {
      String msg = LNG.get("IAS_PERMISSION_INCLUDED_WITH_SUCCESS");
      msg = MessageFormat.format(msg, new Object[] { permission.getName() });
      StandardDialogs.showInfoDialog(this, windowLabel, msg);
      permission = null;
      clearFields();
    }
  }

  /**
   * Executa a ao de modificar a permisso, usando os dados preenchidos nos
   * campos da janela.
   */
  private void modifyPermission() {
    Permission modifiedPermission = getScreenPermission();
    if (modifiedPermission == null) {
      return;
    }
    modifiedPermission = PermissionProxy.modifyPermission(this, windowLabel,
      LNG.get("IAS_WAITING_PERMISSION_MODIFICATION"), permission.getId(),
      modifiedPermission);
    if (modifiedPermission != null) {
      this.permission = modifiedPermission;
      StandardDialogs.showInfoDialog(this, windowLabel,
        MessageFormat.format(LNG.get("IAS_PERMISSION_MODIFIED_WITH_SUCCESS"),
          new Object[] { permission.getName() }));
      if (selectedPermissionClass.equals(USER_PASSWORD_PERMISSION_CLASS)) {
        ((UserPasswordPanel) permissionAtributePanels.getCurrentCard())
          .clearPasswordFields();
      }
      includePermissionButton.setEnabled(false);
    }
  }

  /**
   * @return Nome da permisso selecionada na combo de permisses.
   */
  PermissionClassName getPermissionClassName() {
    return (PermissionClassName) permissionClass.getSelectedItem();
  }

  /**
   * Classe que funciona como um Wrapper de classe de permisso, guardando
   * somente os dados relevantes para exibio.
   */
  class PermissionClassName implements Comparable<PermissionClassName> {
    /** Nome classe da permisso. */
    private String className;

    /** Nome da permisso. */
    private String presentationName;

    /** Nome da superclasse da permisso. */
    private String superClassName;

    /**
     * Cria o wrapper de permisso.
     *
     * @param className Nome classe da permisso.
     * @throws ClassNotFoundException caso no seja possvel instanciar a
     *         superclasse.
     */
    public PermissionClassName(String className) throws ClassNotFoundException {
      this.className = className;
      this.presentationName = LNG.get(className);
      this.superClassName = getSuperClassName(className);
    }

    /**
     * Obtm a classe base da permisso da qual herda.
     *
     * @param className
     * @return Classe base da permisso.
     * @throws ClassNotFoundException
     */
    private String getSuperClassName(String className)
      throws ClassNotFoundException {
      Class<?> permissionClass = Class.forName(className);
      Class<?> attributes = Class.forName(ATTRIBUTES_PERMISSION_CLASS);
      Class<?> userPass = Class.forName(USER_PASSWORD_PERMISSION_CLASS);
      Class<?> choicePermission = Class.forName(CHOICE_PERMISSION_CLASS);
      if (attributes.isAssignableFrom(permissionClass)) {
        return ATTRIBUTES_PERMISSION_CLASS;
      }
      else if (userPass.isAssignableFrom(permissionClass)) {
        return USER_PASSWORD_PERMISSION_CLASS;
      }
      else if (choicePermission.isAssignableFrom(permissionClass)) {
        return CHOICE_PERMISSION_CLASS;
      }
      else {
        return SIMPLE_PERMISSION_CLASS;
      }
    }

    /**
     * Obtm o nome da superclasse da permisso
     *
     * @return superclasse (ou classe genrica) da permisso
     */
    public String getSuperClassName() {
      return superClassName;
    }

    /**
     * Obtm a classe da permisso
     *
     * @return classe da permisso
     */
    public String getClassName() {
      return className;
    }

    /**
     * Obtm o nome a ser exibido para apresentao da permisso.
     *
     * @return nome de apresentao (em contraste ao nome lgico) da permisso
     */
    public String getPresentationName() {
      return presentationName;
    }

    /**
     * Compara uma permisso a outra, para decidir seu posicionamento em uma
     * lista ordenada.
     *
     * @param o .
     *
     * @return int -1 se for "menor", 0 se for "igual", 1 se for "maior"
     */
    @Override
    public int compareTo(PermissionClassName o) {
      return this.presentationName.compareTo((o).presentationName);
    }

    /**
     * Verifica se o objeto passado como parmetro  igual ao objeto no qual o
     * mtodo foi invocado.
     *
     * @param o o objeto a ser comparado
     *
     * @return true se ambos forem iguais
     */
    @Override
    public boolean equals(Object o) {
      return (o instanceof PermissionClassName
        && ((PermissionClassName) o).presentationName
          .equals(this.presentationName));
    }

    /**
     * Calcula o cdigo hash do objeto.
     *
     * @return Inteiro que representa o cdigo hash do objeto.
     */
    @Override
    public int hashCode() {
      // XXX presentationName pode ser nulo?
      if (presentationName == null) {
        return "presentationName is null".hashCode();
      }
      return presentationName.hashCode();
    }

    /**
     * Retorna uma representao textual do objeto
     *
     * @return representao textual do objeto
     */
    @Override
    public String toString() {
      return presentationName;
    }
  }
}
