package csbase.client.openbus;

import csbase.client.desktop.DesktopFrame;

/**
 * Tratador de eventos padro nulo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class OpenBusEventHandler implements OpenBusEventHandlerInterface {

  /**
   * {@link DesktopFrame#getClientInstanceId()}
   * 
   * @return o id
   */
  final protected String getClientInstanceId() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    return desktop.getClientInstanceId();
  }

  /**
   * {@link DesktopFrame#getFatherClientInstanceId()}
   * 
   * @return o id
   */
  final protected String getFatherClientInstanceId() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    return desktop.getFatherClientInstanceId();
  }

  /**
   * Verifica se um id  correspondente ao meu cliente.
   * 
   * @param id id a ser verificado
   * @return indicativo
   */
  final protected boolean isMyId(final String id) {
    if (isAnonymousId(id)) {
      return false;
    }
    final String myId = getClientInstanceId();
    if (myId == null) {
      return false;
    }
    final boolean isMe = myId.equals(id.trim());
    return isMe;
  }

  /**
   * Verifica se um id  correspondente ao meu cliente.
   * 
   * @param id id a ser verificado
   * @return indicativo
   */
  final protected boolean isMyFatherId(final String id) {
    if (id == null) {
      return false;
    }
    final String fatherId = getFatherClientInstanceId();
    if (fatherId == null) {
      return false;
    }
    final boolean isMyFather = id.equals(fatherId.trim());
    return isMyFather;
  }

  /**
   * Indica um id (destino ou origem)  annimo.
   * 
   * @param id id a ser testado.
   * @return indicativo
   */
  final protected boolean isAnonymousId(String id) {
    return (id == null);
  }

  /**
   * Indica se uma mensagem  endereada diretamente entre um cliente pai que
   * disparou efetivamente o cliente corrente.
   * 
   * @param sourceId id da origem
   * @param destId id do destino
   * @return indicativo
   * 
   *         Veja definio de: {@link #isMyId(String)} Veja definio de:
   *         {@link #isMyFatherId(String)}
   */
  final protected boolean isEventFromFatherAndToMe(String sourceId,
    String destId) {
    if (isAnonymousId(destId)) {
      return false;
    }
    if (isAnonymousId(sourceId)) {
      return false;
    }
    if (!isMyId(destId)) {
      return false;
    }
    if (!isMyFatherId(sourceId)) {
      return false;
    }
    return true;
  }

  /**
   * Indica se uma mensagem  endereada diretamente para o cliente corrente e
   * tem origem definida.
   * 
   * @param sourceId id da origem
   * @param destId id do destino
   * @return indicativo
   * 
   *         Veja definio de: {@link #isMyId(String)} e
   *         {@link #isAnonymousId(String)}.
   */
  final protected boolean isEventFromIdentifiedSourceAndToMe(String sourceId,
    String destId) {
    if (isAnonymousId(sourceId)) {
      return false;
    }
    if (!isMyId(destId)) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithNoValue(String sourceId, String destId,
    String type) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithStringValue(String sourceId, String destId,
    String type, String stringValue) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithBooleanValue(String sourceId, String destId,
    String type, boolean booleanValue) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean sendEventWithNoValue(String sourceId, String destId,
    String type) {
    final OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
    final boolean treated =
      oap.getSession().internalSendEventWithNoValue(sourceId, destId, type);
    logSentEvent(treated, sourceId, destId, type, null);
    return treated;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean sendEventWithStringValue(String sourceId, String destId,
    String type, String stringValue) {
    final OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
    final boolean treated =
      oap.getSession().internalSendEventWithStringValue(sourceId, destId, type,
        stringValue);
    logSentEvent(treated, sourceId, destId, type, stringValue);
    return treated;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean sendEventWithBooleanValue(String sourceId,
    String destId, String type, boolean booleanValue) {
    final OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
    final boolean treated =
      oap.getSession().internalSendEventWithBooleanValue(sourceId, destId,
        type, booleanValue);
    logSentEvent(treated, sourceId, destId, type, booleanValue);
    return treated;
  }

  /**
   * Construtor padro que se cadastra para receber e enviar evventos.
   */
  public OpenBusEventHandler() {
    final OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
    oap.getSession().addEventReceiver(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logReceivedEvent(boolean treated, String sourceId, String destId,
    String type, Object value) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logSentEvent(boolean treated, String sourceId, String destId,
    String type, Object value) {
  }
}
