/*
 * UserProxy.java @author Andr Luiz Clinio $Revision: 175519 $ - $Date:
 * 2006-08-15 12:00:05 -0300 (Tue, 15 Aug 2006) $
 */
package csbase.client.remote.srvproxies;

import java.awt.Window;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.administration.AdministrationDeleteException;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.logic.UserOutline;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe que modela um proxy Cliente/User.
 * 
 * @author Tecgraf
 */
public class UserProxy {
  /**
   * Obtem a lista de outlines de usuarios cadastrados. TODO - Mudar tipo de
   * retorno para {@code List<UserOutline>}
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * 
   * @return o vetor de usurios.
   */
  public static Vector<UserOutline> getAllOutlines(Window owner, String title,
    String message) {
    RemoteTask<Vector<UserOutline>> task =
      new RemoteTask<Vector<UserOutline>>() {
        @Override
        public void performTask() throws Exception {
          setResult(new Vector<UserOutline>(User.getAllOutlines()));
        }
      };
    if (task.execute(owner, title, message)) {
      Vector<UserOutline> outlinesVector = task.getResult();
      if ((outlinesVector == null) || (outlinesVector.size() <= 0)) {
        return new Vector<UserOutline>();
      }
      return outlinesVector;
    }
    return new Vector<UserOutline>();
  }

  /**
   * Obtem a lista de usuarios cadastrados. TODO - Mudar tipo de retorno para
   * {@code List<User>}
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * 
   * @return o vetor de usurios.
   */
  public static Vector<User> getAllUsers(Window owner, String title,
    String message) {
    RemoteTask<Vector<User>> task = new RemoteTask<Vector<User>>() {
      @Override
      public void performTask() throws Exception {
        setResult(new Vector<User>(User.getAllUsers()));
      }
    };
    if (task.execute(owner, title, message)) {
      Vector<User> usersVector = task.getResult();
      if ((usersVector == null) || (usersVector.size() <= 0)) {
        return new Vector<User>();
      }

      return usersVector;
    }
    return new Vector<User>();
  }

  /**
   * Obtm um usurio com uma dada identificao.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param userId o identificador do usurio.
   * 
   * @return o usurio requisitado.
   */
  public static User getUser(Window owner, String title, String message,
    final Object userId) {
    RemoteTask<User> task = new RemoteTask<User>() {
      @Override
      public void performTask() throws Exception {
        setResult(User.getUser(userId));
      }
    };
    if (task.execute(owner, title, message)) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Cria um usurio.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param info As informaes do novo usurio.
   * 
   * @return O usurio criado.
   */
  public static User createUser(Window owner, String title, String message,
    final UserInfo info) {
    RemoteTask<User> task = new RemoteTask<User>() {
      @Override
      public void performTask() throws Exception {
        setResult(User.createUser(info));
      }
    };
    if (task.execute(owner, title, message)) {
      User user = task.getResult();
      if (user == null) {
        // Usurio j existente
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_USER_ALREADY_EXISTS_ERROR"));
      }
      return user;
    }
    return null;
  }

  /**
   * Busca a foto de um usurio
   * 
   * @param owner .
   * @param userId o identificador do usurio.
   * 
   * @return a imagem.
   */
  public static ImageIcon getPhoto(final Window owner, final Object userId) {
    RemoteTask<ImageIcon> task = new RemoteTask<ImageIcon>() {
      @Override
      public void performTask() throws Exception {
        setResult(User.getPhoto(userId));
      }
    };
    if (task.execute(owner, "", LNG.get("IAS_PHOTO_WAITING"))) {
      ImageIcon img = task.getResult();
      if (img == null) {
        img = new ImageIcon();
      }
      return img;
    }
    return null;
  }

  /**
   * Modifica um usurio.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param userId o identificador do usurio a ser alterado.
   * @param info As informaes de modificao do usurio.
   * 
   * @return O usurio modificado.
   */
  public static User modifyUser(Window owner, String title, String message,
    final Object userId, final UserInfo info) {
    RemoteTask<User> task = new RemoteTask<User>() {
      @Override
      public void performTask() throws Exception {
        setResult(User.modifyUser(userId, info));
      }
    };
    if (task.execute(owner, title, message)) {
      User user = task.getResult();
      if (user == null) {
        // Usurio no encontrado
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_USER_NOT_FOUND_ERROR"));
      }
      return user;
    }
    return null;
  }

  /**
   * Modifica uma lista de usurios.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param users Lista de usurios que sero alterados.
   */
  public static void modifyUsers(Window owner, String title, String message,
    final List<User> users) {
    final List<User> usersNotFound = new ArrayList<User>();

    RemoteTask<User> task = new RemoteTask<User>() {
      @Override
      public void performTask() throws Exception {
        for (User user : users) {
          User modifiedUser = User.modifyUser(user.getId(), user.getUserInfo());
          if (modifiedUser == null) {
            usersNotFound.add(user);
          }
        }
      }
    };

    task.execute(owner, title, message);

    if (usersNotFound.size() > 0) {
      // Usurios no encontrado
      String msg = "<html>";
      msg += LNG.get("IAS_USERS_NOT_FOUND_ERROR");
      for (int i = 0; i < usersNotFound.size(); i++) {
        User user = usersNotFound.get(i);
        msg += user.getId();
        if (i > 200) {
          break;
        }
        if (i < usersNotFound.size() - 1) {
          msg += ", ";
        }
        if (i % 20 == 0 && i != 0) {
          msg += "<br/>";
        }
      }
      msg += ".</html>";
      StandardErrorDialogs.showErrorDialog(owner, title, msg);
    }
  }

  /**
   * Remove um usurio.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param userId o identificador do usurio a ser removido.
   */
  public static void deleteUser(final Window owner, final String title,
    String message, final Object userId) {
    class DeleteUserTask extends RemoteTask<Void> {
      boolean removeProjects = false;

      @Override
      public void performTask() throws Exception {
        User.deleteUser(userId);
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof AdministrationDeleteException) {
          String errorMsg = LNG.get("IAS_CONFIRM_USER_REMOVAL");
          int res = JOptionPane.showOptionDialog(owner, errorMsg, title,
            JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null,
            new Object[] { LNG.get("IAS_YES"), LNG.get("IAS_NO") }, LNG.get(
              "IAS_NO"));
          if (res == 0) {
            removeProjects = true;
          }
        }
        else {
          super.handleError(error);
        }
      }
    }
    DeleteUserTask deleteTask = new DeleteUserTask();
    if (!deleteTask.execute(owner, title, message)
      && deleteTask.removeProjects) {

      RemoteTask<Boolean> rtask = new RemoteTask<Boolean>() {
        @Override
        public void performTask() throws Exception {
          boolean projectsRemoved = CommonClientProject.removeUserProjects(
            userId);
          if (projectsRemoved) {
            User.deleteUser(userId);
          }
          setResult(projectsRemoved);
        }
      };
      rtask.execute(owner, title, message);
      if (!rtask.getResult()) {
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_USER_PROJECTS_NOT_DELETED"));
      }
    }
  }
}
