package csbase.client.remote.srvproxies.messageservice;

import java.awt.Window;
import java.rmi.RemoteException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import tecgraf.javautils.core.filter.CompositeFilter;
import tecgraf.javautils.core.filter.IFilter;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.ClientServerManager;
import csbase.client.desktop.DesktopFrame;
import csbase.client.remote.srvproxies.messageservice.consumers.AdaptableMessageConsumer;
import csbase.client.remote.srvproxies.messageservice.consumers.IMessageConsumer;
import csbase.client.remote.srvproxies.messageservice.consumers.IMessageConsumer.IListener;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.remote.ClientRemoteLocator;
import csbase.util.data.channel.DataChannel;
import csbase.util.data.dispatcher.DispatcherAdapter;
import csbase.util.data.dispatcher.ExecutorDispatcher;
import csbase.util.data.dispatcher.IDispatchListener;
import csbase.util.data.dispatcher.IDispatcher;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.MessageListenerDispatcher;
import csbase.util.messages.filters.BodyTypeFilter;

/**
 * Proxy para o servio de mensagens. Ela escolhe a melhor estratgia para se
 * obter as mensagens do servio de mensagens, tirando esse encargo do usurio.
 *
 * @author Tecgraf
 */
public class MessageProxy {

  /** Tempo de vida padro de uma mensagem, em dias. */
  public static final int DEFAULT_TIME_TO_LIVE = 10;

  /** Estratgia para consumir as mensagens do servio de mensagens. */
  private static AdaptableMessageConsumer consumer;
  /**
   * Responsvel por receber as mensagens obtidas pela estrutura da estratgia
   * de consumo, e repassar para o barramento de mensagens.
   */
  private static IMessageConsumer.IListener consumerListener;
  /**
   * Canal responsvel por difundir as mensagens entre os ouvintes.
   */
  private static DataChannel<IMessageListener, Message> channel;
  /**
   * Mapa de filtros de mensagens pelo o ouvinte. Utilizado para guardar quais
   * filtros devero ser repassados ao consumidor.
   */
  private static Map<IMessageListener, IFilter<Message>> filtersByListener;

  static {
    consumer = new AdaptableMessageConsumer();
    filtersByListener = new HashMap<IMessageListener, IFilter<Message>>();

    /*
     * Cria o despachante de mensagens que ir entregar as mensagens aos
     * ouvintes do cliente.
     */
    IDispatcher<IMessageListener, Message> dispatcher =
      new ExecutorDispatcher<IMessageListener, Message>(
        new ThreadPoolExecutor(0, 10, 60L, TimeUnit.SECONDS,
          new LinkedBlockingQueue<Runnable>()),
        new MessageListenerDispatcher());
    /*
     * Cria o ouvinte do despachante que ir remover do canal de mensagens os
     * ouvinte que lanarem exceo.
     */
    IDispatchListener<IMessageListener, Message> dispatchListener =
      new DispatcherAdapter<IMessageListener, Message>() {
        @Override
        public void onExceptionThrown(Exception e, IMessageListener listener,
          Message... messages) {
          // Remove o listener da estrutura.
          channel.unsubscribe(listener);
          filtersByListener.remove(listener);
          onFiltersChanged();

          /*
           * No queremos que aparea uma janela de erro com uma exceo cada
           * vez que o servidor cai. J mostramos uma mensagem indicando que o
           * servidor caiu...
           */
          // showError(LNG.get("MESSAGE_PROXY_CLIENT_DISPATCH_FAILURE"), e);
        }
      };
    channel =
      new DataChannel<IMessageListener, Message>(dispatcher, dispatchListener);

    consumerListener = new IListener() {
      @Override
      public void onMessagesReceived(Message... messages) {
        channel.publish(messages);
      }

      @Override
      public void onExceptionThrown(Exception e) {
        // Verifica se  uma exceo gerada por problemas de conexo com o servidor.
        if (e instanceof RemoteException) {
          /*
           * Se for, deixamos o ClientServerManager cuidar dela, pedindo que
           * revalide a conexo com o servidor. Caso a conexo tenha cado, a
           * revalidao ir criar um dilogo indicando a queda ao cliente.
           */
          ClientServerManager.getInstance().invalidate();
        }
        else {
          /*
           * Caso contrrio, apresentamor uma janela de erro com a exceo.
           */
          /*
           * No queremos que aparea uma janela de erro com uma exceo cada
           * vez que o servidor cai. J mostramos uma mensagem indicando que o
           * servidor caiu...
           */
          //showError(LNG.get("MESSAGE_PROXY_SERVER_RECEIVE_FAILURE"), e);
        }
      }
    };
  }

  /**
   * Remove os ouvintes de mensagem, interrompendo assim o consumo de mensagens
   * do servidor.
   */
  public static void clearListeners() {
    consumer.clearListener();
    channel.unsubscribeAll();
    filtersByListener.clear();
  }

  /**
   * <p>
   * Envia uma mensagem, com tempo de vida padro, a vrios destinatrios.
   * </p>
   * <p>
   * O tempo de vida padro  {@value #DEFAULT_TIME_TO_LIVE} dias.
   * </p>
   *
   * @param message Mensagem a ser enviada.
   * @param usersLogin Login dos usurios de destino.
   *
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   * @throws CSBaseException Se no foi possvel entregar a mensagem por
   *         problemas que no sejam de comunicao.
   */
  public static void send(Message message, String... usersLogin)
    throws CSBaseException, RemoteException {
    long timeToLive = TimeUnit.DAYS.toMillis(DEFAULT_TIME_TO_LIVE);
    send(message, timeToLive, usersLogin);
  }

  /**
   * Envia uma mensagem a vrios destinatrios.
   *
   * @param message Mensagem a ser enviada.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   * @param usersLogin Login dos usurios de destino.
   *
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   * @throws CSBaseException Se no foi possvel entregar a mensagem por
   *         problemas que no sejam de comunicao.
   */
  public static void send(Message message, long timeToLive,
    String... usersLogin) throws CSBaseException, RemoteException {
    ClientRemoteLocator.messageService.send(message, timeToLive, usersLogin);
  }

  /**
   * <p>
   * Envia uma mensagem para si mesmo.
   * </p>
   * <p>
   * Essa mensagem no vai para o servidor,  utilizada para comunicao interna
   * do cliente.
   * </p>
   *
   * @param message Mensagem a ser enviada.
   */
  public static void sendLocal(Message message) {
    channel.publish(message);
  }

  /**
   * <p>
   * Envia uma mensagem, com tempo de vida padro, para todos os usurios.
   * </p>
   * <p>
   * O tempo de vida padro  {@value #DEFAULT_TIME_TO_LIVE} dias.
   * </p>
   *
   * @param message Mensagem a ser enviada.
   *
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   * @throws CSBaseException Se no foi possvel entregar a mensagem por
   *         problemas que no sejam de comunicao.
   */
  public static void broadcast(Message message) throws CSBaseException,
    RemoteException {
    long timeToLive = TimeUnit.DAYS.toMillis(DEFAULT_TIME_TO_LIVE);
    broadcast(message, timeToLive);
  }

  /**
   * Envia uma mensagem para todos os usurios.
   *
   * @param message Mensagem a ser enviada.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   *
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   * @throws CSBaseException Se no foi possvel entregar a mensagem por
   *         problemas que no sejam de comunicao.
   */
  public static void broadcast(Message message, long timeToLive)
    throws CSBaseException, RemoteException {
    ClientRemoteLocator.messageService.broadcast(message, timeToLive);
  }

  /**
   * Adiciona um ouvinte interessado em receber mensagens de um determinado
   * tipo.
   *
   * @param listener O ouvinte de mensagens.
   * @param filter Filtro que determina as mensagens que sero repassadas ao
   *        ouvinte.
   */
  public static synchronized void addListener(IMessageListener listener,
    IFilter<Message> filter) {

    channel.subscribe(listener, filter);
    filtersByListener.put(listener, filter);
    onFiltersChanged();
  }

  /**
   * Adiciona um ouvinte interessado em receber mensagens que carreguem um
   * determinado tipo de dado.
   *
   * @param listener o ouvinte de mensagens.
   * @param bodyType filtro apenas mensagens cujo corpo seja uma instncia de
   *        bodyType.
   */
  public static void addListener(IMessageListener listener,
    final Class<?> bodyType) {
    IFilter<Message> filter = new BodyTypeFilter(bodyType);
    addListener(listener, filter);
  }

  /**
   * <p>
   * Remove um determinado ouvinte.
   * </p>
   * <p>
   * Uma vez removido, esse ouvinte no ir mais receber mensagens a menos que
   * seja novamente inserido atravs do mtodo
   * {@link #addListener(IMessageListener, IFilter)}.
   * </p>
   *
   * @param listener Ouvinte a ser removido.
   */
  public static synchronized void removeListener(IMessageListener listener) {
    channel.unsubscribe(listener);
    filtersByListener.remove(listener);
    onFiltersChanged();
  }

  /**
   * <p>
   * Remove, atribui ou reatribui com um novo filtro, o ouvinte da estratgia de
   * consumo de mensagens.
   * </p>
   * <p>
   *  utilizado quando for necessrio substituir o filtro de mensagens anexado
   * a estratgia de consumo. Isso ocorre quando ouvintes de mensagens so
   * inseridos ou removidos desta estrutura.
   * </p>
   *
   * @see #addListener(IMessageListener, Class)
   * @see #addListener(IMessageListener, IFilter)
   */
  private static void onFiltersChanged() {
    final Collection<IFilter<Message>> filters = filtersByListener.values();
    if (filters.size() == 0) {
      consumer.clearListener();
    }
    else {
      CompositeFilter<Message> composedFilter =
        new CompositeFilter<Message>(filters);
      consumer.setListener(consumerListener, composedFilter);
    }
  }

  /**
   * Mtodo de aviso de erro
   *
   * @param msg mensagem a ser exibida.
   * @param e exceo causadora do erro (ou null)
   */
  private static void showError(String msg, Exception e) {
    String title = LNG.get("ERRO") + " - " + LNG.get("MESSAGE_PROXY_TITLE");

    Window window = null;
    DesktopFrame mainFrame = DesktopFrame.getInstance();
    if (mainFrame != null) {
      window = mainFrame.getDesktopFrame();
    }

    if (e != null) {
      StandardErrorDialogs.showErrorDialog(window, title, msg, e);
    }
    else {
      StandardErrorDialogs.showErrorDialog(window, title, msg);
    }
  }
}
