/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree;

import java.awt.datatransfer.DataFlavor;
import java.io.File;
import java.io.Serializable;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.tree.TreeNode;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.applications.algorithmsmanager.versiontree.actions.CopyFile;
import csbase.client.applications.algorithmsmanager.versiontree.actions.ExportExecutableFile;
import csbase.client.applications.algorithmsmanager.versiontree.actions.ImportExecutableFile;
import csbase.client.applications.algorithmsmanager.versiontree.actions.PasteFile;
import csbase.client.applications.algorithmsmanager.versiontree.actions.RemoveExecutableFiles;
import csbase.client.applications.algorithmsmanager.versiontree.actions.VersionWarningActionDecorator;
import csbase.client.applications.algorithmsmanager.versiontree.datatransfer.VersionTreeTransferHandler;
import csbase.logic.FileInfo;
import csbase.logic.IPathFactory;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.ExecutablePathFactory;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         N que representa arquivos executveis.
 */
final class ExecutableFileNode extends AbstractFileInfoNode {

  /**
   * Descreve o formato de objeto que pode ser transferido pelas operaces de
   * transferncia (drag&drop ou clipboard) na rvore.
   */
  public static final DataFlavor DATA_FLAVOR = new DataFlavor(
    ExecutableFileNode.class, ExecutableFileNode.class.getSimpleName());

  /** Plataforma que detm este n. */
  private final String platform;

  /**
   * Construtor.
   * 
   * @param tree rvore que detm este n.
   * @param version Verso que detm o arquivo representado por este n.
   * @param platform Plataforma que detm o arquivo representado por este n.
   * @param file Arquivo representado por este n.
   */
  public ExecutableFileNode(final VersionTree tree,
    final AlgorithmVersionInfo version, final String platform,
    final FileInfo file) {
    super(tree, version, file, file1 -> new ExecutableFileNode(tree, version, platform, file1), true);

    this.platform = platform;
  }

  /**
   * Cria um menu pop-up para este n.
   * 
   * @return um menu pop-up para este n.
   */
  @Override
  public JPopupMenu createPopupMenu() {
    JPopupMenu popupMenu = new JPopupMenu();
    popupMenu.add(new JMenuItem(new VersionWarningActionDecorator(new CopyFile(
      getTree(), LNG.get("algomanager.menu.bin_file.copy")))));
    if (getFile().isDirectory()) {
      popupMenu.add(new JMenuItem(new VersionWarningActionDecorator(
        new ImportExecutableFile(getTree(), getVersion(), platform,
          getExecFileDestPath()))));
      popupMenu.add(new JMenuItem(new VersionWarningActionDecorator(
        new PasteFile(getTree(), LNG.get("algomanager.menu.bin_file.paste"),
          ExecutableFileNode.DATA_FLAVOR))));
    }
    popupMenu.add(new JMenuItem(new VersionWarningActionDecorator(
      new RemoveExecutableFiles(getTree(), getVersion(), platform,
        getSelectedFiles()))));
    if (!getFile().isDirectory()) {
      popupMenu
        .add(new JMenuItem(
          new VersionWarningActionDecorator(new ExportExecutableFile(getTree(),
            getVersion(), platform, getFile()))));
    }

    return popupMenu;
  }

  /**
   * Obtm o path de destino de importao do arquivo executvel.O path 
   * construdo a partir do n plataforma.
   *
   * @return o path de destino de importao do arquivo executvel.
   */
  private String getExecFileDestPath() {
    // No precisa testar se o n  Plataforma pois so objetos distintos.
    boolean found = false;
    StringBuilder name = new StringBuilder();

    for (TreeNode node : getPath()) {
      if (found) {
        name.append(File.separator).append(node.toString());
      }
      if (node.toString().equals(platform)) {
        found = true;
      }
    }
    return name.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataFlavor getDataFlavor() {
    return DATA_FLAVOR;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final DataTransferObject createDataTransferObject() {
    return new ExecutableDTO(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final IPathFactory createSourcePathFactory(DataTransferObject dto) {
    AlgorithmVersionInfo sourceVersion = dto.getVersion();
    String sourcePlatform = ((ExecutableDTO) dto).getPlatformName();
    return new ExecutablePathFactory(sourceVersion, sourcePlatform);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final IPathFactory createTargetPathFactory() {
    return new ExecutablePathFactory(getVersion(), platform);
  }

  /**
   * @author Tecgraf / PUC-Rio
   * 
   *         Usado na transferncia de arquivos via
   *         {@link VersionTreeTransferHandler TransferHandler}.
   */
  public static class ExecutableDTO extends DataTransferObject implements
    Serializable {

    // Plataforma do arquivo sendo exportado pelo {@link
    // VersionTreeTransferHandler TransferHandler}.
    private final String platform;

    /**
     * COnstrutor.
     * 
     * @param node n que representa o arquivo a ser exportado pelo
     *        {@link VersionTreeTransferHandler TransferHandler}.
     */
    public ExecutableDTO(ExecutableFileNode node) {
      super(node);
      platform = node.platform;
    }

    /**
     * Obtm o nome da plataforma do arquivo sendo exportado pelo
     * {@link VersionTreeTransferHandler TransferHandler}.
     * 
     * @return o nome da plataforma do arquivo sendo exportado pelo
     *         {@link VersionTreeTransferHandler TransferHandler}.
     */
    public String getPlatformName() {
      return platform;
    }
  }
}
