package csbase.client.applications.projectsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.ProjectSharingInfoPanel;

/**
 * Dilogo para modificao do status de alocao de projetos.
 * 
 * @author jnlopes
 * 
 */
public class ProjectStatusModificationDialog extends ProjectsManagerDialog {

  /**
   * Campo para entrada do valor alocado para os projetos.
   */
  final private JTextField allocatedSpaceField = new JTextField();

  /**
   * Insets.
   */
  final private Insets insets = new Insets(5, 5, 5, 5);

  /**
   * Lista de projetos selecionados.
   */
  private List<ProjectsManagerData> projectList;

  /**
   * Espao a ser definido como alocado.
   */
  private int allocatedSpace = -1;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao.
   * @param projectList Lista de projetos selecionados.
   */
  private ProjectStatusModificationDialog(
    final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projectList) {
    super(projectsManager);
    this.projectList = projectList;

    setModal(true);
    setTitle(getString("ProjectStatusModificationDialog.title"));
    buildPanel();
  }

  /**
   * Monta o painel com os botes.
   * 
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {

    final JButton modifyButton = new JButton(new AbstractAction() {
      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        final ProjectsManager projectsManager = getProjectsManager();
        final ApplicationFrame appFrame = projectsManager.getApplicationFrame();
        try {
          allocatedSpace = Integer.parseInt(allocatedSpaceField.getText());
        }
        catch (NumberFormatException nfe) {
          allocatedSpace = -1;
        }
        if (allocatedSpace <= 0) {
          StandardDialogs.showErrorDialog(appFrame,
            getString("ProjectStatusModificationDialog.number.error.message"),
            getString("ProjectStatusModificationDialog.number.error.title"));
          return;
        }
        ProjectStatusModificationDialog.this.dispose();
      }
    });
    modifyButton
      .setText(getString("ProjectStatusModificationDialog.modify.text"));

    final JButton cancelButton = new JButton(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectStatusModificationDialog.this.dispose();
      }
    });
    cancelButton
      .setText(getString("ProjectStatusModificationDialog.cancel.text"));

    final JComponent[] buttons = { modifyButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);

    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(modifyButton);
    buttonsPanel.add(cancelButton);
    return buttonsPanel;
  }

  /**
   * Monta o painel de espao alocado.
   * 
   * @return o painel.
   */
  private JPanel buildTextPanel() {

    final JPanel panel = new JPanel(new GridBagLayout());
    panel.add(new JLabel(
      getString("ProjectStatusModificationDialog.allocated.space.label")),
      new GBC(0, 0).insets(insets));
    panel.add(allocatedSpaceField, new GBC(1, 0).horizontal().insets(insets));
    panel.add(new JLabel("gb"), new GBC(2, 0).insets(insets));
    return panel;
  }

  /**
   * Monta o componente dos projetos.
   * 
   * @return o componente
   */
  private ProjectSharingInfoPanel buildProjectPanel() {
    final ProjectsManager projectsManager = getProjectsManager();
    ProjectSharingInfoPanel projectsPanel =
      new ProjectSharingInfoPanel(projectsManager, projectList, false);
    return projectsPanel;
  }

  /**
   * Monta o painel.
   */
  private void buildPanel() {
    final JPanel buttonsPanel = buildButtonsPanel();
    final ProjectSharingInfoPanel projectsPanel = buildProjectPanel();

    final JLabel label =
      new JLabel(getString("ProjectStatusModificationDialog.warning.label"),
        ApplicationImages.ICON_WARNING_32, SwingConstants.LEFT);
    setLayout(new GridBagLayout());
    add(label, new GBC(0, 0).insets(insets));
    add(projectsPanel, new GBC(0, 1).horizontal().insets(insets));
    add(buildTextPanel(), new GBC(0, 2).both().insets(insets));
    add(buttonsPanel, new GBC(0, 3).horizontal().insets(insets));
  }

  /**
   * Mtodo esttico para abertura do painel de modificao de status de
   * alocao de projetos.
   * 
   * @param projectsManager A aplicao.
   * @param projectList A lista de projetos selecionados.
   * @return O espao de reserva de rea para os projetos selecionados.
   */
  public static int openDialog(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projectList) {
    final ProjectStatusModificationDialog psmd =
      new ProjectStatusModificationDialog(projectsManager, projectList);
    psmd.pack();
    psmd.setVisible(true);
    return psmd.allocatedSpace;
  }
}
