/**
 * $Id$
 */
package csbase.client.project;

import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Stack;

import javax.swing.JTree;
import javax.swing.event.TreeModelEvent;
import javax.swing.tree.TreePath;

import csbase.logic.ClientProjectFile;

/**
 * Implementao de um listener para eventos da rvore de projetos que busca
 * manter o aspecto e as selees da rvore aps mudanas em sua estrutura. 
 * importante no guardar referncias diretas para os elementos da rvore
 * justamente porque sua estrutura vai ser alterada, ento os elementos podem
 * mudar.
 */
final class ProjectTreeStructureListenerImpl implements
  ProjectTreeStructureListener {
  /**
   * {@link JTree} que representa a rvore de projetos.
   */
  private final JTree tree;

  /**
   * Pilha de dados para mudanas na rvore.  importante ter uma pilha porque
   * dentro de uma alterao de estrutura, outra pode ocorrer.
   */
  private Stack<ChangeData> stack;

  /**
   * Modela os dados necessrios para retornar a rvore  forma de exibio aps
   * uma alterao em sua estrutura.
   */
  private static class ChangeData {
    /**
     * Lista de paths expandidos ("abertos").
     */
    List<String[]> expandedPaths = new ArrayList<String[]>();

    /**
     * Lista de paths selecionados.
     */
    List<String[]> selectedPaths = new ArrayList<String[]>();

    /**
     * rea visvel da rvore.
     */
    Rectangle visibleRectangle;
  }

  /**
   * @param projectTree
   */
  ProjectTreeStructureListenerImpl(JTree projectTree) {
    this.tree = projectTree;
    stack = new Stack<ChangeData>();
  }

  /**
   * {@inheritDoc}
   */
  public void treeWillChange(TreeModelEvent event) {
    ChangeData data = new ChangeData();
    saveExpandedPaths(data);
    saveSelectionPaths(data);
    saveVisibleArea(data);
    stack.push(data);
  }

  /**
   * Salva a lista de ns expandidos ("abertos").  importante salvar apenas os
   * caminhos para cada n, pois os ns em si podem ser alterados na mudana da
   * estrutura que a rvora est para sofrer.
   * 
   * @param data Estrutura na qual o estado deve ser salvo.
   */
  private void saveExpandedPaths(ChangeData data) {
    ProjectTreeNode rootNode = (ProjectTreeNode) tree.getModel().getRoot();
    if (rootNode == null) {
      return;
    }
    Enumeration<TreePath> treePaths =
      tree.getExpandedDescendants(rootNode.getTreePath());
    if (treePaths == null) {
      return;
    }
    while (treePaths.hasMoreElements()) {
      TreePath treePath = treePaths.nextElement();
      ProjectTreeNode node = (ProjectTreeNode) treePath.getLastPathComponent();
      ClientProjectFile cpf = node.getClientProjectFile();
      data.expandedPaths.add(cpf.getPath());
    }
  }

  /**
   * Salva a lista de ns selecionados.  importante salvar apenas os caminhos
   * para cada n, pois os ns em si podem ser alterados na mudana da estrutura
   * que a rvora est para sofrer.
   * 
   * @param data Estrutura na qual o estado deve ser salvo.
   */
  private void saveSelectionPaths(ChangeData data) {
    data.selectedPaths.clear();
    TreePath[] treePaths = tree.getSelectionPaths();
    if (treePaths == null) {
      return;
    }
    for (TreePath treePath : treePaths) {
      ProjectTreeNode node = (ProjectTreeNode) treePath.getLastPathComponent();
      ClientProjectFile cpf = node.getClientProjectFile();
      data.selectedPaths.add(cpf.getPath());
    }
  }

  /**
   * Salva a rea atualmente sendo exibida.
   * 
   * @param data Estrutura na qual o estado deve ser salvo.
   */
  private void saveVisibleArea(ChangeData data) {
    data.visibleRectangle = tree.getVisibleRect();
  }

  /**
   * {@inheritDoc}
   */
  public void treeWasChanged(TreeModelEvent event) {
    if (stack.isEmpty()) {
      return;
    }
    ChangeData data = stack.pop();
    restoreExpandPaths(data);
    restoreSelectPaths(data);
    restoreVisibleArea(data);
  }

  /**
   * Restaura os ns que estavam expandidos anteriormente.
   * 
   * @param data Estrutura na qual o estado anterior est salvo.
   */
  private void restoreExpandPaths(ChangeData data) {
    ProjectTreeNode rootNode = (ProjectTreeNode) tree.getModel().getRoot();
    if (rootNode == null) {
      return;
    }
    for (String[] path : data.expandedPaths) {
      ProjectTreeNode node = rootNode;
      for (int i = 0; i < path.length && node != null; i++) {
        node = node.getChild(path[i]);
      }
      if (node != null && tree.isCollapsed(node.getTreePath())) {
        tree.expandPath(node.getTreePath());
      }
    }
  }

  /**
   * Restaura os ns que estavam selecionados anteriormente.
   * 
   * @param data Estrutura na qual o estado anterior est salvo.
   */
  private void restoreSelectPaths(ChangeData data) {
    ProjectTreeNode rootNode = (ProjectTreeNode) tree.getModel().getRoot();
    if (rootNode == null) {
      return;
    }
    List<TreePath> treePaths = new ArrayList<TreePath>();
    for (String[] path : data.selectedPaths) {
      ProjectTreeNode node = rootNode;
      for (int i = 0; i < path.length && node != null; i++) {
        node = node.getChild(path[i]);
      }
      if (node != null) {
        treePaths.add(node.getTreePath());
      }
    }
    tree.setSelectionPaths(treePaths.toArray(new TreePath[0]));
  }

  /**
   * Restaura a rea anteriormente visvel da rvore.
   * 
   * @param data Estrutura na qual o estado anterior est salvo.
   */
  private void restoreVisibleArea(ChangeData data) {
    tree.scrollRectToVisible(data.visibleRectangle);
  }
}