/*
 * $Id$
 */

package csbase.client.util.gui.log;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Window;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.rmi.RemoteException;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;

import csbase.client.Client;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.util.event.EventListener;
import csbase.client.util.event.EventManager;
import csbase.client.util.event.IEvent;
import csbase.exception.BugException;
import csbase.exception.CSBaseRuntimeException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.exception.UnavailableServiceException;
import csbase.logic.ClientProjectFile;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Painel a ser adicionado a uma aplicao, dando-lhe assim a funcionalidade de
 * visualizao de logs.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class LogPanel extends JPanel implements AutoReloadable {

  /**
   * Representa um evento de arquivo que pode ser um arquivo de log sendo
   * aberto, fechado, atualizado ou no encontrado.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public static class FileEvent implements IEvent {

    /**
     * Tipos de eventos de arquivo que podem ocorrer.
     */
    public enum Type {
      /**
       * O arquivo de log requisitado foi aberto com sucesso.
       */
      OPENED,
      /**
       * O arquivo de log corrente, se existir, foi fechado com sucesso.
       */
      CLOSED,
      /**
       * O arquivo de log requisitado foi atualizado com sucesso.
       */
      RELOADED,
      /**
       * O arquivo de log no foi encontrado para ser aberto, atualizado ou
       * paginado.
       */
      NOT_FOUND
    }

    /**
     * Arquivo
     */
    private final ClientProjectFile file;

    /**
     * Tipo de evento
     */
    private final Type type;

    /**
     * Retorna o arquivo que gerou o evento.
     * 
     * @return o arquivo
     */
    public ClientProjectFile getFile() {
      return file;
    }

    /**
     * Retorna o tipo de evento gerado.
     * 
     * @return o tipo
     */
    public Type getType() {
      return type;
    }

    /**
     * Construtor
     * 
     * @param file arquivo
     * @param type tipo de evento
     */
    FileEvent(final ClientProjectFile file, final Type type) {
      super();
      this.file = file;
      this.type = type;
    }
  }

  /**
   * Estende a RemoteTask sobrescrevendo o tratamento de exceo,
   */
  private abstract class LogRemoteTask extends RemoteTask<Void> {

    /**
     * As excees lanadas seram agora repassadas aos interessados, junto com
     * uma mensagem descritiva, sob a forma de um {@link ThrowableEvent evento}.
     */
    @Override
    protected final void handleError(final Exception error) {
      if (error instanceof RemoteException) {
        /* Fora a verificao do estado do servidor. */
        ClientRemoteMonitor.getInstance().invalidate();
      }
      eventManager.fireEvent(new ThrowableEvent(error));
    }
  }

  /**
   * Representa um evento de paginao indicando que a pgina corrente foi
   * alterada.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public static class PagingEvent implements IEvent {

    /**
     * Pgina
     */
    private final long page;

    /**
     * Contador de pginas
     */
    private final long countPages;

    /**
     * Retorna o total de pginas do arquivo quando o evento foi gerado.
     * 
     * @return o total
     */
    public long countPages() {
      return countPages;
    }

    /**
     * Retorna a pgina corrente quando o evento foi gerado.
     * 
     * @return a pgina
     */
    public long getPageNumber() {
      return page;
    }

    /**
     * Construtor
     * 
     * @param page pgina
     * @param countPages contagem de pginas
     */
    PagingEvent(final long page, final long countPages) {
      super();
      this.page = page;
      this.countPages = countPages;
    }
  }

  /**
   * Representa um evento a ser lanado quando ocorre um {@link Throwable}
   * exceo que deve ser tratado pela aplicao dona do LogPanel.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public static class ThrowableEvent implements IEvent {

    /**
     * Exceo
     */
    private final Throwable throwable;

    /**
     * Obtm uma mensagem explicativa sobre o evento gerado.
     * 
     * @return uma mensagem explicativa sobre o evento gerado.
     */
    public String getMessage() {

      if (throwable instanceof CSBaseRuntimeException) {
        final String msg = throwable.getMessage();
        if (msg != null) {
          return msg;
        }
        else if (throwable instanceof UnavailableServiceException) {
          return LogPanel.getString(
            "event.throwable.UnavailableServiceException");
        }
        else if (throwable instanceof ServiceFailureException) {
          return LogPanel.getString("event.throwable.ServiceFailureException");
        }
        else if (throwable instanceof PermissionException) {
          return LogPanel.getString("event.throwable.PermissionException");
        }
        else if (throwable instanceof BugException) {
          return LogPanel.getString("event.throwable.BugException");
        }
        else {
          return LogPanel.getString("event.throwable.unknown");
        }
      }
      /* falha no RMI */
      else if (throwable instanceof RemoteException) {
        return LogPanel.getString("event.throwable.RemoteException");
      }
      else if (throwable instanceof OutOfMemoryError) {
        return LogPanel.getString("event.throwable.OutOfMemoryError");
      }
      else {
        return LogPanel.getString("event.throwable.clientbug");
      }
    }

    /**
     * Obtm o {@link Throwable} que gerou o evento.
     * 
     * @return o {@link Throwable} que gerou o evento.
     */
    public Throwable getThrowable() {
      return throwable;
    }

    /**
     * Construtor
     * 
     * @param throwable a exceo
     */
    ThrowableEvent(final Throwable throwable) {
      super();
      this.throwable = throwable;
    }
  }

  /**
   * Adiciona a barra de ferramentas a funcionalidade de abrir e fechar
   * arquivos.
   */
  public static final int OPEN_CLOSE = LogPanelToolBar.OPEN_CLOSE;

  /**
   * Adiciona a barra de ferramentas a funcionalidade de auto-recarga do arquivo
   * corrente.
   */
  public static final int RELOAD = LogPanelToolBar.RELOAD;

  /**
   * Adiciona a barra de ferramentas a funcionalidade de exportar o arquivo de
   * log corrente.
   */
  public static final int EXPORT = LogPanelToolBar.EXPORT;

  /**
   * Adiciona a barra de ferramentas a funcionalidade de paginao (ir para a
   * primeira pgina, pgina anterior, pgina seguinte, ltima pgina).
   */
  public static final int PAGING = LogPanelToolBar.PAGING;

  /**
   * Indica a primeira pgina de qualquer arquivo de log aberto.
   */
  public static final long FIRST_PAGE = 1L;

  /**
   * Tamanho da pgina (default) em Kb. 0 &lt; (
   * {@value #DEFAULT_PAGE_SIZE_KB}*1024) &lt;= {@link Integer#MAX_VALUE}
   */
  public static final int DEFAULT_PAGE_SIZE_KB = 100;

  /**
   * Mtodo utilitrio para criao de painel de log com barra de ferramentas
   * completa.
   * 
   * @param pageSizeKb o tamanho (em Kb) da pgina.
   * @return um painel de log com a barra de ferramentas completa com opes de:
   *         abrir e fechar arquivo, recarga automtica, paginao e ajuda.
   */
  public static final LogPanel createLogPanelWithCompleteToolBar(
    final int pageSizeKb) {
    return LogPanel.createLogPanelWithToolBar(LogPanel.OPEN_CLOSE
      | LogPanel.RELOAD | LogPanel.PAGING, pageSizeKb);
  }

  /**
   * Mtodo utilitrio para criao de painel de log rolvel sem barra de
   * ferramentas.
   * 
   * @param pageSizeKb o tamanho (em Kb) da pgina.
   * @return um painel de log sem barra de ferramentas.
   */
  public static final LogPanel createLogPanelWithoutToolBar(
    final int pageSizeKb) {
    return new LogPanel(pageSizeKb);
  }

  /**
   * Mtodo utilitrio para criao de painel de log com barra de ferramentas
   * customizada.
   * 
   * @param toolBarFlags uma mscara de bits indicando que ferramentas devem
   *        aparecer na toolbar. As opes so: {@link #OPEN_CLOSE},
   *        {@link #RELOAD} e {@link #PAGING}.
   * @param pageSizeKb o tamanho (em Kb) da pgina.
   * 
   * @return cria um painel de log com uma barra de ferramenta customizada.
   */
  public static final LogPanel createLogPanelWithToolBar(final int toolBarFlags,
    final int pageSizeKb) {
    return new LogPanel(toolBarFlags, pageSizeKb);
  }

  /**
   * Retorna um texto do arquivo de idioma correto, de acordo com a
   * internacionalizao, utilizando com o chave a String "LogPanel."
   * concatenada com a tag. <br>
   * Ex.: tag = "test", ento ser procurado o texto no arquivo de idiomas
   * referente a chave "LogPanel.test".
   * 
   * @param tag a tag
   * @param args argumento para a formatao de texto estilo
   *        {@link String#format(String, Object...)}
   * 
   * @return um texto sem espaos antes nem depois caso a chave tenha sido
   *         encontrada ou nulo caso contrrio.
   */
  public static final String getString(final String tag, final Object... args) {
    final String key = String.format("LogPanel.%s", tag);
    if (!LNG.hasKey(key)) {
      return null;
    }

    final String format = LNG.get(key).trim();
    return String.format(format, args);
  }

  /**
   * Tamanho da pgina em bytes.
   */
  private final int pageSize;

  /**
   * Gerente de recarga do painel com o texto de modo temporizado.
   */
  private final LogPanelReloader reloader;

  /**
   * rea de texto do painel
   */
  private final LogPanelTextArea textArea;

  /**
   * Gerente de eventos
   */
  private final EventManager eventManager;

  /**
   * Toolbar
   */
  private JToolBar toolBar;

  /**
   * sado para identificar que o arquivo no foi encontrado no servidor.
   */
  private final AtomicBoolean fileNotFound;

  /**
   * Arquivo corrente que est sendo visualizado
   */
  private ClientProjectFile file;

  /**
   * Nmero de pginas do arquivo corrente
   */
  private long pagesCount;

  /**
   * Pgina corrente
   */
  private long page;

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link FileEvent}.
   * 
   * @param listener o listener
   */
  public void addFileEventListener(final EventListener<FileEvent> listener) {
    eventManager.addEventListener(listener, FileEvent.class);
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link PagingEvent}.
   * 
   * @param listener o listener
   */
  public void addPagingEventListener(
    final EventListener<PagingEvent> listener) {
    eventManager.addEventListener(listener, PagingEvent.class);
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link ThrowableEvent}.
   * 
   * @param listener o listener
   */
  public void addThrowableEventListener(
    final EventListener<ThrowableEvent> listener) {
    eventManager.addEventListener(listener, ThrowableEvent.class);
  }

  /**
   * Fecha o arquivo de log corrente e desliga a auto-recarga se esta estiver
   * ligada.
   */
  public final void closeFile() {
    if (null != getFile()) {
      if (setFile(null)) {
        eventManager.fireEvent(new FileEvent(file, FileEvent.Type.CLOSED));
        setPage(0);
      }
    }
  }

  /**
   * @return o total de pginas do arquivo.
   */
  public long countPages() {
    return pagesCount;
  }

  /**
   * @return o arquivo de log.
   */
  public final ClientProjectFile getFile() {
    return file;
  }

  /**
   * @return a aplicao que detm esta instncia.
   */
  public Window getOwner() {
    return SwingUtilities.windowForComponent(this);
  }

  /**
   * @return a pgina corrente [{@link #FIRST_PAGE}..{@link #countPages()}] ou 0
   *         caso o arquivo no exista.
   */
  public long getPageNumber() {
    return page;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final LogPanelReloader getReloader() {
    return reloader;
  }

  /**
   * @return a rea de texto.
   */
  public final LogPanelTextArea getTextArea() {
    return textArea;
  }

  /**
   * @return o ttulo da janela que detm esta instncia ou, caso este seja
   *         {@code null}, o retorno de {@link #getString(String, Object...)}
   *         com o parmetro tag tendo o valor "title.default".
   */
  public String getTitle() {
    final Window owner = getOwner();
    if (null != owner) {
      final Class<?> clazz = owner.getClass();
      if (Frame.class.isAssignableFrom(clazz)) {
        return ((Frame) owner).getTitle();
      }
      else if (Dialog.class.isAssignableFrom(clazz)) {
        return ((Dialog) owner).getTitle();
      }
    }

    return LogPanel.getString("title.default");
  }

  /**
   * @return a barra de ferramentas ou {@code null} caso este painel tenha sido
   *         criado pelo mtodo {@link #createLogPanelWithoutToolBar(int)}
   */
  public final JToolBar getToolBar() {
    return toolBar;
  }

  /**
   * Altera a pgina corrente.
   * 
   * @param newPage a nova pgina
   */
  public void goToPage(final long newPage) {
    if (reloadData()) {
      setPage(newPage);
    }
  }

  /**
   * @return o trecho do arquivo referente a pgina escolhida.
   */
  private String loadText() {
    try {
      if (null == getFile()) {
        return null;
      }
      if (0 >= getFile().size()) {
        return "";
      }

      final StringBuilder stringBuffer = new StringBuilder(pageSize);
      final long jumpSize = pageSize * (getPageNumber() - 1);

      final RemoteTask<Void> task = new LogRemoteTask() {
        @Override
        public void performTask() throws Exception {
          final InputStream stream = getFile().getInputStream();
          final InputStreamReader reader = new InputStreamReader(stream, Client
            .getInstance().getSystemDefaultCharset());
          /** Pula para a pgina desejada. */
          reader.skip(jumpSize);
          final char[] buffer = new char[pageSize];
          final int nbytes = reader.read(buffer);
          if (0 < nbytes) {
            stringBuffer.append(buffer, 0, nbytes);
          }
          reader.close();
        }
      };

      final String msg = LogPanel.getString("loading.page.msg");
      final ImageIcon icon = ApplicationImages.ICON_OPEN_24;
      final Window window = getOwner();
      final String title = getTitle();
      final boolean ok = task.execute(window, title, msg, icon);
      if (ok) {
        return stringBuffer.toString();
      }
      return null;
    }
    catch (final Throwable t) {
      eventManager.fireEvent(new ThrowableEvent(t));
      return null;
    }
  }

  /**
   * Abre um arquivo de log.
   * 
   * @param newFile o arquivo a ser aberto.
   */
  public final void openFile(final ClientProjectFile newFile) {
    if (newFile == null) {
      final String err = "Tentando abrir um arquivo invlido";
      throw new IllegalArgumentException(err);
    }

    /* Fechamos o arquivo corrente antes de abrir o novo arquivo. */
    closeFile();

    if (setFile(newFile)) {
      eventManager.fireEvent(new FileEvent(newFile, FileEvent.Type.OPENED));
      setPage(countPages());
      textArea.goToTail();
    }
  }

  /**
   * Recarrega as informaes do arquivo com dados do servidor e em seguida
   * calcula e salva, para uso posterior, o total de pginas do arquivo.
   * 
   * @return true se os dados do arquivo e de total de pginas foram
   *         recarregados com sucesso.
   */
  private boolean reloadData() {

    /**
     * Atualiza as informaes do arquivo para trabalharmos com os dados
     * corretos.
     */
    if (!reloadFileFromServer()) {
      /**
       * Caso no tenha sido possvel recarregar as informaes do
       * ClientProjectFile devido a um erro.
       */
      pagesCount = 0;
      return false;
    }
    if (fileNotFound.get()) {
      /**
       * Caso o arquivo no tenha sido encontrado, o usurio  informado com uma
       * mensagem de erro na barra de status.
       * 
       * Esse erro ocorre quando o arquivo  movido ou apagado.
       */
      pagesCount = 0;
      eventManager.fireEvent(new FileEvent(getFile(),
        FileEvent.Type.NOT_FOUND));
      return false;
    }

    /**
     * O arquivo foi encontrado e a recarga das informaes foi feita com
     * sucesso. Calculamos e salvamos, para uso posterior, o total de pginas do
     * arquivo.
     */
    if (null == getFile()) {
      pagesCount = 0;
    }
    else {
      final long fileSize = getFile().size();
      pagesCount = (long) Math.ceil((double) fileSize / pageSize);
      pagesCount = Math.max(1, pagesCount);
    }
    return true;
  }

  /**
   * Faz a recarga do arquivo corrente e mostra o final da ltima pgina.
   */
  @Override
  public void reload() {
    try {
      if (reloadData()) {
        Runnable runnable = new Runnable() {
          @Override
          public void run() {
            eventManager.fireEvent(new FileEvent(getFile(),
              FileEvent.Type.RELOADED));
            setPage(countPages());
            getTextArea().goToTail();
          }
        };
        if (SwingThreadDispatcher.isEventDispatchThread()) {
          runnable.run();
        }
        else {
          SwingThreadDispatcher.invokeAndWait(runnable);
        }
      }
    }
    catch (final Exception e) {
      eventManager.fireEvent(new ThrowableEvent(e));
    }
  }

  /**
   * Atualiza as informaes do arquivo de log, caso ele exista, com dados do
   * servidor.
   * 
   * @return true se recarregou com sucesso.
   */
  private boolean reloadFileFromServer() {
    /* Se no tem arquivo, no precisa recarregar pois eles no existe nada. */
    if (null == getFile()) {
      return true;
    }

    final RemoteTask<Void> task = new LogRemoteTask() {
      @Override
      public void performTask() throws Exception {
        final ClientProjectFile panelFile = getFile();
        if (panelFile != null && panelFile.exists()) {
          panelFile.updateInfo();
          fileNotFound.set(false);
        }
        else {
          fileNotFound.set(true);
        }
      }
    };
    final String msg = LogPanel.getString("loading.page.msg");
    final ImageIcon icon = ApplicationImages.ICON_OPEN_24;
    final Window window = getOwner();
    final String title = getTitle();

    /**
     * Executa a task que recarrega os dados do ClientProjectFile.
     */
    return task.execute(window, title, msg, icon);
  }

  /**
   * Atribui o arquivo como sendo o arquivo corrente e recarrega seus dados.
   * 
   * @param file o novo arquivo corrente.
   * 
   * @return se os dados do novo arquivo foram recarregados com sucesso.
   */
  private boolean setFile(final ClientProjectFile file) {
    this.file = file;
    return reloadData();
  }

  /**
   * Atribui a pgina como sendo a pgina corrente e a carrega na rea de texto.
   * 
   * @param newPage a nova pgina.
   */
  private void setPage(final long newPage) {
    if (newPage > countPages()) {
      final Window window = getOwner();
      final String title = getTitle();
      final String warningMsg = LogPanel.getString(
        "dialog.warning.file.page.outOfBounds", newPage, countPages(), getFile()
          .getName());

      final String[] buttons = new String[] { LogPanel.getString("button.ok") };
      JOptionPane.showOptionDialog(window, warningMsg, title,
        JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE, null, buttons,
        buttons[0]);
    }

    if (LogPanel.FIRST_PAGE > newPage) {
      this.page = 0;
      getTextArea().setText("");
    }
    else {
      this.page = Math.min(newPage, countPages());
      final String text = loadText();
      if (null != text) {
        getTextArea().setText(text);
      }
    }

    eventManager.fireEvent(new PagingEvent(getPageNumber(), countPages()));
  }

  /**
   * Construtor para um painel de log sem barra de ferramentas.
   * 
   * @param pageSizeKb o tamanho pr-definido da pgina.
   */
  private LogPanel(final int pageSizeKb) {
    /*
     * Atribui o valor to tamanho de uma pgina em bytes, tomando cuidado para
     * que o valor no seja negativo.
     */

    /* Converso Kb -> bytes */
    final int pageSizeBytes = pageSizeKb * 1024;
    final int defaultSizeBytes = LogPanel.DEFAULT_PAGE_SIZE_KB * 1024;
    this.pageSize = pageSizeBytes < 0 ? defaultSizeBytes : pageSizeBytes;
    this.eventManager = new EventManager();
    this.reloader = new LogPanelReloader(this);
    this.toolBar = null;
    this.textArea = createTextArea();
    this.fileNotFound = new AtomicBoolean(false);

    setLayout(new BorderLayout());
    add(textArea, BorderLayout.CENTER);
  }

  /**
   * Cria e inicializa o estado da rea de texto de log.
   * 
   * @return a rea de texto.
   */
  private LogPanelTextArea createTextArea() {
    final LogPanelTextArea logTextArea = new LogPanelTextArea(this, true);

    logTextArea.setPrefix(getPageNumber());

    /* Atualiza o prefixo em caso de eventos de paginao. */
    addPagingEventListener(new EventListener<LogPanel.PagingEvent>() {
      @Override
      public void eventFired(final PagingEvent event) {
        final long pg = event.getPageNumber();
        logTextArea.setPrefix(pg);
      }
    });

    return logTextArea;
  }

  /**
   * Construtor para um painel de log com barra de ferramentas customizada.
   * 
   * @param toolBarFlags uma mscara de bits indicando que ferramentas devem
   *        aparecer na toolbar. As opes so: {@link #OPEN_CLOSE},
   *        {@link #RELOAD}, {@link #EXPORT}, {@link #PAGING}.
   * @param pageSize o tamanho (em bytes) pr-definido da pgina.
   */
  private LogPanel(final int toolBarFlags, final int pageSize) {
    this(pageSize);
    this.toolBar = new LogPanelToolBar(this, toolBarFlags);

    add(toolBar, BorderLayout.NORTH);
  }

}
