/*
 * $Id:$
 */

package csbase.client;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import csbase.exception.PermissionException;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ProjectFileType;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Classe que representa um arquivo local no lado cliente.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ClientLocalFile implements ClientFile {

  /**
   * O arquivo local.
   */
  private File file;

  /**
   * Arquivo randmico.
   */
  private RandomAccessFile randomFile = null;

  /**
   * Retorna se existe o arquivo fsico local.
   *
   * @return indicativo
   */
  @Override
  final public boolean exists() {
    return file.exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return file.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(final boolean readOnly) throws Exception {
    if (file.isDirectory()) {
      throw new Exception("Diretrio no pode ter open()!");
    }
    final String mode = readOnly ? "r" : "rws";
    checkReadPermission();
    if (!readOnly) {
      checkWritePermission();
    }
    randomFile = new RandomAccessFile(file, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, long position) throws Exception {
    checkReadPermission();
    if (randomFile == null) {
      throw new IllegalStateException("Arquivo no foi aberto!");
    }
    randomFile.seek(position);
    return randomFile.read(dst, 0, dst.length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len, long position)
    throws Exception {
    checkReadPermission();
    if (randomFile == null) {
      throw new IllegalStateException("Arquivo no foi aberto!");
    }
    randomFile.seek(position);
    return randomFile.read(dst, off, len);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long size() {
    return file.length();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    return file.getAbsolutePath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    String path = file.getPath();
    String[] splitPath = path.split(Pattern.quote(File.separator));
    return splitPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    checkReadPermission();
    return new FileInputStream(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    checkWritePermission();
    return new FileOutputStream(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFile[] getChildren() {
    checkReadPermission();
    if (!file.isDirectory()) {
      return null;
    }

    if (!file.canRead()) {
      return null;
    }

    final List<ClientFile> list = new ArrayList<ClientFile>();
    for (File f : file.listFiles()) {
      list.add(new ClientLocalFile(f));
    }
    if (list.size() == 0) {
      return null;
    }
    final ClientFile[] array = list.toArray(new ClientFile[list.size()]);
    return array;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    final String localFileName = getName();

    String localFileExtension = "";

    if (localFileName != null && !localFileName.equals("")) {
      localFileExtension = FileUtils.getFileExtension(localFileName);
    }
    final ProjectFileType pft = ProjectFileType.getProjectFileTypeFromExtension(
      localFileExtension, isDirectory());
    return pft.getCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientLocalFile getParent() {
    final File parentFile = file.getParentFile();
    if (parentFile == null) {
      return null;
    }

    final ClientLocalFile parent = new ClientLocalFile(parentFile);
    if (!parent.isDirectory()) {
      throw new IllegalStateException("Parent no  diretrio!");
    }
    return parent;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate() {
    if (!file.canRead()) {
      return 0L;
    }
    final long lastModified = file.lastModified();
    return lastModified;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws IOException {
    if (randomFile != null) {
      RandomAccessFile raf = randomFile;
      if (force) {
        randomFile = null;
      }
      raf.close();
      randomFile = null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((file == null) ? 0 : file.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof ClientLocalFile)) {
      return false;
    }
    ClientLocalFile other = (ClientLocalFile) obj;
    if (file == null) {
      if (other.file != null) {
        return false;
      }
    }
    else if (!file.equals(other.file)) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFileType getClientFileType() {
    return ClientFileType.LOCAL;
  }

  /**
   * Check interno de permisso
   */
  private void checkReadPermission() {
    if (exists() && !canRead()) {
      throw new PermissionException();
    }
  }

  /**
   * Check interno de permisso
   */
  private void checkWritePermission() {
    if (exists() && !canWrite()) {
      throw new PermissionException();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException {
    if (randomFile == null) {
      throw new IllegalStateException("Arquivo no foi aberto!");
    }
    randomFile.seek(position);
    randomFile.write(src, off, len);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position) throws IOException {
    if (randomFile == null) {
      throw new IllegalStateException("Arquivo no foi aberto!");
    }
    randomFile.seek(position);
    randomFile.write(src);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long position() throws IOException {
    final long fp = randomFile.getFilePointer();
    return fp;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    randomFile.seek(newPosition);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return file.canExecute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return file.canRead();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return file.canWrite();
  }

  /**
   * Construtor.
   *
   * @param file O arquivo local.
   */
  public ClientLocalFile(File file) {
    if (file == null) {
      final String err = "Null detectado na criao de ClientLocalFile!";
      throw new IllegalArgumentException(err);
    }
    this.file = file;
    this.randomFile = null;
  }

  /**
   * Renomeia o arquivo
   *
   * @param newName novo nome do arquivo.
   * @return se foi possivel renomear
   */
  public boolean rename(String newName) {
    if (newName == null) {
      final String err = "Nome do arquivo no pode ser nulo";
      throw new IllegalArgumentException(err);
    }

    if (randomFile != null) {
      throw new IllegalStateException("Arquivo aberto no pode ser renomeado!");
    }

    if (isDirectory()) {
      throw new IllegalStateException("Diretrio no pode ser renomeado!");
    }

    Path path = file.toPath();
    File dir = new File(getParent().getStringPath());
    Path target = dir.toPath().resolve(newName);

    if(Files.exists(target)) {
      return false;
    }

    try {
      Path newPath = Files.move(path, target);
      if (newPath != null) {
        this.file = newPath.toFile();
        return true;
      }
      return false;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Renomeia o arquivo
   *
   * @param newParent novo diretrio pai do arquivo.
   * @return se foi possivel renomear
   */
  public boolean move(ClientLocalFile newParent) {
    if (newParent == null) {
      final String err = "Diretrio pai no pode ser nulo";
      throw new IllegalArgumentException(err);
    }

    if (randomFile != null) {
      throw new IllegalStateException("Arquivo aberto no pode ser movido!");
    }

    if(!exists()) {
      throw new IllegalStateException("Arquivo no existe!");
    }

    if (isDirectory()) {
      throw new IllegalStateException("Diretrio no pode ser movido!");
    }

    if (!newParent.isDirectory()) {
      throw new IllegalStateException(
        "Destino precisa ser diretrio: " + newParent.getStringPath());
    }

    Path path = file.toPath();
    File dir = new File(newParent.getStringPath());
    Path target = dir.toPath().resolve(getName());

    if(Files.exists(target)) {
      return false;
    }

    try {
      Path newPath = Files.move(path, target);
      if (newPath != null) {
        this.file = newPath.toFile();
        return true;
      }
      return false;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Copia o arquivo
   *
   * @param newParent diretrio pai do novo arquivo.
   * @return a cpia do arquivo.
   * @throws Exception em caso de erro durante a cpia.
   */
  public ClientLocalFile copy(ClientLocalFile newParent) throws Exception {
    if (newParent == null) {
      final String err = "Diretrio pai no pode ser nulo";
      throw new IllegalArgumentException(err);
    }

    if (!newParent.isDirectory()) {
      throw new IllegalStateException(
        "Caminho precisa ser diretrio: " + newParent.getStringPath());
    }

    if(!exists()) {
      throw new IllegalStateException("Arquivo no existe!");
    }

    Path targetDir = new File(newParent.getStringPath()).toPath();
    Path target = targetDir.resolve(file.getName());
    if(Files.exists(target)) {
      return null;
    }

    Path copy = Files.copy(file.toPath(), target);
    return new ClientLocalFile(copy.toFile());
  }

  /**
   * Remove o arquivo.
   *
   * @return {@code true} caso o arquivo seja removido com sucesso ou {@code
   * false}, caso contrrio.
   */
  public boolean remove() {
    // No pode renomear arquivo aberto
    if (randomFile != null) {
      return false;
    }
    return file.delete();
  }
}
