/*
 * $Id$
 */
package csbase.client.algorithms.commands.cache.events;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import csbase.client.algorithms.commands.cache.CommandsFilter;
import csbase.client.algorithms.commands.cache.events.CacheUpdatedEvent.Type;
import csbase.client.util.event.EventListener;
import csbase.logic.CommandInfo;

/**
 * @author Tecgraf / PUC-Rio
 * 
 * Representa o contrato de um observador de eventos do tipo
 * {@link CacheUpdatedEvent} que filtra os comandos antes de repass-los para o
 * mtodo que deve tratar o evento.
 * 
 * @see CacheUpdatedEvent
 */
public abstract class AbstractCacheUpdatedEventListener implements
  EventListener<CacheUpdatedEvent> {

  private ReentrantReadWriteLock lock;
  private CommandsFilter filter;

  /**
   * Construtor.
   * 
   * @param filter filtro que define que comandos devem ser passados ao mtodo
   *        {@link #eventFired(Collection)} a cada evento disparado..<br>
   *        Se for <tt>null</tt> os eventos sero ignorados.
   * 
   */
  public AbstractCacheUpdatedEventListener(CommandsFilter filter) {
    this.lock = new ReentrantReadWriteLock();
    setFilter(filter);
  }

  /**
   * Altera o filtro que define que comandos devem ser passados ao mtodo
   * {@link #eventFired(Collection)} a cada evento disparado..<br>
   * Se for <tt>null</tt> os eventos sero ignorados.
   * 
   * @param filter filtro que define que comandos devem ser passados ao mtodo
   *        {@link #eventFired(Collection)} a cada evento disparado..<br>
   *        Se for <tt>null</tt> os eventos sero ignorados.
   */
  public void setFilter(CommandsFilter filter) {
    lock.writeLock().lock();
    try {
      this.filter = filter;
    }
    finally {
      lock.writeLock().unlock();
    }
  }

  /**
   * Obtm o filtro que define que comandos devem ser passados ao mtodo
   * {@link #eventFired(Collection)} a cada evento disparado..<br>
   * Se for <tt>null</tt> os eventos sero ignorados.
   * 
   * @return o filtro que define que comandos devem ser passados ao mtodo
   *         {@link #eventFired(Collection)} a cada evento disparado..<br>
   *         Se for <tt>null</tt> os eventos sero ignorados.
   */
  public CommandsFilter getFilter() {
    lock.readLock().lock();
    try {
      return this.filter;
    }
    finally {
      lock.readLock().unlock();
    }
  }

  @Override
  public final void eventFired(CacheUpdatedEvent event) {
    if (Type.exception == event.getType()) {
      eventInterrupted(event.getException(), event.getExceptionDescription());
    }
    else {
      Collection<CommandInfo> commands;
      lock.readLock().lock();
      try {
        if (null == filter) {
          return;
        }

        commands = new ArrayList<CommandInfo>();
        for (CommandInfo aCommand : event.getCommands()) {
          if (filter.accept(aCommand)) {
            commands.add(aCommand);
          }
        }
      }
      finally {
        lock.readLock().unlock();
      }
      eventFired(commands);
    }
  }

  /**
   * Indica que a cache foi atualizada com sucesso.
   * 
   * @param commands comandos da cache que foram aceitos pelo filtro.
   */
  public abstract void eventFired(Collection<CommandInfo> commands);

  /**
   * Indica que o processo de atualizao da cache foi interrompido com uma
   * exceo.
   * 
   * @param exception Exceo que interrompeu o processo de atualizao da
   *        cache.
   * @param description Descrio da exceo (pode ser <tt>null</tt>).
   */
  protected void eventInterrupted(Exception exception, String description) {
  }
}
